<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Product entity
 */
class Product
{
    /** @var string */
    public $id;

    /** @var string */
    public $name;

    /** @var string|null */
    public $content;

    /** @var string|null */
    public $excerpt;

    /** @var string|null */
    public $searchableText;

    /** @var float|null */
    public $price;

    /** @var float|null */
    public $salePrice;

    /** @var float|null */
    public $regularPrice;

    /** @var bool|null */
    public $onSale;

    /** @var int|null */
    public $stockQuantity;

    /** @var string|null */
    public $stockStatus;

    /** @var string|null */
    public $sku;

    /** @var float|null */
    public $weight;

    /** @var array|null */
    public $dimensions;

    /** @var float|null */
    public $rating;

    /** @var int|null */
    public $reviewCount;

    /** @var array */
    public $categories = [];

    /** @var array */
    public $tags = [];

    /** @var array */
    public $brands = [];

    /** @var string|null */
    public $image;

    /** @var string|null */
    public $url;

    /** @var string|null */
    public $type;

    /** @var string|null */
    public $status;

    /** @var string|null */
    public $visibility;

    /** @var bool|null */
    public $featured;

    /** @var bool|null */
    public $virtual;

    /** @var bool|null */
    public $downloadable;

    /** @var string|null */
    public $createdAt;

    /** @var string|null */
    public $updatedAt;

    /** @var array|null */
    public $attributes;

    /** @var array|null */
    public $metadata;

    /** @var string|null Document source (wordpress, shopify, manual, api) */
    public $source;

    /** @var float|null Search score */
    public $score;

    /** @var array|null Matching text chunks from search */
    public $matchingChunks;

    /**
     * Create a Product from API response data
     *
     * @param array $data
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $product = new self();

        $product->id = $data['id'] ?? '';
        $product->name = $data['name'] ?? '';
        $product->content = $data['content'] ?? null;
        $product->excerpt = $data['excerpt'] ?? null;
        $product->searchableText = $data['searchableText'] ?? null;
        $product->price = isset($data['price']) ? (float) $data['price'] : null;
        $product->salePrice = isset($data['sale_price']) ? (float) $data['sale_price'] : null;
        $product->regularPrice = isset($data['regular_price']) ? (float) $data['regular_price'] : null;
        $product->onSale = $data['on_sale'] ?? null;
        $product->stockQuantity = isset($data['stock_quantity']) ? (int) $data['stock_quantity'] : null;
        $product->stockStatus = $data['stock_status'] ?? null;
        $product->sku = $data['sku'] ?? null;
        $product->weight = isset($data['weight']) ? (float) $data['weight'] : null;
        $product->dimensions = $data['dimensions'] ?? null;
        $product->rating = isset($data['rating']) ? (float) $data['rating'] : null;
        $product->reviewCount = isset($data['review_count']) ? (int) $data['review_count'] : null;
        $product->categories = $data['categories'] ?? [];
        $product->tags = $data['tags'] ?? [];
        $product->brands = $data['brands'] ?? [];
        $product->image = $data['image'] ?? null;
        $product->url = $data['url'] ?? null;
        $product->type = $data['type'] ?? null;
        $product->status = $data['status'] ?? null;
        $product->visibility = $data['visibility'] ?? null;
        $product->featured = $data['featured'] ?? null;
        $product->virtual = $data['virtual'] ?? null;
        $product->downloadable = $data['downloadable'] ?? null;
        $product->createdAt = $data['created_at'] ?? null;
        $product->updatedAt = $data['updated_at'] ?? null;
        $product->attributes = $data['attributes'] ?? null;
        $product->metadata = $data['metadata'] ?? null;
        $product->source = $data['source'] ?? null;
        $product->score = isset($data['score']) ? (float) $data['score'] : null;
        $product->matchingChunks = $data['matching_chunks'] ?? null;

        return $product;
    }

    /**
     * Convert the product to an array for API requests
     *
     * @return array
     */
    public function toArray(): array
    {
        $data = array_filter([
            'id' => $this->id,
            'name' => $this->name,
            'content' => $this->content,
            'excerpt' => $this->excerpt,
            'searchableText' => $this->searchableText,
            'price' => $this->price,
            'sale_price' => $this->salePrice,
            'regular_price' => $this->regularPrice,
            'on_sale' => $this->onSale,
            'stock_quantity' => $this->stockQuantity,
            'stock_status' => $this->stockStatus,
            'sku' => $this->sku,
            'weight' => $this->weight,
            'dimensions' => $this->dimensions,
            'rating' => $this->rating,
            'review_count' => $this->reviewCount,
            'categories' => $this->categories,
            'tags' => $this->tags,
            'brands' => $this->brands,
            'image' => $this->image,
            'url' => $this->url,
            'type' => $this->type,
            'status' => $this->status,
            'visibility' => $this->visibility,
            'featured' => $this->featured,
            'virtual' => $this->virtual,
            'downloadable' => $this->downloadable,
            'attributes' => $this->attributes,
            'metadata' => $this->metadata,
            'source' => $this->source,
        ], function ($value) {
            return $value !== null;
        });

        return $data;
    }

    /**
     * Check if the product is on sale
     *
     * @return bool
     */
    public function isOnSale(): bool
    {
        return $this->onSale === true
            || ($this->salePrice && $this->regularPrice && $this->salePrice < $this->regularPrice);
    }

    /**
     * Get the effective price (sale price if on sale, otherwise regular price)
     *
     * @return float|null
     */
    public function getEffectivePrice(): ?float
    {
        if ($this->isOnSale() && $this->salePrice !== null) {
            return $this->salePrice;
        }

        return $this->price ?? $this->regularPrice;
    }

    /**
     * Check if the product is in stock
     *
     * @return bool
     */
    public function isInStock(): bool
    {
        return $this->stockStatus === 'instock' ||
               ($this->stockQuantity !== null && $this->stockQuantity > 0);
    }

    /**
     * Get a specific attribute value
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function getAttribute(string $key, $default = null)
    {
        return $this->attributes[$key] ?? $default;
    }

    /**
     * Get a specific metadata value
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function getMetadata(string $key, $default = null)
    {
        return $this->metadata[$key] ?? $default;
    }
}
