<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

class SearchResult
{
    public array $results;
    public int $total;
    /**
     * Faceted aggregations for filter UI.
     * Keys are field names (e.g., 'categories', 'brands', 'attributes.color', 'price_ranges')
     * Values are arrays of buckets with 'key' and 'count' (and optionally 'from'/'to' for ranges)
     * Only present when includeFacets=true in the search request.
     * @var array<string, array<array{key: string, count: int, from?: float, to?: float}>>|null
     */
    public ?array $facets;
    public array $query;
    public ?array $classification;
    public bool $enhanced;
    public ?string $originalText;
    public bool $agentUsed;
    public bool $fallbackUsed;
    public ?float $usageMultiplier;

    public static function fromArray(array $data, ?string $entityClass = null): self
    {
        $searchResult = new self();

        $searchResult->results = [];
        if (isset($data['results']) && is_array($data['results'])) {
            foreach ($data['results'] as $item) {
                if ($entityClass && method_exists($entityClass, 'fromArray')) {
                    $searchResult->results[] = $entityClass::fromArray($item);
                } else {
                    $searchResult->results[] = $item;
                }
            }
        }

        $searchResult->total = $data['total'] ?? count($searchResult->results);
        $searchResult->facets = $data['facets'] ?? null;
        $searchResult->query = $data['query'] ?? [];
        $searchResult->classification = $data['classification'] ?? null;
        $searchResult->enhanced = $data['enhanced'] ?? false;
        $searchResult->originalText = $data['original_text'] ?? null;
        $searchResult->agentUsed = $data['agent_used'] ?? false;
        $searchResult->fallbackUsed = $data['fallback_used'] ?? false;
        $searchResult->usageMultiplier = isset($data['usage_multiplier']) ? (float) $data['usage_multiplier'] : null;

        return $searchResult;
    }

    public function getResults(): array
    {
        return $this->results;
    }

    public function getTotal(): int
    {
        return $this->total;
    }

    public function getQuery(): array
    {
        return $this->query;
    }

    public function wasEnhanced(): bool
    {
        return $this->enhanced;
    }

    public function wasAgentUsed(): bool
    {
        return $this->agentUsed;
    }

    public function wasFallbackUsed(): bool
    {
        return $this->fallbackUsed;
    }

    public function getClassification(): ?array
    {
        return $this->classification;
    }

    public function getComplexity(): ?string
    {
        return $this->classification['complexity'] ?? null;
    }

    public function isComplexQuery(): bool
    {
        return $this->getComplexity() === 'complex';
    }

    public function getOriginalText(): ?string
    {
        return $this->originalText;
    }

    public function getUsageMultiplier(): ?float
    {
        return $this->usageMultiplier;
    }

    public function hasResults(): bool
    {
        return !empty($this->results);
    }

    public function getFirstResult(): mixed
    {
        return $this->results[0] ?? null;
    }

    public function getIterator(): \ArrayIterator
    {
        return new \ArrayIterator($this->results);
    }

    public function count(): int
    {
        return count($this->results);
    }

    /**
     * Get faceted aggregations for filter UI.
     * @return array<string, array<array{key: string, count: int, from?: float, to?: float}>>|null
     */
    public function getFacets(): ?array
    {
        return $this->facets;
    }

    /**
     * Check if facets are available in the response.
     */
    public function hasFacets(): bool
    {
        return !empty($this->facets);
    }

    /**
     * Get a specific facet by field name.
     * @param string $field The facet field (e.g., 'categories', 'brands', 'attributes.color')
     * @return array<array{key: string, count: int, from?: float, to?: float}>|null
     */
    public function getFacet(string $field): ?array
    {
        return $this->facets[$field] ?? null;
    }

    /**
     * Get facet values as a simple key => count array.
     * Useful for building filter UI dropdowns.
     * @param string $field The facet field
     * @return array<string, int>
     */
    public function getFacetCounts(string $field): array
    {
        $facet = $this->getFacet($field);
        if ($facet === null) {
            return [];
        }

        $counts = [];
        foreach ($facet as $bucket) {
            $counts[$bucket['key']] = $bucket['count'];
        }
        return $counts;
    }
}
