<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Helper class for parsing and handling streaming chat chunks
 */
class StreamingChunk
{
    public string $type;
    public mixed $data;
    public ?array $metadata;

    private function __construct(string $type, mixed $data, ?array $metadata = null)
    {
        $this->type = $type;
        $this->data = $data;
        $this->metadata = $metadata;
    }

    /**
     * Create a StreamingChunk from a raw decoded chunk array
     */
    public static function fromArray(array $chunk): self
    {
        return new self(
            $chunk['type'] ?? 'unknown',
            $chunk['data'] ?? null,
            $chunk['metadata'] ?? null
        );
    }

    /**
     * Check if this is a text content chunk
     */
    public function isTextDelta(): bool
    {
        return $this->type === StreamingChunkType::TEXT_DELTA;
    }

    /**
     * Check if this is a product chunk
     */
    public function isProduct(): bool
    {
        return $this->type === StreamingChunkType::PRODUCT;
    }

    /**
     * Check if this is a post chunk
     */
    public function isPost(): bool
    {
        return $this->type === StreamingChunkType::POST;
    }

    /**
     * Check if this is a post delta (streaming summary update)
     */
    public function isPostDelta(): bool
    {
        return $this->type === StreamingChunkType::POST_DELTA;
    }

    /**
     * Check if this is a prompts chunk
     */
    public function isPrompts(): bool
    {
        return $this->type === StreamingChunkType::PROMPTS;
    }

    /**
     * Check if this is a domain offer chunk
     */
    public function isDomainOffer(): bool
    {
        return $this->type === StreamingChunkType::DOMAIN_OFFER;
    }

    /**
     * Check if this is a domain offer success chunk
     */
    public function isDomainOfferSuccess(): bool
    {
        return $this->type === StreamingChunkType::DOMAIN_OFFER_SUCCESS;
    }

    /**
     * Check if this is a stream completion chunk
     */
    public function isComplete(): bool
    {
        return $this->type === StreamingChunkType::COMPLETE;
    }

    /**
     * Check if this is an error chunk
     */
    public function isError(): bool
    {
        return $this->type === StreamingChunkType::ERROR;
    }

    /**
     * Check if this is a progress indicator chunk
     */
    public function isProgress(): bool
    {
        return $this->type === StreamingChunkType::PROGRESS;
    }

    /**
     * Get text content if this is a text delta chunk
     */
    public function getTextDelta(): ?string
    {
        if ($this->isTextDelta() && is_string($this->data)) {
            return $this->data;
        }
        return null;
    }

    /**
     * Get product data if this is a product chunk
     */
    public function getProductData(): ?array
    {
        if ($this->isProduct() && is_array($this->data)) {
            return $this->data;
        }
        return null;
    }

    /**
     * Get post data if this is a post chunk
     */
    public function getPostData(): ?array
    {
        if ($this->isPost() && is_array($this->data)) {
            return $this->data;
        }
        return null;
    }

    /**
     * Get post delta if this is a summary update chunk
     * Returns ['id' => string, 'delta' => string]
     */
    public function getPostDelta(): ?array
    {
        if ($this->isPostDelta() && is_array($this->data)) {
            return $this->data;
        }
        return null;
    }

    /**
     * Get prompts array if this is a prompts chunk
     */
    public function getPrompts(): ?array
    {
        if ($this->isPrompts() && is_array($this->data)) {
            return $this->data;
        }
        return null;
    }

    /**
     * Get domain offer data if this is a domain offer chunk
     * @return array{domain: string, message: string, ui: array, context?: array}|null
     */
    public function getDomainOffer(): ?array
    {
        if ($this->isDomainOffer() && is_array($this->data)) {
            /** @var array{domain: string, message: string, ui: array, context?: array} */
            return $this->data;
        }
        return null;
    }

    /**
     * Get domain offer success data if this is a domain offer success chunk
     * @return array{domain: string, status: string, agent?: array, conversationId?: string}|null
     */
    public function getDomainOfferSuccess(): ?array
    {
        if ($this->isDomainOfferSuccess() && is_array($this->data)) {
            /** @var array{domain: string, status: string, agent?: array, conversationId?: string} */
            return $this->data;
        }
        return null;
    }

    /**
     * Get error message if this is an error chunk
     */
    public function getErrorMessage(): ?string
    {
        if ($this->isError() && is_array($this->data) && isset($this->data['error'])) {
            return $this->data['error'];
        }
        return null;
    }

    /**
     * Get progress message if this is a progress chunk
     */
    public function getProgressMessage(): ?string
    {
        if ($this->isProgress() && is_array($this->data) && isset($this->data['message'])) {
            return $this->data['message'];
        }
        return null;
    }
}
