<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Webhook entity representing a webhook configuration
 */
class Webhook
{
    /** @var string */
    public $id;

    /** @var string */
    public $name;

    /** @var string|null */
    public $description;

    /** @var string */
    public $url;

    /** @var string */
    public $method;

    /** @var string|null */
    public $secret;

    /** @var int */
    public $timeoutMs;

    /** @var array */
    public $events;

    /** @var array|null */
    public $eventFilters;

    /** @var string */
    public $status;

    /** @var array */
    public $retryPolicy;

    /** @var array|null */
    public $headers;

    /** @var string */
    public $authType;

    /** @var array|null */
    public $authConfig;

    /** @var int */
    public $totalDeliveries;

    /** @var int */
    public $successfulDeliveries;

    /** @var int */
    public $failedDeliveries;

    /** @var float|null */
    public $avgResponseTimeMs;

    /** @var string|null */
    public $lastDeliveryAt;

    /** @var string|null */
    public $lastDeliveryStatus;

    /** @var int|null */
    public $lastDeliveryResponseCode;

    /** @var string|null */
    public $lastDeliveryError;

    /** @var string */
    public $tenantId;

    /** @var string|null */
    public $workspaceId;

    /** @var string */
    public $createdBy;

    /** @var string */
    public $updatedBy;

    /** @var string */
    public $createdAt;

    /** @var string */
    public $updatedAt;

    /** @var array|null */
    public $schedule;

    /** @var array|null */
    public $metadata;

    /**
     * Create Webhook from array data
     *
     * @param array $data
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $webhook = new self();

        $webhook->id = $data['id'] ?? $data['external_id'] ?? '';
        $webhook->name = $data['name'] ?? '';
        $webhook->description = $data['description'] ?? null;
        $webhook->url = $data['url'] ?? '';
        $webhook->method = $data['method'] ?? 'POST';
        $webhook->secret = $data['secret'] ?? null;
        $webhook->timeoutMs = (int)($data['timeout_ms'] ?? 30000);
        $webhook->events = $data['events'] ?? [];
        $webhook->eventFilters = $data['event_filters'] ?? null;
        $webhook->status = $data['status'] ?? 'active';
        $webhook->retryPolicy = $data['retry_policy'] ?? [];
        $webhook->headers = $data['headers'] ?? null;
        $webhook->authType = $data['auth_type'] ?? 'none';
        $webhook->authConfig = $data['auth_config'] ?? null;
        $webhook->totalDeliveries = (int)($data['total_deliveries'] ?? 0);
        $webhook->successfulDeliveries = (int)($data['successful_deliveries'] ?? 0);
        $webhook->failedDeliveries = (int)($data['failed_deliveries'] ?? 0);
        $webhook->avgResponseTimeMs = isset($data['avg_response_time_ms'])
            ? (float)$data['avg_response_time_ms']
            : null;
        $webhook->lastDeliveryAt = $data['last_delivery_at'] ?? null;
        $webhook->lastDeliveryStatus = $data['last_delivery_status'] ?? null;
        $webhook->lastDeliveryResponseCode = isset($data['last_delivery_response_code'])
            ? (int)$data['last_delivery_response_code']
            : null;
        $webhook->lastDeliveryError = $data['last_delivery_error'] ?? null;
        $webhook->tenantId = $data['tenant_id'] ?? '';
        $webhook->workspaceId = $data['workspace_id'] ?? null;
        $webhook->createdBy = $data['created_by'] ?? '';
        $webhook->updatedBy = $data['updated_by'] ?? '';
        $webhook->createdAt = $data['created_at'] ?? '';
        $webhook->updatedAt = $data['updated_at'] ?? '';
        $webhook->schedule = $data['schedule'] ?? null;
        $webhook->metadata = $data['metadata'] ?? null;

        return $webhook;
    }

    /**
     * Convert webhook to array for API requests
     *
     * @param bool $forUpdate Whether this is for an update request (excludes read-only fields)
     * @return array
     */
    public function toArray(bool $forUpdate = false): array
    {
        $data = [
            'name' => $this->name,
            'description' => $this->description,
            'url' => $this->url,
            'method' => $this->method,
            'secret' => $this->secret,
            'timeout_ms' => $this->timeoutMs,
            'events' => $this->events,
            'event_filters' => $this->eventFilters,
            'status' => $this->status,
            'retry_policy' => $this->retryPolicy,
            'headers' => $this->headers,
            'auth_type' => $this->authType,
            'auth_config' => $this->authConfig,
            'schedule' => $this->schedule,
            'metadata' => $this->metadata,
        ];

        // Include ID for updates
        if ($forUpdate && !empty($this->id)) {
            $data['id'] = $this->id;
        }

        // Remove null values
        return array_filter($data, function ($value) {
            return $value !== null;
        });
    }

    /**
     * Get success rate as percentage
     *
     * @return float
     */
    public function getSuccessRate(): float
    {
        if ($this->totalDeliveries === 0) {
            return 0.0;
        }

        return ($this->successfulDeliveries / $this->totalDeliveries) * 100;
    }

    /**
     * Check if webhook is active
     *
     * @return bool
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if webhook has authentication configured
     *
     * @return bool
     */
    public function hasAuth(): bool
    {
        return $this->authType !== 'none' && !empty($this->authConfig);
    }

    /**
     * Get formatted retry policy
     *
     * @return array
     */
    public function getRetryPolicy(): array
    {
        return array_merge([
            'max_attempts' => 3,
            'initial_delay_ms' => 1000,
            'max_delay_ms' => 60000,
            'backoff_strategy' => 'exponential',
        ], $this->retryPolicy);
    }
}
