<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Webhook statistics entity
 */
class WebhookStats
{
    /** @var string */
    public $webhookId;

    /** @var int */
    public $totalDeliveries;

    /** @var int */
    public $successfulDeliveries;

    /** @var int */
    public $failedDeliveries;

    /** @var float */
    public $successRate;

    /** @var float */
    public $avgResponseTimeMs;

    /** @var string|null */
    public $lastDeliveryAt;

    /** @var string|null */
    public $lastSuccessAt;

    /** @var string|null */
    public $lastFailureAt;

    /**
     * Create WebhookStats from array data
     *
     * @param array $data
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $stats = new self();

        $stats->webhookId = $data['webhook_id'] ?? '';
        $stats->totalDeliveries = (int)($data['total_deliveries'] ?? 0);
        $stats->successfulDeliveries = (int)($data['successful_deliveries'] ?? 0);
        $stats->failedDeliveries = (int)($data['failed_deliveries'] ?? 0);
        $stats->successRate = (float)($data['success_rate'] ?? 0.0);
        $stats->avgResponseTimeMs = (float)($data['avg_response_time_ms'] ?? 0.0);
        $stats->lastDeliveryAt = $data['last_delivery_at'] ?? null;
        $stats->lastSuccessAt = $data['last_success_at'] ?? null;
        $stats->lastFailureAt = $data['last_failure_at'] ?? null;

        return $stats;
    }

    /**
     * Convert to array
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'webhook_id' => $this->webhookId,
            'total_deliveries' => $this->totalDeliveries,
            'successful_deliveries' => $this->successfulDeliveries,
            'failed_deliveries' => $this->failedDeliveries,
            'success_rate' => $this->successRate,
            'avg_response_time_ms' => $this->avgResponseTimeMs,
            'last_delivery_at' => $this->lastDeliveryAt,
            'last_success_at' => $this->lastSuccessAt,
            'last_failure_at' => $this->lastFailureAt,
        ];
    }

    /**
     * Get formatted success rate as percentage string
     *
     * @param int $decimals
     * @return string
     */
    public function getFormattedSuccessRate(int $decimals = 1): string
    {
        return number_format($this->successRate, $decimals) . '%';
    }
}
