<?php

declare(strict_types=1);

namespace Intufind\AI\Entities;

/**
 * Webhook test result entity
 */
class WebhookTestResult
{
    /** @var bool */
    public $success;

    /** @var int|null */
    public $responseCode;

    /** @var int */
    public $responseTimeMs;

    /** @var string|null */
    public $responseBody;

    /** @var string|null */
    public $errorMessage;

    /** @var string */
    public $timestamp;

    /**
     * Create WebhookTestResult from array data
     *
     * @param array $data
     * @return self
     */
    public static function fromArray(array $data): self
    {
        $result = new self();

        $result->success = (bool)($data['success'] ?? false);
        $result->responseCode = isset($data['response_code']) ? (int)$data['response_code'] : null;
        $result->responseTimeMs = (int)($data['response_time_ms'] ?? 0);
        $result->responseBody = $data['response_body'] ?? null;
        $result->errorMessage = $data['error_message'] ?? null;
        $result->timestamp = $data['timestamp'] ?? '';

        return $result;
    }

    /**
     * Convert to array
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'success' => $this->success,
            'response_code' => $this->responseCode,
            'response_time_ms' => $this->responseTimeMs,
            'response_body' => $this->responseBody,
            'error_message' => $this->errorMessage,
            'timestamp' => $this->timestamp,
        ];
    }

    /**
     * Check if the test was successful
     *
     * @return bool
     */
    public function isSuccessful(): bool
    {
        return $this->success;
    }

    /**
     * Get formatted response time
     *
     * @return string
     */
    public function getFormattedResponseTime(): string
    {
        if ($this->responseTimeMs < 1000) {
            return $this->responseTimeMs . 'ms';
        }

        return number_format($this->responseTimeMs / 1000, 2) . 's';
    }
}
