<?php

declare(strict_types=1);

namespace Intufind\AI\Exceptions;

use Throwable;

/**
 * Exception thrown when the user's free trial has expired
 *
 * This exception includes helpful properties to guide users to upgrade:
 * - trialEndedAt: When the trial ended
 * - upgradeUrl: URL to upgrade their subscription
 */
class TrialExpiredException extends ApiException
{
    /** @var string|null */
    private $trialEndedAt;

    /** @var string|null */
    private $upgradeUrl;

    public function __construct(
        string $message = 'Your free trial has ended',
        int $statusCode = 402,
        ?Throwable $previous = null,
        array $errorDetails = []
    ) {
        parent::__construct($message, $statusCode, $previous, $errorDetails);
        $this->trialEndedAt = $errorDetails['trial_ended_at'] ?? null;
        $this->upgradeUrl = $errorDetails['upgrade_url'] ?? null;
    }

    /**
     * Get when the trial ended
     *
     * @return string|null ISO 8601 timestamp
     */
    public function getTrialEndedAt(): ?string
    {
        return $this->trialEndedAt;
    }

    /**
     * Get the URL to upgrade the subscription
     *
     * @return string|null
     */
    public function getUpgradeUrl(): ?string
    {
        return $this->upgradeUrl;
    }
}
