<?php

declare(strict_types=1);

namespace Intufind\AI\Http;

use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use Intufind\AI\Config\Configuration;
use Intufind\AI\Exceptions\ApiException;
use Intufind\AI\Exceptions\AuthenticationException;
use Intufind\AI\Exceptions\NetworkException;
use Intufind\AI\Exceptions\RateLimitException;
use Intufind\AI\Exceptions\TrialExpiredException;
use Intufind\AI\Exceptions\ValidationException;
use Psr\Http\Message\ResponseInterface;

class HttpClient
{
    private Configuration $config;
    private GuzzleClient $client;

    public function __construct(Configuration $config)
    {
        $this->config = $config;
        $this->client = $this->createGuzzleClient();
    }

    public function get(string $endpoint, array $query = [], array $headers = []): array
    {
        return $this->request('GET', $endpoint, [
            'query' => $query,
            'headers' => array_merge($this->config->getDefaultHeaders(), $headers),
        ]);
    }

    public function post(string $endpoint, array $data = [], array $headers = []): array
    {
        return $this->request('POST', $endpoint, [
            'json' => $data,
            'headers' => array_merge($this->config->getDefaultHeaders(), $headers),
        ]);
    }

    public function put(string $endpoint, array $data = [], array $headers = []): array
    {
        return $this->request('PUT', $endpoint, [
            'json' => $data,
            'headers' => array_merge($this->config->getDefaultHeaders(), $headers),
        ]);
    }

    public function delete(string $endpoint, array $data = [], array $headers = []): array
    {
        $options = [
            'headers' => array_merge($this->config->getDefaultHeaders(), $headers),
        ];

        // Support body for bulk delete operations
        if (!empty($data)) {
            $options['json'] = $data;
        }

        return $this->request('DELETE', $endpoint, $options);
    }

    public function stream(string $endpoint, array $data, callable $callback, array $headers = []): void
    {
        $streamingEndpoint = $this->config->getStreamingEndpoint();
        if (!$streamingEndpoint) {
            throw new ApiException('Streaming endpoint not configured');
        }

        $url = rtrim($streamingEndpoint, '/') . '/' . ltrim($endpoint, '/');

        $options = [
            'json' => $data,
            'headers' => array_merge($this->config->getDefaultHeaders(), $headers),
            'stream' => true,
            'timeout' => 0,
        ];

        try {
            $response = $this->client->request('POST', $url, $options);
            $body = $response->getBody();

            while (!$body->eof()) {
                $line = $this->readLine($body);
                if ($line !== '') {
                    $decoded = json_decode($line, true);
                    if ($decoded !== null) {
                        $callback($decoded);
                    }
                }
            }
        } catch (RequestException $e) {
            $this->handleException($e);
        }
    }

    private function request(string $method, string $endpoint, array $options = []): array
    {
        $url = rtrim($this->config->getApiEndpoint(), '/') . '/' . ltrim($endpoint, '/');

        $attempts = 0;
        $maxAttempts = $this->config->getRetryAttempts() + 1;

        while ($attempts < $maxAttempts) {
            try {
                $response = $this->client->request($method, $url, $options);
                return $this->parseResponse($response);
            } catch (RequestException $e) {
                $attempts++;

                // Don't retry on client errors (4xx) except rate limiting
                // This includes 401 (auth), 402 (trial expired), 400 (validation), etc.
                // Note: ClientException always has a response
                if ($e instanceof ClientException && $e->getResponse()->getStatusCode() !== 429) {
                    $this->handleException($e);
                }

                if ($attempts >= $maxAttempts) {
                    $this->handleException($e);
                }

                if ($attempts < $maxAttempts) {
                    usleep(min(1000000 * pow(2, $attempts - 1), 5000000));
                }
            }
        }

        throw new ApiException('Maximum retry attempts exceeded');
    }

    private function createGuzzleClient(): GuzzleClient
    {
        $stack = HandlerStack::create();

        if ($this->config->isDebug()) {
            $stack->push(
                Middleware::log(
                    $this->config->getLogger(),
                    new \GuzzleHttp\MessageFormatter('{method} {uri} HTTP/{version} {req_body}')
                )
            );
        }

        $options = array_merge([
            'timeout' => $this->config->getTimeout(),
            'connect_timeout' => $this->config->getConnectTimeout(),
            'handler' => $stack,
            'http_errors' => true,
        ], $this->config->getHttpClientOptions());

        return new GuzzleClient($options);
    }

    private function parseResponse(ResponseInterface $response): array
    {
        $content = $response->getBody()->getContents();
        $decoded = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new ApiException('Invalid JSON response: ' . json_last_error_msg());
        }

        // Extract 'data' field from API response envelope if present
        if (is_array($decoded) && array_key_exists('data', $decoded)) {
            return $decoded['data'] ?? [];
        }

        return $decoded ?: [];
    }

    private function handleException(RequestException $e): void
    {
        // ConnectException indicates network-level failures (DNS, connection refused, etc.)
        // @phpstan-ignore instanceof.alwaysFalse (defensive check for connection errors)
        if ($e instanceof ConnectException) {
            throw new NetworkException('Connection failed: ' . $e->getMessage(), 0, $e);
        }

        $response = $e->getResponse();
        $statusCode = $response ? $response->getStatusCode() : 0;
        $responseBody = $response ? $response->getBody()->getContents() : '';

        $errorMessage = $e->getMessage();
        $errorDetails = [];

        if ($responseBody) {
            $decoded = json_decode($responseBody, true);
            if ($decoded && isset($decoded['error'])) {
                $errorMessage = $decoded['error'];
            } elseif ($decoded && isset($decoded['message'])) {
                $errorMessage = $decoded['message'];
            }
            $errorDetails = $decoded ?: [];
        }

        switch ($statusCode) {
            case 400:
                throw new ValidationException($errorMessage, $statusCode, $e);
            case 401:
                throw new AuthenticationException($errorMessage, $statusCode, $e);
            case 402:
                throw new TrialExpiredException($errorMessage, $statusCode, $e, $errorDetails);
            case 429:
                throw new RateLimitException($errorMessage, $statusCode, $e);
            case 0:
                throw new NetworkException($errorMessage, $statusCode, $e);
            default:
                throw new ApiException($errorMessage, $statusCode, $e, $errorDetails);
        }
    }

    /**
     * @param \Psr\Http\Message\StreamInterface $stream
     */
    private function readLine(\Psr\Http\Message\StreamInterface $stream): string
    {
        $line = '';
        while (!$stream->eof()) {
            $char = $stream->read(1);
            if ($char === "\n") {
                break;
            }
            $line .= $char;
        }
        return rtrim($line, "\r");
    }
}
