<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Config\Configuration;
use Intufind\AI\Http\HttpClient;

abstract class BaseService
{
    protected HttpClient $httpClient;
    protected Configuration $config;

    public function __construct(HttpClient $httpClient, Configuration $config)
    {
        $this->httpClient = $httpClient;
        $this->config = $config;
    }

    protected function buildSearchQuery(array $params): array
    {
        $query = [];

        foreach (['limit', 'offset', 'categories', 'tags', 'status', 'post_type'] as $key) {
            if (isset($params[$key])) {
                $query[$key] = is_array($params[$key]) ? implode(',', $params[$key]) : $params[$key];
            }
        }

        return $query;
    }

    protected function buildSearchBody(array $params): array
    {
        $body = [];

        if (isset($params['text'])) {
            $body['text'] = $params['text'];
        }

        if (isset($params['filters'])) {
            $body['filters'] = $params['filters'];
        }

        if (isset($params['limit'])) {
            $body['limit'] = (int) $params['limit'];
        }

        if (isset($params['useAgent'])) {
            $body['useAgent'] = (bool) $params['useAgent'];
        }

        if (isset($params['searchContext'])) {
            $body['searchContext'] = $params['searchContext'];
        }

        // Faceted search parameters
        if (isset($params['includeFacets'])) {
            $body['includeFacets'] = (bool) $params['includeFacets'];
        }

        if (isset($params['facetFields']) && is_array($params['facetFields'])) {
            $body['facetFields'] = $params['facetFields'];
        }

        if (isset($params['facetSize'])) {
            $body['facetSize'] = (int) $params['facetSize'];
        }

        return $body;
    }

    protected function validateRequired(array $params, array $required): void
    {
        foreach ($required as $key) {
            if (!isset($params[$key]) || (is_string($params[$key]) && trim($params[$key]) === '')) {
                throw new \InvalidArgumentException("Parameter '{$key}' is required");
            }
        }
    }

    protected function cleanArray(array $data): array
    {
        return array_filter($data, function ($value) {
            return $value !== null && $value !== '';
        });
    }

    protected function handleBulkResult(array $response): array
    {
        return [
            'message' => $response['message'] ?? '',
            'successful_ids' => $response['successful_ids'] ?? [],
            'failed_items' => $response['failed_items'] ?? [],
        ];
    }

    protected function handleDeleteResult(array $response): array
    {
        return [
            'message' => $response['message'] ?? '',
            'found_count' => $response['found_count'] ?? 0,
            'deleted_count' => $response['deleted_count'] ?? 0,
            'deleted_ids' => $response['deleted_ids'] ?? [],
        ];
    }
}
