<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\ChatResponse;
use Intufind\AI\Entities\StreamingChunk;

class ChatService extends BaseService
{
    public function send(array $params): ChatResponse
    {
        $this->validateRequired($params, ['message', 'threadId']);
        $body = $this->buildChatBody($params);
        $response = $this->httpClient->post('chat', $body);
        return ChatResponse::fromArray($response);
    }

    public function sendStreaming(array $params, callable $callback): void
    {
        $this->validateRequired($params, ['message', 'threadId']);

        if (!$this->config->getStreamingEndpoint()) {
            throw new \InvalidArgumentException('Streaming endpoint not configured');
        }

        $body = $this->buildChatBody($params);

        $this->httpClient->stream('', $body, function ($chunk) use ($callback) {
            if (isset($chunk['type'])) {
                $callback([
                    'type' => $chunk['type'],
                    'data' => $chunk['data'] ?? null,
                    'metadata' => $chunk['metadata'] ?? null
                ]);
            } else {
                $callback($chunk);
            }
        });
    }

    /**
     * Send a streaming chat message with typed StreamingChunk objects
     *
     * This is a convenience wrapper around sendStreaming that provides
     * StreamingChunk objects instead of raw arrays.
     *
     * @param array $params Chat parameters (message, threadId, etc.)
     * @param callable $callback Callback receiving StreamingChunk objects
     */
    public function sendStreamingTyped(array $params, callable $callback): void
    {
        $this->sendStreaming($params, function ($chunk) use ($callback) {
            $callback(StreamingChunk::fromArray($chunk));
        });
    }

    public function sendStreamingGenerator(array $params): \Generator
    {
        $chunks = [];
        $error = null;

        $this->sendStreaming($params, function ($chunk) use (&$chunks, &$error) {
            if (isset($chunk['type']) && $chunk['type'] === 'error') {
                $error = $chunk['data']['error'] ?? 'Unknown error';
                return;
            }

            // Skip completion signals, but include all content chunks
            if (isset($chunk['type']) && $chunk['type'] === 'complete') {
                return;
            }

            // Skip progress indicators
            if (isset($chunk['type']) && $chunk['type'] === 'progress') {
                return;
            }

            $chunks[] = $chunk;
        });

        if ($error) {
            throw new \RuntimeException("Streaming error: {$error}");
        }

        foreach ($chunks as $chunk) {
            yield $chunk;
        }
    }

    /**
     * Send a streaming chat message and yield typed StreamingChunk objects
     *
     * @param array $params Chat parameters (message, threadId, etc.)
     * @return \Generator<StreamingChunk>
     */
    public function sendStreamingGeneratorTyped(array $params): \Generator
    {
        foreach ($this->sendStreamingGenerator($params) as $chunk) {
            yield StreamingChunk::fromArray($chunk);
        }
    }

    public function getThread(string $threadId, array $params = []): array
    {
        if (empty($threadId)) {
            throw new \InvalidArgumentException('Thread ID is required');
        }

        $query = array_filter([
            'limit' => $params['limit'] ?? null,
            'offset' => $params['offset'] ?? null,
        ]);

        try {
            return $this->httpClient->get("threads/{$threadId}", $query);
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), '404') !== false) {
                return ['messages' => [], 'total' => 0];
            }
            throw $e;
        }
    }

    public function deleteThread(string $threadId): array
    {
        if (empty($threadId)) {
            throw new \InvalidArgumentException('Thread ID is required');
        }
        return $this->httpClient->delete("threads/{$threadId}");
    }

    public function searchThreads(array $params): array
    {
        $body = $this->buildSearchBody($params);
        return $this->httpClient->post('threads/search', $body);
    }

    private function buildChatBody(array $params): array
    {
        $body = [
            'message' => $params['message'],
            'threadId' => $params['threadId'],
        ];

        $optionalFields = [
            'workspaceId', 'userJwt', 'productFilter', 'postFilter', 'promptId',
            'customInstructions', 'workspaceDescription'
        ];

        foreach ($optionalFields as $field) {
            if (isset($params[$field])) {
                $body[$field] = $params[$field];
            }
        }

        if (isset($params['liveAgentEnabled'])) {
            $body['liveAgentEnabled'] = (bool) $params['liveAgentEnabled'];
        }

        if (isset($params['liveAgentEscalationPrompt'])) {
            $body['liveAgentEscalationPrompt'] = (string) $params['liveAgentEscalationPrompt'];
        }

        if (isset($params['contactFormEnabled'])) {
            $body['contactFormEnabled'] = (bool) $params['contactFormEnabled'];
        }

        return $body;
    }

    public function startConversation(array $params): ChatResponse
    {
        if (!isset($params['threadId'])) {
            $params['threadId'] = 'thread_' . uniqid() . '_' . time();
        }
        return $this->send($params);
    }

    public function continueConversation(string $threadId, string $message, array $params = []): ChatResponse
    {
        $params['threadId'] = $threadId;
        $params['message'] = $message;
        return $this->send($params);
    }

    public function sendWithProductContext(
        string $message,
        string $threadId,
        array $productFilter = [],
        array $params = []
    ): ChatResponse {
        $params['message'] = $message;
        $params['threadId'] = $threadId;
        $params['productFilter'] = $productFilter;
        return $this->send($params);
    }

    public function sendWithContentContext(
        string $message,
        string $threadId,
        array $postFilter = [],
        array $params = []
    ): ChatResponse {
        $params['message'] = $message;
        $params['threadId'] = $threadId;
        $params['postFilter'] = $postFilter;
        return $this->send($params);
    }

    public function sendWithPrompt(
        string $message,
        string $threadId,
        string $promptId,
        array $params = []
    ): ChatResponse {
        $params['message'] = $message;
        $params['threadId'] = $threadId;
        $params['promptId'] = $promptId;
        return $this->send($params);
    }
}
