<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\ColorScheme;

/**
 * Config service for AI-powered configuration helpers
 */
class ConfigService extends BaseService
{
    /**
     * Generate a complete color scheme for the chat widget
     *
     * @param string $primaryColor Primary brand color (hex format, e.g., '#0084ff')
     * @param string|null $secondaryColor Optional secondary brand color (hex format)
     * @return ColorScheme
     * @throws \InvalidArgumentException If color format is invalid
     */
    public function generateColorScheme(string $primaryColor, ?string $secondaryColor = null): ColorScheme
    {
        // Validate hex color format
        if (!$this->isValidHexColor($primaryColor)) {
            throw new \InvalidArgumentException("Primary color must be a valid hex color (e.g., '#0084ff')");
        }

        if ($secondaryColor !== null && !$this->isValidHexColor($secondaryColor)) {
            throw new \InvalidArgumentException("Secondary color must be a valid hex color (e.g., '#0084ff')");
        }

        $body = ['primaryColor' => $primaryColor];
        if ($secondaryColor !== null) {
            $body['secondaryColor'] = $secondaryColor;
        }

        $response = $this->httpClient->post('config/color-scheme', $body);

        return ColorScheme::fromArray($response);
    }

    /**
     * Validate hex color format
     *
     * @param string $color Color to validate
     * @return bool
     */
    private function isValidHexColor(string $color): bool
    {
        return (bool) preg_match('/^#[0-9A-Fa-f]{6}$/', $color);
    }
}
