<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\Feedback;

/**
 * Feedback service for managing user feedback
 */
class FeedbackService extends BaseService
{
    /**
     * Submit feedback
     *
     * @param array $params Feedback parameters
     * @return array
     */
    public function submit(array $params): array
    {
        $this->validateRequired($params, ['rating', 'source']);

        $body = $this->buildFeedbackBody($params);
        // HttpClient throws on errors, so if we get here it succeeded
        // HttpClient::parseResponse extracts 'data' field, so $response IS the data
        $response = $this->httpClient->post('feedback', $body);

        return [
            'feedback_id' => $response['feedback_id'] ?? $response['id'] ?? null,
            'message' => $response['message'] ?? 'Feedback submitted successfully',
        ];
    }

    /**
     * Search feedback (admin only)
     *
     * @param array $params Search parameters
     * @return array
     */
    public function search(array $params = []): array
    {
        $query = $this->buildFeedbackSearchQuery($params);
        // HttpClient::parseResponse extracts 'data' field, so $response IS the data
        $response = $this->httpClient->get('feedback', $query);

        $results = [];
        if (isset($response['results']) && is_array($response['results'])) {
            foreach ($response['results'] as $item) {
                $results[] = Feedback::fromArray($item);
            }
        }

        return [
            'results' => $results,
            'total_count' => $response['total_count'] ?? 0,
            'limit' => $response['limit'] ?? 10,
            'offset' => $response['offset'] ?? 0,
        ];
    }

    /**
     * Get feedback analytics
     *
     * @param array $params Analytics parameters (date_from, date_to)
     * @return array
     */
    public function getAnalytics(array $params = []): array
    {
        $query = [];
        if (isset($params['date_from'])) {
            $query['date_from'] = $params['date_from'];
        }
        if (isset($params['date_to'])) {
            $query['date_to'] = $params['date_to'];
        }

        $response = $this->httpClient->get('feedback/analytics', $query);

        return $response;
    }

    /**
     * Get feedback for a specific thread
     *
     * @param string $threadId Thread ID
     * @return array
     */
    public function getThreadFeedback(string $threadId): array
    {
        if (empty($threadId)) {
            throw new \InvalidArgumentException('Thread ID is required');
        }

        // HttpClient::parseResponse extracts 'data' field, so $response IS the data
        $response = $this->httpClient->get("feedback/thread/{$threadId}");

        $results = [];
        if (isset($response['feedback']) && is_array($response['feedback'])) {
            foreach ($response['feedback'] as $item) {
                $results[] = Feedback::fromArray($item);
            }
        }

        return $results;
    }

    /**
     * Delete feedback
     *
     * @param string $feedbackId Feedback ID
     * @param array $params Optional parameters (userJwt for ownership check)
     * @return array
     */
    public function delete(string $feedbackId, array $params = []): array
    {
        if (empty($feedbackId)) {
            throw new \InvalidArgumentException('Feedback ID is required');
        }

        // HttpClient throws on errors, so if we get here it succeeded
        // HttpClient::parseResponse extracts 'data' field, so $response IS the data
        $response = $this->httpClient->delete("feedback/{$feedbackId}");

        return [
            'id' => $response['id'] ?? $feedbackId,
        ];
    }

    /**
     * Build feedback body for submission
     *
     * @param array $params
     * @return array
     */
    private function buildFeedbackBody(array $params): array
    {
        $body = [];

        // Required fields
        $body['rating'] = $params['rating'];
        $body['source'] = $params['source'];

        // Optional fields
        $optionalFields = [
            'thread_id', 'content_id', 'content_type', 'comment', 'category',
            'query_text', 'model_used', 'user_id', 'session_id', 'userJwt'
        ];

        foreach ($optionalFields as $field) {
            if (isset($params[$field]) && $params[$field] !== '') {
                $body[$field] = $params[$field];
            }
        }

        return $this->cleanArray($body);
    }

    /**
     * Build query parameters for feedback search
     *
     * @param array $params
     * @return array
     */
    private function buildFeedbackSearchQuery(array $params): array
    {
        $query = [];

        $searchFields = [
            'query', 'rating', 'source', 'content_id', 'content_type',
            'thread_id', 'user_id', 'date_from', 'date_to', 'limit', 'offset',
            'sort_by', 'sort_order', 'include_aggregations'
        ];

        foreach ($searchFields as $field) {
            if (isset($params[$field]) && $params[$field] !== '') {
                if ($field === 'include_aggregations') {
                    $query[$field] = $params[$field] ? 'true' : 'false';
                } else {
                    $query[$field] = $params[$field];
                }
            }
        }

        return $query;
    }
}
