<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\Post;
use Intufind\AI\Entities\SearchResult;

/**
 * Post service for managing posts/content
 */
class PostService extends BaseService
{
    /**
     * Search posts
     *
     * @param array $params Search parameters
     * @return SearchResult
     */
    public function search(array $params): SearchResult
    {
        $this->validateRequired($params, ['text']);

        $body = $this->buildSearchBody($params);
        $response = $this->httpClient->post('post/search', $body);

        return SearchResult::fromArray($response, Post::class);
    }

    /**
     * Get post by ID
     *
     * @param string $id
     * @return Post|null
     */
    public function getById(string $id): ?Post
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Post ID is required');
        }

        try {
            $response = $this->httpClient->get("post/{$id}");
            $data = $response;
            return Post::fromArray($data);
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), '404') !== false || strpos($e->getMessage(), 'not found') !== false) {
                return null;
            }
            throw $e;
        }
    }

    /**
     * Create or update a post
     *
     * @param array|Post $post
     * @return array
     */
    public function upsert($post): array
    {
        if ($post instanceof Post) {
            $data = $post->toArray();
        } else {
            $data = $post;
        }

        $this->validateRequired($data, ['id', 'title']);

        // HttpClient throws on errors, returns data on success
        return $this->httpClient->post('post', $this->cleanArray($data));
    }

    /**
     * Delete a post by ID
     *
     * @param string $id
     * @return array
     */
    public function delete(string $id): array
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Post ID is required');
        }

        // HttpClient throws on errors, returns data on success
        return $this->httpClient->delete("post/{$id}");
    }

    /**
     * Bulk upsert posts
     *
     * @param array $posts Array of posts
     * @return array
     */
    public function bulkUpsert(array $posts): array
    {
        if (empty($posts)) {
            throw new \InvalidArgumentException('Posts array cannot be empty');
        }

        $data = [];
        foreach ($posts as $post) {
            if ($post instanceof Post) {
                $data[] = $post->toArray();
            } else {
                $data[] = $post;
            }
        }

        $response = $this->httpClient->post('post/bulk', ['entities' => $data]);
        return $this->handleBulkResult($response);
    }

    /**
     * Bulk delete posts by IDs
     *
     * @param array $ids Array of post IDs
     * @return array
     */
    public function bulkDelete(array $ids): array
    {
        if (empty($ids)) {
            throw new \InvalidArgumentException('IDs array cannot be empty');
        }

        $response = $this->httpClient->delete('post/bulk', ['ids' => $ids]);
        return $this->handleBulkResult($response);
    }

    /**
     * Delete posts by query
     *
     * @param array $filters OpenSearch query filters
     * @return array
     */
    public function deleteByQuery(array $filters): array
    {
        if (empty($filters)) {
            throw new \InvalidArgumentException('Filters array cannot be empty');
        }

        $response = $this->httpClient->post('post/delete-by-query', ['filters' => $filters]);
        return $this->handleDeleteResult($response);
    }

    /**
     * Get post IDs (lightweight operation for cleanup)
     *
     * @param array $params Query parameters
     * @return array
     */
    public function getIds(array $params = []): array
    {
        $query = $this->buildSearchQuery($params);
        $response = $this->httpClient->get('post/ids', $query);
        return $response;
    }

    /**
     * Search posts with AI agent enhancement
     *
     * @param array $params Search parameters
     * @return SearchResult
     */
    public function searchWithAgent(array $params): SearchResult
    {
        $params['useAgent'] = true;
        return $this->search($params);
    }

    /**
     * Find similar posts
     *
     * @param string $postId Post ID to find similar posts for
     * @param array $params Additional search parameters
     * @return SearchResult
     */
    public function findSimilar(string $postId, array $params = []): SearchResult
    {
        $post = $this->getById($postId);
        if (!$post) {
            throw new \InvalidArgumentException("Post with ID '{$postId}' not found");
        }

        // Build search text from post data
        $searchParts = array_filter([
            $post->title,
            $post->excerpt,
            !empty($post->categories) ? 'Categories: ' . implode(', ', $post->categories) : null,
            !empty($post->tags) ? 'Tags: ' . implode(', ', $post->tags) : null,
        ]);

        $params['text'] = implode(' ', $searchParts);

        // Exclude the original post from results
        $params['filters'] = array_merge($params['filters'] ?? [], [
            'must_not' => [['term' => ['external_id' => $postId]]]
        ]);

        return $this->search($params);
    }

    /**
     * Get posts by category
     *
     * @param string|array $categories Category name(s)
     * @param array $params Additional search parameters
     * @return SearchResult
     */
    public function getByCategory($categories, array $params = []): SearchResult
    {
        if (is_string($categories)) {
            $categories = [$categories];
        }

        $params['filters'] = array_merge($params['filters'] ?? [], [
            'terms' => ['categories' => $categories]
        ]);

        // Use a generic search if no text provided
        if (!isset($params['text'])) {
            $params['text'] = implode(' ', $categories);
        }

        return $this->search($params);
    }

    /**
     * Get posts by author
     *
     * @param string $authorId Author ID
     * @param array $params Additional search parameters
     * @return SearchResult
     */
    public function getByAuthor(string $authorId, array $params = []): SearchResult
    {
        $params['filters'] = array_merge($params['filters'] ?? [], [
            'term' => ['author_id' => $authorId]
        ]);

        if (!isset($params['text'])) {
            $params['text'] = 'posts';
        }

        return $this->search($params);
    }

    /**
     * Get published posts
     *
     * @param array $params Additional search parameters
     * @return SearchResult
     */
    public function getPublished(array $params = []): SearchResult
    {
        $params['filters'] = array_merge($params['filters'] ?? [], [
            'term' => ['status' => 'publish']
        ]);

        if (!isset($params['text'])) {
            $params['text'] = 'published posts';
        }

        return $this->search($params);
    }

    /**
     * Get draft posts
     *
     * @param array $params Additional search parameters
     * @return SearchResult
     */
    public function getDrafts(array $params = []): SearchResult
    {
        $params['filters'] = array_merge($params['filters'] ?? [], [
            'term' => ['status' => 'draft']
        ]);

        if (!isset($params['text'])) {
            $params['text'] = 'draft posts';
        }

        return $this->search($params);
    }

    /**
     * Get posts by type
     *
     * @param string $postType Post type
     * @param array $params Additional search parameters
     * @return SearchResult
     */
    public function getByType(string $postType, array $params = []): SearchResult
    {
        $params['filters'] = array_merge($params['filters'] ?? [], [
            'term' => ['post_type' => $postType]
        ]);

        if (!isset($params['text'])) {
            $params['text'] = $postType;
        }

        return $this->search($params);
    }

    /**
     * Get recent posts
     *
     * @param int $days Number of days to look back
     * @param array $params Additional search parameters
     * @return SearchResult
     */
    public function getRecent(int $days = 30, array $params = []): SearchResult
    {
        $timestamp = strtotime("-{$days} days");
        $date = date('Y-m-d\TH:i:s\Z', $timestamp !== false ? $timestamp : time());

        $params['filters'] = array_merge($params['filters'] ?? [], [
            'range' => ['published_at' => ['gte' => $date]]
        ]);

        if (!isset($params['text'])) {
            $params['text'] = 'recent posts';
        }

        return $this->search($params);
    }
}
