<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\Product;
use Intufind\AI\Entities\SearchResult;

class ProductService extends BaseService
{
    public function search(array $params): SearchResult
    {
        $this->validateRequired($params, ['text']);
        $body = $this->buildSearchBody($params);
        $response = $this->httpClient->post('product/search', $body);
        return SearchResult::fromArray($response, Product::class);
    }

    public function getById(string $id): ?Product
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Product ID is required');
        }

        try {
            $response = $this->httpClient->get("product/{$id}");
            return Product::fromArray($response);
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), '404') !== false || strpos($e->getMessage(), 'not found') !== false) {
                return null;
            }
            throw $e;
        }
    }

    /**
     * @param Product|array $product
     */
    public function upsert(Product|array $product): array
    {
        if ($product instanceof Product) {
            $data = $product->toArray();
        } else {
            $data = $product;
        }

        $this->validateRequired($data, ['id', 'name']);
        return $this->httpClient->post('product', $this->cleanArray($data));
    }

    public function delete(string $id): array
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Product ID is required');
        }
        return $this->httpClient->delete("product/{$id}");
    }

    public function bulkUpsert(array $products): array
    {
        if (empty($products)) {
            throw new \InvalidArgumentException('Products array cannot be empty');
        }

        $data = [];
        foreach ($products as $product) {
            if ($product instanceof Product) {
                $data[] = $product->toArray();
            } else {
                $data[] = $product;
            }
        }

        $response = $this->httpClient->post('product/bulk', ['entities' => $data]);
        return $this->handleBulkResult($response);
    }

    public function bulkDelete(array $ids): array
    {
        if (empty($ids)) {
            throw new \InvalidArgumentException('IDs array cannot be empty');
        }

        $response = $this->httpClient->delete('product/bulk', ['ids' => $ids]);
        return $this->handleBulkResult($response);
    }

    public function deleteByQuery(array $filters): array
    {
        if (empty($filters)) {
            throw new \InvalidArgumentException('Filters array cannot be empty');
        }

        $response = $this->httpClient->post('product/delete-by-query', ['filters' => $filters]);
        return $this->handleDeleteResult($response);
    }

    public function getIds(array $params = []): array
    {
        $query = $this->buildSearchQuery($params);
        return $this->httpClient->get('product/ids', $query);
    }

    public function searchWithAgent(array $params): SearchResult
    {
        $params['useAgent'] = true;
        return $this->search($params);
    }

    /**
     * Search products with faceted aggregations for building filter UI.
     *
     * @param array $params Search parameters including:
     *   - text: Search query (required)
     *   - filters: Array of filters
     *   - limit: Max results
     *   - facetFields: Array of fields to facet on (e.g., ['categories', 'brands', 'attributes.color', 'price_ranges'])
     *   - facetSize: Max buckets per facet (default: 20)
     * @return SearchResult Result with facets property populated
     */
    public function searchWithFacets(array $params): SearchResult
    {
        $params['includeFacets'] = true;
        return $this->search($params);
    }

    public function findSimilar(string $productId, array $params = []): SearchResult
    {
        $product = $this->getById($productId);
        if (!$product) {
            throw new \InvalidArgumentException("Product with ID '{$productId}' not found");
        }

        $searchParts = array_filter([
            $product->name,
            $product->excerpt,
            !empty($product->categories) ? 'Categories: ' . implode(', ', $product->categories) : null,
            !empty($product->tags) ? 'Tags: ' . implode(', ', $product->tags) : null,
            !empty($product->brands) ? 'Brands: ' . implode(', ', $product->brands) : null,
        ]);

        $params['text'] = implode(' ', $searchParts);
        return $this->search($params);
    }

    /**
     * @param string|array<string> $categories
     */
    public function getByCategory(string|array $categories, array $params = []): SearchResult
    {
        if (is_string($categories)) {
            $categories = [$categories];
        }

        $params['filters'] = array_merge($params['filters'] ?? [], ['categories' => $categories]);

        if (!isset($params['text'])) {
            $params['text'] = implode(' ', $categories);
        }

        return $this->search($params);
    }

    public function getByPriceRange(?float $minPrice, ?float $maxPrice, array $params = []): SearchResult
    {
        $priceFilter = [];
        if ($minPrice !== null) {
            $priceFilter['gte'] = $minPrice;
        }
        if ($maxPrice !== null) {
            $priceFilter['lte'] = $maxPrice;
        }

        if (!empty($priceFilter)) {
            $params['filters'] = array_merge($params['filters'] ?? [], ['price' => $priceFilter]);
        }

        if (!isset($params['text'])) {
            $params['text'] = 'products';
        }

        return $this->search($params);
    }

    public function getOnSale(array $params = []): SearchResult
    {
        $params['filters'] = array_merge($params['filters'] ?? [], ['on_sale' => true]);

        if (!isset($params['text'])) {
            $params['text'] = 'sale products';
        }

        return $this->search($params);
    }
}
