<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

/**
 * Prompt service for managing chat prompts
 */
class PromptService extends BaseService
{
    /**
     * Get a single prompt by ID
     *
     * @param string $id Prompt ID
     * @return array|null Prompt data or null if not found
     */
    public function get(string $id): ?array
    {
        if (empty($id)) {
            return null;
        }

        $response = $this->httpClient->get("prompt/{$id}");

        return $response;
    }

    /**
     * Search prompts by text query
     *
     * @param string $query Search query
     * @param array $filters Additional search filters
     * @param int $limit Maximum number of results
     * @return array Search results
     */
    public function search(string $query, array $filters = [], int $limit = 20): array
    {
        // Return empty results for empty queries to avoid cloud API errors
        if (trim($query) === '') {
            return [];
        }

        $payload = [
            'text' => $query,
            'limit' => $limit
        ];

        if (!empty($filters)) {
            $payload['filters'] = $filters;
        }

        $response = $this->httpClient->post('prompt/search', $payload);

        return $response;
    }

    /**
     * List prompts with pagination
     *
     * @param int $page Page number (1-based)
     * @param int $perPage Items per page
     * @return array Paginated list of prompts
     */
    public function list(int $page = 1, int $perPage = 20): array
    {
        $params = [
            'page' => $page,
            'per_page' => $perPage
        ];

        $response = $this->httpClient->get('prompt', $params);

        return $response;
    }

    /**
     * Create or update a prompt
     *
     * @param array $promptData Prompt data
     * @return array Response from cloud service
     */
    public function upsert(array $promptData): array
    {
        $this->validateRequired($promptData, ['title', 'prompt']);

        // Allow server to auto-generate ID. Only require prompt text.
        if (empty($promptData['prompt'])) {
            throw new \InvalidArgumentException('Prompt text is required');
        }

        // Remove empty ID field to let server auto-generate
        if (empty($promptData['id'])) {
            unset($promptData['id']);
        }

        $response = $this->httpClient->post('prompt', $promptData);

        return $response;
    }

    /**
     * Delete a prompt by ID
     *
     * @param string $id Prompt ID
     * @return array Response from cloud service
     */
    public function delete(string $id): array
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Prompt ID is required');
        }

        $response = $this->httpClient->delete("prompt/{$id}");

        return $response;
    }

    /**
     * Get initial prompts for chat widget
     *
     * @param array $filters Optional filters
     * @param int $limit Maximum number of prompts
     * @return array Initial prompts
     */
    public function getInitial(array $filters = [], int $limit = 10): array
    {
        $params = [
            'limit' => $limit,
            'metadata.initial' => 'true'
        ];

        // Add any additional metadata filters
        if (!empty($filters)) {
            foreach ($filters as $key => $value) {
                if (strpos($key, 'metadata.') === 0) {
                    $params[$key] = is_bool($value) ? ($value ? 'true' : 'false') : $value;
                } else {
                    $params["metadata.{$key}"] = is_bool($value) ? ($value ? 'true' : 'false') : $value;
                }
            }
        }

        $response = $this->httpClient->get('prompt', $params);

        $data = $response;
        return $data['results'] ?? $data;
    }

    /**
     * Get prompts by category
     *
     * @param string $category Category name
     * @param int $limit Maximum number of prompts
     * @return array Prompts in category
     */
    public function getByCategory(string $category, int $limit = 20): array
    {
        $payload = [
            'category' => $category,
            'limit' => $limit
        ];

        $response = $this->httpClient->post('prompt/category', $payload);

        return $response;
    }

    /**
     * Get prompts by tags
     *
     * @param array $tags Array of tag names
     * @param int $limit Maximum number of prompts
     * @return array Prompts matching tags
     */
    public function getByTags(array $tags, int $limit = 20): array
    {
        $payload = [
            'tags' => $tags,
            'limit' => $limit
        ];

        $response = $this->httpClient->post('prompt/tags', $payload);

        return $response;
    }
}
