<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

/**
 * Provisioning service for tenant activation via multiple providers
 *
 * Supported providers:
 * - EDD (Easy Digital Downloads): License key based activation
 * - Manual: Admin-provisioned tenants (requires admin API key)
 *
 * AUTHENTICATION MODEL:
 *
 * - Provider credentials (license key, etc.): Used for initial activation
 *   Passed in request body to the provisioning endpoint
 *
 * - API Key: Returned from activate(), used for ALL subsequent API calls
 *   Store this securely and configure it in the SDK
 *
 * TYPICAL FLOW:
 *
 * 1. User enters provider credentials (e.g., EDD license key)
 * 2. Call activateEdd($licenseKey) or activateManual($config)
 * 3. Store returned apiKey securely (e.g., WordPress options, database)
 * 4. Create new Configuration with api_key set
 * 5. All subsequent calls (chat, search, sync, etc.) use API key
 */
class ProvisioningService extends BaseService
{
    /**
     * Supported provider types
     */
    public const PROVIDER_EDD = 'edd';
    public const PROVIDER_MANUAL = 'manual';
    public const PROVIDER_SHOPIFY = 'shopify';

    /**
     * Activate via EDD license key
     *
     * This method calls the EDD provisioning endpoint which:
     * 1. Validates the license key with EDD
     * 2. Creates or retrieves an existing tenant
     * 3. Generates and returns an API key
     *
     * IMPORTANT: Store the returned API key and use it for all subsequent
     * API calls. The license key is NOT valid for API authentication.
     *
     * @param string $licenseKey EDD license key to activate
     * @param string|null $siteUrl Optional site URL (defaults to WordPress home_url)
     * @return array{
     *     success: bool,
     *     message: string,
     *     apiKey?: string,
     *     apiKeyId?: string,
     *     tenantId?: string,
     *     workspaceId?: string,
     *     tier?: string,
     *     isNew?: bool
     * }
     * @throws \InvalidArgumentException If license key is empty
     */
    public function activateEdd(string $licenseKey, ?string $siteUrl = null): array
    {
        if (empty($licenseKey)) {
            throw new \InvalidArgumentException('License key is required');
        }

        $response = $this->httpClient->post('provision/edd', [
            'license_key' => $licenseKey,
            'site_url' => $siteUrl ?? $this->getSiteUrl(),
        ]);

        return $this->normalizeProvisionResponse($response);
    }

    /**
     * Activate via manual provisioning (admin-created tenant)
     *
     * This method requires an admin API key in the configuration.
     * It creates a new tenant with the specified configuration.
     *
     * @param array{
     *     tenantId?: string,
     *     tier?: string,
     *     label?: string,
     *     metadata?: array<string, mixed>
     * } $config Tenant configuration
     * @return array{
     *     success: bool,
     *     message: string,
     *     apiKey?: string,
     *     apiKeyId?: string,
     *     tenantId?: string,
     *     workspaceId?: string,
     *     tier?: string,
     *     isNew?: bool
     * }
     */
    public function activateManual(array $config = []): array
    {
        if (!$this->config->hasAuthentication()) {
            throw new \InvalidArgumentException('Admin API key is required for manual provisioning');
        }

        $response = $this->httpClient->post('provision/manual', $config);

        return $this->normalizeProvisionResponse($response);
    }

    /**
     * Deactivate the current tenant and clean up cloud resources
     *
     * REQUIRES a valid API key in the configuration for authentication.
     *
     * This method:
     * 1. Authenticates using your API key
     * 2. Marks the subscription as cancelled
     * 3. Optionally purges indexed data
     * 4. For EDD: deactivates the license on the licensing server
     *
     * After deactivation, the API key is no longer valid.
     *
     * @param string $provider Provider type (edd, manual, shopify)
     * @param array{
     *     license_key?: string,
     *     site_url?: string,
     *     purge_data?: bool
     * } $options Provider-specific options
     * @return array{
     *     success: bool,
     *     message?: string
     * }
     * @throws \InvalidArgumentException If API key not configured
     */
    public function deactivate(string $provider = self::PROVIDER_EDD, array $options = []): array
    {
        if (!$this->config->hasAuthentication()) {
            throw new \InvalidArgumentException('API key is required for deactivation');
        }

        $payload = array_merge([
            'provider' => $provider,
        ], $options);

        // Add site_url if not provided and we're in WordPress
        if (!isset($payload['site_url'])) {
            $payload['site_url'] = $this->getSiteUrl();
        }

        $response = $this->httpClient->post('deprovision', $payload);

        return [
            'success' => $response['success'] ?? true,
            'message' => $response['message'] ?? 'Deactivation successful',
        ];
    }

    /**
     * Deactivate EDD license specifically
     *
     * @param string $licenseKey License key to deactivate
     * @param bool $purgeData Whether to delete all indexed data
     * @return array
     */
    public function deactivateEdd(string $licenseKey, bool $purgeData = false): array
    {
        return $this->deactivate(self::PROVIDER_EDD, [
            'license_key' => $licenseKey,
            'purge_data' => $purgeData,
        ]);
    }

    /**
     * Get list of supported providers
     *
     * @return string[]
     */
    public function getSupportedProviders(): array
    {
        return [
            self::PROVIDER_EDD,
            self::PROVIDER_MANUAL,
            self::PROVIDER_SHOPIFY,
        ];
    }

    /**
     * Normalize the provisioning response to a consistent format
     *
     * @param array $response Raw response from the API
     * @return array{success: bool, message: string, apiKey?: string,
     *               apiKeyId?: string, tenantId?: string, workspaceId?: string,
     *               tier?: string, isNew?: bool}
     */
    private function normalizeProvisionResponse(array $response): array
    {
        $data = $response['data'] ?? $response;

        return [
            'success' => true,
            'message' => 'Provisioning successful',
            'apiKey' => $data['apiKey'] ?? null,
            'apiKeyId' => $data['apiKeyId'] ?? null,
            'tenantId' => $data['tenantId'] ?? null,
            'workspaceId' => $data['workspaceId'] ?? null,
            'tier' => $data['tier'] ?? null,
            'isNew' => $data['isNew'] ?? false,
        ];
    }

    /**
     * Get the site URL for provisioning operations
     *
     * @return string Site URL
     */
    private function getSiteUrl(): string
    {
        // Check if we're in a WordPress context
        if (!function_exists('home_url')) {
            return '';
        }

        /** @var string $url */
        $url = home_url();
        if (empty($url) && function_exists('site_url')) {
            /** @var string $url */
            $url = site_url();
        }
        if (empty($url) && function_exists('get_option')) {
            /** @var string $url */
            $url = (string) get_option('home', '');
        }

        return $url !== '' ? $url : '';
    }
}
