<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\Product;

/**
 * Recommendations service for AI-powered product recommendations
 */
class RecommendationsService extends BaseService
{
    /**
     * Get product recommendations
     *
     * @param array $params Recommendation parameters
     * @return array
     */
    public function getProducts(array $params): array
    {
        $body = $this->buildRecommendationBody($params);
        $response = $this->httpClient->post('recommendations/products', $body);

        $data = $response;

        // Convert product results to Product entities if available
        if (isset($data['recommendations']) && is_array($data['recommendations'])) {
            $data['recommendations'] = array_map(function ($item) {
                if (isset($item['product']) && is_array($item['product'])) {
                    $item['product'] = Product::fromArray($item['product']);
                }
                return $item;
            }, $data['recommendations']);
        }

        return $data;
    }

    /**
     * Get recommendations based on user behavior
     *
     * @param string $userId User ID
     * @param array $params Additional parameters
     * @return array
     */
    public function getForUser(string $userId, array $params = []): array
    {
        if (empty($userId)) {
            throw new \InvalidArgumentException('User ID is required');
        }

        $params['user_id'] = $userId;
        return $this->getProducts($params);
    }

    /**
     * Get recommendations based on a specific product
     *
     * @param string $productId Product ID
     * @param array $params Additional parameters
     * @return array
     */
    public function getForProduct(string $productId, array $params = []): array
    {
        if (empty($productId)) {
            throw new \InvalidArgumentException('Product ID is required');
        }

        $params['product_id'] = $productId;
        return $this->getProducts($params);
    }

    /**
     * Get recommendations based on shopping cart contents
     *
     * @param array $cartItems Array of product IDs or product objects
     * @param array $params Additional parameters
     * @return array
     */
    public function getForCart(array $cartItems, array $params = []): array
    {
        if (empty($cartItems)) {
            throw new \InvalidArgumentException('Cart items cannot be empty');
        }

        // Extract product IDs from cart items
        $productIds = array_map(function ($item) {
            if (is_array($item)) {
                return $item['id'] ?? $item['product_id'] ?? null;
            } elseif ($item instanceof Product) {
                return $item->id;
            } else {
                return $item; // Assume it's already a product ID
            }
        }, $cartItems);

        $productIds = array_filter($productIds);

        if (empty($productIds)) {
            throw new \InvalidArgumentException('No valid product IDs found in cart items');
        }

        $params['cart_items'] = $productIds;
        return $this->getProducts($params);
    }

    /**
     * Get recommendations based on search query
     *
     * @param string $query Search query
     * @param array $params Additional parameters
     * @return array
     */
    public function getForQuery(string $query, array $params = []): array
    {
        if (empty($query)) {
            throw new \InvalidArgumentException('Search query is required');
        }

        $params['query'] = $query;
        return $this->getProducts($params);
    }

    /**
     * Get recommendations based on category
     *
     * @param string|array $categories Category name(s)
     * @param array $params Additional parameters
     * @return array
     */
    public function getForCategory($categories, array $params = []): array
    {
        if (is_string($categories)) {
            $categories = [$categories];
        }

        if (empty($categories)) {
            throw new \InvalidArgumentException('Categories cannot be empty');
        }

        $params['categories'] = $categories;
        return $this->getProducts($params);
    }

    /**
     * Get trending products
     *
     * @param array $params Additional parameters
     * @return array
     */
    public function getTrending(array $params = []): array
    {
        $params['strategy'] = 'trending';
        return $this->getProducts($params);
    }

    /**
     * Get popular products
     *
     * @param array $params Additional parameters
     * @return array
     */
    public function getPopular(array $params = []): array
    {
        $params['strategy'] = 'popular';
        return $this->getProducts($params);
    }

    /**
     * Get recently viewed products for a user
     *
     * @param string $userId User ID
     * @param array $params Additional parameters
     * @return array
     */
    public function getRecentlyViewed(string $userId, array $params = []): array
    {
        if (empty($userId)) {
            throw new \InvalidArgumentException('User ID is required');
        }

        $params['user_id'] = $userId;
        $params['strategy'] = 'recently_viewed';
        return $this->getProducts($params);
    }

    /**
     * Get personalized recommendations
     *
     * @param string $userId User ID
     * @param array $params Additional parameters
     * @return array
     */
    public function getPersonalized(string $userId, array $params = []): array
    {
        if (empty($userId)) {
            throw new \InvalidArgumentException('User ID is required');
        }

        $params['user_id'] = $userId;
        $params['strategy'] = 'personalized';
        return $this->getProducts($params);
    }

    /**
     * Get cross-sell recommendations
     *
     * @param string $productId Product ID
     * @param array $params Additional parameters
     * @return array
     */
    public function getCrossSell(string $productId, array $params = []): array
    {
        if (empty($productId)) {
            throw new \InvalidArgumentException('Product ID is required');
        }

        $params['product_id'] = $productId;
        $params['strategy'] = 'cross_sell';
        return $this->getProducts($params);
    }

    /**
     * Get upsell recommendations
     *
     * @param string $productId Product ID
     * @param array $params Additional parameters
     * @return array
     */
    public function getUpsell(string $productId, array $params = []): array
    {
        if (empty($productId)) {
            throw new \InvalidArgumentException('Product ID is required');
        }

        $params['product_id'] = $productId;
        $params['strategy'] = 'upsell';
        return $this->getProducts($params);
    }

    /**
     * Get recommendations with price filtering
     *
     * @param array $params Base parameters
     * @param float|null $minPrice Minimum price
     * @param float|null $maxPrice Maximum price
     * @return array
     */
    public function getWithPriceFilter(array $params, ?float $minPrice = null, ?float $maxPrice = null): array
    {
        if ($minPrice !== null) {
            $params['price_min'] = $minPrice;
        }

        if ($maxPrice !== null) {
            $params['price_max'] = $maxPrice;
        }

        return $this->getProducts($params);
    }

    /**
     * Build recommendation request body
     *
     * @param array $params
     * @return array
     */
    private function buildRecommendationBody(array $params): array
    {
        $body = [];

        // Map snake_case parameters to camelCase for cloud API
        $paramMapping = [
            'user_id' => 'userId',
            'product_id' => 'productId',
            'query' => 'query',
            'categories' => 'categories',
            'cart_items' => 'cartItems',
            'strategy' => 'strategy',
            'limit' => 'maxRecommendations', // Cloud API uses maxRecommendations
            'price_min' => 'priceMin',
            'price_max' => 'priceMax',
            'filters' => 'filters',
            'excludeProductIds' => 'excludeProductIds',
            'weights' => 'weights',
            'basketAffinity' => 'basketAffinity',
            'complimentaryProducts' => 'complimentaryProducts'
        ];

        foreach ($paramMapping as $phpParam => $cloudParam) {
            if (isset($params[$phpParam])) {
                $body[$cloudParam] = $params[$phpParam];
            }
        }

        // Set default maxRecommendations if not provided
        if (!isset($body['maxRecommendations'])) {
            $body['maxRecommendations'] = 10;
        }

        return $body;
    }
}
