<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

/**
 * Taxonomy service for managing taxonomy terms
 */
class TaxonomyService extends BaseService
{
    /**
     * Bulk upsert taxonomy terms
     *
     * @param array $terms Array of taxonomy terms to upsert
     * @return array Response from cloud service
     * @throws \InvalidArgumentException If terms array is empty
     */
    public function bulkUpsert(array $terms): array
    {
        if (empty($terms)) {
            throw new \InvalidArgumentException('Terms array cannot be empty');
        }

        $response = $this->httpClient->post('taxonomy/bulk', $terms);

        return $this->handleBulkResult($response);
    }

    /**
     * Bulk delete taxonomy terms
     *
     * @param array $items Array of items to delete with taxonomy_name and id
     * @return array Response from cloud service
     * @throws \InvalidArgumentException If items array is empty
     */
    public function bulkDelete(array $items): array
    {
        if (empty($items)) {
            throw new \InvalidArgumentException('Items array cannot be empty');
        }

        $response = $this->httpClient->delete('taxonomy/bulk', ['items' => $items]);

        return $this->handleDeleteResult($response);
    }

    /**
     * Delete taxonomy term by ID
     *
     * @param string $termId Term ID to delete
     * @return array Response from cloud service
     */
    public function delete(string $termId): array
    {
        $response = $this->httpClient->delete("taxonomy/{$termId}");

        return $this->handleDeleteResult($response);
    }

    /**
     * Get taxonomy term IDs for cleanup operations
     *
     * @param array $filters Optional filters (taxonomy_name, etc.)
     * @param int $limit Maximum number of IDs to return
     * @param int $offset Offset for pagination
     * @return array Response with taxonomy term IDs
     */
    public function getIds(array $filters = [], int $limit = 10000, int $offset = 0): array
    {
        $queryParams = [
            'limit' => $limit,
            'offset' => $offset,
        ];

        // Add filters as query parameters
        if (!empty($filters['taxonomy_name'])) {
            $queryParams['taxonomy_name'] = $filters['taxonomy_name'];
        }

        $response = $this->httpClient->get('taxonomy/ids', $queryParams);

        return $response;
    }

    /**
     * Delete taxonomy terms by query filters
     *
     * @param array $filters OpenSearch filters for deletion
     * @return array Response from cloud service
     * @throws \InvalidArgumentException If filters array is empty
     */
    public function deleteByQuery(array $filters): array
    {
        if (empty($filters)) {
            throw new \InvalidArgumentException('Filters array cannot be empty');
        }

        $response = $this->httpClient->post('taxonomy/delete-by-query', [
            'filters' => $filters
        ]);

        return $this->handleDeleteResult($response);
    }

    /**
     * Search taxonomy terms
     *
     * @param string $text Search text
     * @param array $filters Optional filters
     * @param int $limit Maximum results
     * @return array Search results
     */
    public function search(string $text, array $filters = [], int $limit = 10): array
    {
        $this->validateRequired(['text' => $text], ['text']);

        $body = [
            'text' => $text,
            'filters' => $filters,
            'limit' => $limit,
        ];

        $response = $this->httpClient->post('taxonomy/search', $body);

        return $response;
    }
}
