<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Config\Configuration;
use Intufind\AI\Http\HttpClient;

class TenantService extends BaseService
{
    public function __construct(HttpClient $httpClient, Configuration $config)
    {
        parent::__construct($httpClient, $config);
    }

    // ========================================
    // STATUS & USAGE
    // ========================================

    /**
     * Get tenant status including subscription validation
     * Lightweight call - use for checking if subscription is valid
     *
     * @return array{
     *   status: string,
     *   timestamp: string,
     *   publishableKey: ?string,
     *   subscription: array{
     *     valid: bool,
     *     tier: string,
     *     expires: ?string,
     *     limits: array,
     *     provider?: string,
     *     validation?: array
     *   }
     * }
     */
    public function getStatus(): array
    {
        /** @var array{status: string, timestamp: string, publishableKey: string|null, subscription: array{valid: bool, tier: string, expires: string|null, limits: array, provider?: string, validation?: array}} */
        return $this->httpClient->post('/tenant/status', []);
    }

    /**
     * Get comprehensive usage summary with document counts and tier limits
     * Heavier call - use for dashboard/usage displays
     *
     * @return array{
     *   tier: string,
     *   resetTime: int,
     *   features: array,
     *   tier_features: array,
     *   indexes: array,
     *   data_summary: array
     * }
     */
    public function getUsage(): array
    {
        /** @var array{tier: string, resetTime: int, features: array, tier_features: array, indexes: array, data_summary: array} */
        return $this->httpClient->get('/tenant/usage');
    }

    /**
     * Check if subscription is valid
     *
     * @return bool
     */
    public function isSubscriptionValid(): bool
    {
        $status = $this->getStatus();
        return $status['subscription']['valid'];
    }

    /**
     * Get current tier
     *
     * @return string
     */
    public function getTier(): string
    {
        $status = $this->getStatus();
        return $status['subscription']['tier'];
    }

    /**
     * Get indexed document counts
     *
     * @return array{products: int, posts: int, users: int, taxonomies: int, posts_by_type: array}
     */
    public function getIndexedStats(): array
    {
        $usage = $this->getUsage();
        $indexes = $usage['indexes'];

        return [
            'products' => $indexes['products']['count'] ?? 0,
            'posts' => $indexes['posts']['count'] ?? 0,
            'users' => $indexes['users']['count'] ?? 0,
            'taxonomies' => $indexes['taxonomies']['count'] ?? 0,
            'posts_by_type' => $indexes['posts']['by_type'] ?? [],
        ];
    }

    /**
     * Get usage data with features and limits
     *
     * @return array
     */
    public function getUsageData(): array
    {
        return $this->getUsage();
    }

    /**
     * Get status for a specific feature
     *
     * @param string $featureType Feature type (e.g., 'search', 'chat', 'products')
     * @return array{available: bool, used: int, limit: int, remaining: int, percentage: int}
     * @throws \Exception If feature not found
     */
    public function getFeatureStatus(string $featureType): array
    {
        $usage = $this->getUsage();
        $features = $usage['features'];

        $feature = null;
        foreach ($features as $f) {
            if ($f['type'] === $featureType) {
                $feature = $f;
                break;
            }
        }

        if ($feature === null) {
            throw new \Exception("Feature '{$featureType}' not found");
        }

        $featureUsage = $feature['usage'] ?? [];
        $current = $featureUsage['current'] ?? 0;
        $limit = $featureUsage['limit'] ?? 0;

        return [
            'available' => (bool) ($featureUsage['available'] ?? false),
            'used' => (int) $current,
            'limit' => (int) $limit,
            'remaining' => (int) max(0, $limit - $current),
            'percentage' => (int) ($featureUsage['percentage'] ?? 0),
        ];
    }

    // ========================================
    // MCP CONFIGURATION
    // ========================================

    /**
     * Get MCP (Model Context Protocol) configuration for this tenant
     *
     * @return array{mcp: array, configured: bool}
     */
    public function getMcpConfig(): array
    {
        /** @var array{mcp: array, configured: bool} */
        return $this->httpClient->get('/tenant/mcp');
    }

    /**
     * Store/update MCP configuration
     *
     * @param array{
     *   ordersUri?: string,
     *   inventoryUri?: string,
     *   shippingUri?: string,
     *   supportUri?: string,
     *   accountUri?: string,
     *   analyticsUri?: string
     * } $config MCP server URIs for e-commerce operations
     * @return array{message: string, mcp: array}
     */
    public function storeMcpConfig(array $config): array
    {
        /** @var array{message: string, mcp: array} */
        return $this->httpClient->put('/tenant/mcp', $config);
    }

    /**
     * Delete MCP configuration
     *
     * @return array{message: string}
     */
    public function deleteMcpConfig(): array
    {
        /** @var array{message: string} */
        return $this->httpClient->delete('/tenant/mcp');
    }

    /**
     * Check if MCP is configured for this tenant
     *
     * @return bool
     */
    public function isMcpConfigured(): bool
    {
        $response = $this->getMcpConfig();
        return $response['configured'];
    }

    // ========================================
    // LIVE AGENT CREDENTIALS
    // ========================================

    /**
     * Store live agent credentials
     *
     * @param array{
     *   provider: string,
     *   apiToken?: string,
     *   webhookSecret?: string,
     *   subdomain?: string,
     *   integrationId?: string,
     *   botToken?: string,
     *   channelId?: string,
     *   signingSecret?: string
     * } $credentials
     * @return array
     */
    public function storeLiveAgentCredentials(array $credentials): array
    {
        return $this->httpClient->post('/tenant/live-agent/credentials', $credentials);
    }

    /**
     * Delete live agent credentials
     *
     * @param string $provider Provider name (zendesk, intercom, drift, slack)
     * @return array
     */
    public function deleteLiveAgentCredentials(string $provider): array
    {
        return $this->httpClient->delete('/tenant/live-agent/credentials?provider=' . urlencode($provider));
    }

    /**
     * Check live agent credentials status
     *
     * @param string $provider Provider name
     * @return array{configured: bool, provider: string, configurationType?: string,
     *               availabilitySchedule?: string, details?: array}
     */
    public function checkLiveAgentCredentialsStatus(string $provider): array
    {
        /** @var array{configured: bool, provider: string, configurationType?: string, availabilitySchedule?: string, details?: array} */
        return $this->httpClient->get('/tenant/live-agent/credentials/status', ['provider' => $provider]);
    }

    // ========================================
    // LIVE AGENT AVAILABILITY SCHEDULE
    // ========================================

    /**
     * Get the current live agent availability schedule
     *
     * @return array{schedule: ?string, configured: bool}
     *
     * @example
     * $response = $client->tenant()->getLiveAgentSchedule();
     * echo $response['schedule']; // "M-F 9AM-5PM EST"
     */
    public function getLiveAgentSchedule(): array
    {
        /** @var array{schedule: string|null, configured: bool} */
        return $this->httpClient->get('/tenant/live-agent/schedule');
    }

    /**
     * Update the live agent availability schedule
     *
     * The schedule is interpreted by the AI to determine if live agents
     * are available at any given time. Use natural language to describe
     * your business hours.
     *
     * @param string $schedule Free-form schedule description
     * @return array{schedule: ?string, configured: bool, message: string}
     *
     * @example
     * // Set business hours
     * $client->tenant()->updateLiveAgentSchedule("M-F 9AM-5PM EST");
     *
     * // With holidays
     * $client->tenant()->updateLiveAgentSchedule(
     *     "Monday through Friday, 8AM to 5PM EST, closed on 12/25 and 1/1"
     * );
     *
     * // Complex schedule
     * $client->tenant()->updateLiveAgentSchedule(
     *     "Weekdays 9-5, Saturdays 10-2, closed Sundays and US federal holidays"
     * );
     */
    public function updateLiveAgentSchedule(string $schedule): array
    {
        /** @var array{schedule: string|null, configured: bool, message: string} */
        return $this->httpClient->put('/tenant/live-agent/schedule', ['schedule' => $schedule]);
    }

    /**
     * Delete the live agent availability schedule
     *
     * When no schedule is configured, live agent availability is determined
     * solely by real-time online status from the provider (Slack, Zendesk, etc.)
     *
     * @return array{deleted: bool, message: string}
     */
    public function deleteLiveAgentSchedule(): array
    {
        /** @var array{deleted: bool, message: string} */
        return $this->httpClient->delete('/tenant/live-agent/schedule');
    }
}
