<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

/**
 * Thread service for managing conversation threads
 */
class ThreadService extends BaseService
{
    /**
     * Search conversations with advanced filtering and pagination
     *
     * @param array $request Search request parameters
     *   - query: Optional text search across conversation content
     *   - limit: Number of results to return (default 10, max 100)
     *   - offset: Pagination offset (default 0)
     *   - userId: Filter by specific user ID
     *   - dateFrom: ISO date string - conversations after this date
     *   - dateTo: ISO date string - conversations before this date
     *   - minMessages: Minimum message count
     *   - maxMessages: Maximum message count
     *   - minTokens: Minimum token count
     *   - maxTokens: Maximum token count
     *   - sortBy: Sort field (timestamp, message_count, token_count)
     *   - sortOrder: Sort direction (asc, desc)
     * @return array Search results with pagination info
     */
    public function search(array $request = []): array
    {
        $response = $this->httpClient->post('threads/search', $request);
        return $response;
    }

    /**
     * Get detailed conversation by thread ID
     *
     * @param string $threadId Thread ID (external_id from cloud)
     * @return array|null Thread details or null if not found
     */
    public function get(string $threadId): ?array
    {
        if (empty($threadId)) {
            throw new \InvalidArgumentException('Thread ID is required');
        }

        return $this->httpClient->get("threads/{$threadId}");
    }

    /**
     * Get message history for a thread (for polling during live agent handoff)
     *
     * Returns formatted message history including:
     * - User messages
     * - AI assistant messages
     * - Human agent messages (with agent metadata)
     *
     * @param string $threadId Thread ID (external_id from cloud)
     * @return array Message history with metadata
     *   - threadId: The thread ID
     *   - messages: Array of message objects with role, content, timestamp, source, agentInfo
     *   - count: Total number of messages
     */
    public function getMessages(string $threadId): array
    {
        if (empty($threadId)) {
            throw new \InvalidArgumentException('Thread ID is required');
        }

        $response = $this->httpClient->get("threads/{$threadId}/messages");
        return $response;
    }

    /**
     * End an active live agent handoff session
     *
     * Terminates the live agent connection and returns the user to AI chat.
     * Clears Redis session and updates OpenSearch thread metadata.
     *
     * @param string $threadId Thread ID
     * @return array Response data
     *   - message: Status message
     * @throws \InvalidArgumentException If thread ID is empty
     */
    public function endHandoff(string $threadId): array
    {
        if (empty($threadId)) {
            throw new \InvalidArgumentException('Thread ID is required');
        }

        $response = $this->httpClient->post("threads/{$threadId}/handoff/end", []);
        return $response;
    }

    /**
     * Delete a conversation thread
     *
     * @param string $threadId Thread ID to delete
     * @return array Response from cloud service
     */
    public function delete(string $threadId): array
    {
        if (empty($threadId)) {
            throw new \InvalidArgumentException('Thread ID is required');
        }

        $response = $this->httpClient->delete("threads/{$threadId}");
        return $response;
    }

    /**
     * Get conversation analytics and insights
     *
     * @return array Analytics data
     */
    public function getAnalytics(): array
    {
        $response = $this->httpClient->get('threads/analytics');
        return $response;
    }

    /**
     * Export conversation data
     *
     * @param array $request Export request parameters
     *   - format: Export format (json, csv) - default: json
     *   - dateFrom: ISO date string - conversations after this date
     *   - dateTo: ISO date string - conversations before this date
     *   - userId: Filter by specific user ID
     *   - includeContent: Include full conversation content - default: false
     * @return array Export data with download information
     */
    public function export(array $request = []): array
    {
        $response = $this->httpClient->post('threads/export', $request);
        return $response;
    }
}
