<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\Webhook;

/**
 * Webhook builder for fluent webhook creation
 */
class WebhookBuilder
{
    /** @var WebhookService */
    private $service;

    /** @var array */
    private $data;

    public function __construct(WebhookService $service, string $name, string $url)
    {
        $this->service = $service;
        $this->data = [
            'name' => $name,
            'url' => $url,
            'events' => [],
        ];
    }

    /**
     * Set webhook description
     *
     * @param string $description
     * @return $this
     */
    public function description(string $description): self
    {
        $this->data['description'] = $description;
        return $this;
    }

    /**
     * Set HTTP method
     *
     * @param string $method
     * @return $this
     */
    public function method(string $method): self
    {
        $this->data['method'] = strtoupper($method);
        return $this;
    }

    /**
     * Add webhook events
     *
     * @param array $events
     * @return $this
     */
    public function events(array $events): self
    {
        $this->data['events'] = array_merge($this->data['events'], $events);
        return $this;
    }

    /**
     * Add single webhook event
     *
     * @param string $event
     * @return $this
     */
    public function event(string $event): self
    {
        $this->data['events'][] = $event;
        return $this;
    }

    /**
     * Set webhook secret
     *
     * @param string $secret
     * @return $this
     */
    public function secret(string $secret): self
    {
        $this->data['secret'] = $secret;
        return $this;
    }

    /**
     * Set timeout in milliseconds
     *
     * @param int $timeoutMs
     * @return $this
     */
    public function timeout(int $timeoutMs): self
    {
        $this->data['timeout_ms'] = $timeoutMs;
        return $this;
    }

    /**
     * Set bearer token authentication
     *
     * @param string $token
     * @return $this
     */
    public function bearerAuth(string $token): self
    {
        $this->data['auth_type'] = 'bearer';
        $this->data['auth_config'] = ['bearer_token' => $token];
        return $this;
    }

    /**
     * Set basic authentication
     *
     * @param string $username
     * @param string $password
     * @return $this
     */
    public function basicAuth(string $username, string $password): self
    {
        $this->data['auth_type'] = 'basic';
        $this->data['auth_config'] = [
            'username' => $username,
            'password' => $password,
        ];
        return $this;
    }

    /**
     * Set API key authentication
     *
     * @param string $apiKey
     * @param string $header
     * @return $this
     */
    public function apiKeyAuth(string $apiKey, string $header = 'X-API-Key'): self
    {
        $this->data['auth_type'] = 'api_key';
        $this->data['auth_config'] = [
            'api_key' => $apiKey,
            'api_key_header' => $header,
        ];
        return $this;
    }

    /**
     * Set custom headers
     *
     * @param array $headers
     * @return $this
     */
    public function headers(array $headers): self
    {
        $this->data['headers'] = $headers;
        return $this;
    }

    /**
     * Set retry policy
     *
     * @param int $maxAttempts
     * @param string $strategy
     * @param int $initialDelayMs
     * @param int $maxDelayMs
     * @return $this
     */
    public function retryPolicy(
        int $maxAttempts = 3,
        string $strategy = 'exponential',
        int $initialDelayMs = 1000,
        int $maxDelayMs = 60000
    ): self {
        $this->data['retry_policy'] = [
            'max_attempts' => $maxAttempts,
            'backoff_strategy' => $strategy,
            'initial_delay_ms' => $initialDelayMs,
            'max_delay_ms' => $maxDelayMs,
        ];
        return $this;
    }

    /**
     * Set event filters
     *
     * @param array $filters
     * @return $this
     */
    public function filters(array $filters): self
    {
        $this->data['event_filters'] = $filters;
        return $this;
    }

    /**
     * Set webhook status
     *
     * @param string $status
     * @return $this
     */
    public function status(string $status): self
    {
        $this->data['status'] = $status;
        return $this;
    }

    /**
     * Create the webhook
     *
     * @return Webhook
     */
    public function create(): Webhook
    {
        return $this->service->create($this->data);
    }
}
