<?php

declare(strict_types=1);

namespace Intufind\AI\Services;

use Intufind\AI\Entities\SearchResult;
use Intufind\AI\Entities\Webhook;
use Intufind\AI\Entities\WebhookStats;
use Intufind\AI\Entities\WebhookTestResult;

/**
 * Webhook service for managing webhook configurations
 */
class WebhookService extends BaseService
{
    /**
     * Create a new webhook
     *
     * @param array $data Webhook data
     * @return Webhook
     */
    public function create(array $data): Webhook
    {
        $this->validateRequired($data, ['name', 'url', 'events']);

        // Validate URL format
        if (!filter_var($data['url'], FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException('Invalid webhook URL format');
        }

        // Validate events array
        if (empty($data['events']) || !is_array($data['events'])) {
            throw new \InvalidArgumentException('At least one event type is required');
        }

        $response = $this->httpClient->post('webhooks', $data);
        $webhookData = $response;

        return Webhook::fromArray($webhookData);
    }

    /**
     * List webhooks with optional filtering and pagination
     *
     * @param array $params Query parameters
     * @return SearchResult
     */
    public function list(array $params = []): SearchResult
    {
        $queryParams = [];

        // Add pagination
        if (isset($params['page'])) {
            $queryParams['page'] = (int)$params['page'];
        }
        if (isset($params['limit'])) {
            $queryParams['limit'] = (int)$params['limit'];
        }

        // Add filters
        $filterKeys = [
            'status', 'events', 'created_by', 'name_contains', 'url_contains',
            'created_after', 'created_before', 'last_delivery_after', 'last_delivery_before'
        ];

        foreach ($filterKeys as $key) {
            if (isset($params[$key])) {
                if ($key === 'events' && is_array($params[$key])) {
                    $queryParams[$key] = implode(',', $params[$key]);
                } else {
                    $queryParams[$key] = $params[$key];
                }
            }
        }

        // Add sorting
        if (isset($params['sort_by'])) {
            $queryParams['sort_by'] = $params['sort_by'];
        }
        if (isset($params['sort_order'])) {
            $queryParams['sort_order'] = $params['sort_order'];
        }

        $response = $this->httpClient->get('webhooks', $queryParams);
        $data = $response;

        return SearchResult::fromArray($data, Webhook::class);
    }

    /**
     * Get webhook by ID
     *
     * @param string $id
     * @return Webhook|null
     */
    public function getById(string $id): ?Webhook
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Webhook ID is required');
        }

        try {
            $response = $this->httpClient->get("webhooks/{$id}");
            $data = $response;
            return Webhook::fromArray($data);
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), '404') !== false || strpos($e->getMessage(), 'not found') !== false) {
                return null;
            }
            throw $e;
        }
    }

    /**
     * Update webhook
     *
     * @param string $id
     * @param array $data
     * @return Webhook
     */
    public function update(string $id, array $data): Webhook
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Webhook ID is required');
        }

        // Validate URL if provided
        if (isset($data['url']) && !filter_var($data['url'], FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException('Invalid webhook URL format');
        }

        // Validate events if provided
        if (isset($data['events'])) {
            if (empty($data['events']) || !is_array($data['events'])) {
                throw new \InvalidArgumentException('Events must be a non-empty array');
            }
        }

        $response = $this->httpClient->put("webhooks/{$id}", $data);
        $webhookData = $response;

        return Webhook::fromArray($webhookData);
    }

    /**
     * Delete webhook
     *
     * @param string $id
     * @return bool
     */
    public function delete(string $id): bool
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Webhook ID is required');
        }

        try {
            $this->httpClient->delete("webhooks/{$id}");
            return true;
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), '404') !== false || strpos($e->getMessage(), 'not found') !== false) {
                return false;
            }
            throw $e;
        }
    }

    /**
     * Test webhook
     *
     * @param string $id
     * @param array|null $payload Optional test payload
     * @return WebhookTestResult
     */
    public function test(string $id, ?array $payload = null): WebhookTestResult
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Webhook ID is required');
        }

        $body = [];
        if ($payload !== null) {
            $body['payload'] = $payload;
        }

        $response = $this->httpClient->post("webhooks/{$id}/test", $body);
        $data = $response;

        return WebhookTestResult::fromArray($data);
    }

    /**
     * Get webhook statistics
     *
     * @param string $id
     * @return WebhookStats|null
     */
    public function getStats(string $id): ?WebhookStats
    {
        if (empty($id)) {
            throw new \InvalidArgumentException('Webhook ID is required');
        }

        try {
            $response = $this->httpClient->get("webhooks/{$id}/stats");
            $data = $response;
            return WebhookStats::fromArray($data);
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), '404') !== false || strpos($e->getMessage(), 'not found') !== false) {
                return null;
            }
            throw $e;
        }
    }

    /**
     * Bulk update webhook status
     *
     * @param array $webhookIds
     * @param string $status
     * @return array
     */
    public function bulkUpdateStatus(array $webhookIds, string $status): array
    {
        if (empty($webhookIds)) {
            throw new \InvalidArgumentException('Webhook IDs are required');
        }

        if (!in_array($status, ['active', 'inactive', 'suspended'])) {
            throw new \InvalidArgumentException('Invalid status. Must be one of: active, inactive, suspended');
        }

        $body = [
            'webhook_ids' => $webhookIds,
            'status' => $status,
        ];

        $response = $this->httpClient->post('webhooks/bulk-status', $body);
        return $response;
    }

    /**
     * Webhook event type constants
     */
    public const EVENT_TYPES = [
        // Product events
        'product.created',
        'product.updated',
        'product.deleted',
        'product.bulk_deleted',

        // Post events
        'post.created',
        'post.updated',
        'post.deleted',
        'post.bulk_deleted',

        // Prompt events
        'prompt.created',
        'prompt.updated',
        'prompt.deleted',
        'prompt.bulk_deleted',

        // Taxonomy events
        'taxonomy.created',
        'taxonomy.updated',
        'taxonomy.deleted',
        'taxonomy.bulk_deleted',

        // Chat events
        'chat.message.sent',
        'chat.message.received',
        'chat.conversation.started',
        'chat.conversation.ended',

        // Thread events
        'thread.created',
        'thread.updated',
        'thread.deleted',
        'thread.exported',

        // Feedback events
        'feedback.submitted',
        'feedback.deleted',

        // Recommendation events
        'recommendation.generated',
        'recommendation.requested',

        // Search events
        'search.performed',
        'search.failed',
    ];

    /**
     * Get available webhook events
     *
     * @return array
     */
    public function getAvailableEvents(): array
    {
        return self::EVENT_TYPES;
    }

    /**
     * Get available authentication types
     *
     * @return array
     */
    public function getAvailableAuthTypes(): array
    {
        return [
            'none' => 'No Authentication',
            'bearer' => 'Bearer Token',
            'basic' => 'Basic Authentication',
            'api_key' => 'API Key',
            'custom' => 'Custom Header',
        ];
    }

    /**
     * Get available retry strategies
     *
     * @return array
     */
    public function getAvailableRetryStrategies(): array
    {
        return [
            'linear' => 'Linear Backoff',
            'exponential' => 'Exponential Backoff',
        ];
    }

    /**
     * Create webhook with builder pattern
     *
     * @param string $name
     * @param string $url
     * @return WebhookBuilder
     */
    public function builder(string $name, string $url): WebhookBuilder
    {
        return new WebhookBuilder($this, $name, $url);
    }
}
