<?php
/**
 * Admin functionality.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin class.
 *
 * Handles admin menu pages, settings registration, and AJAX handlers.
 */
class Intufind_Admin {

	/**
	 * API client instance.
	 *
	 * @var Intufind_API
	 */
	private $api;

	/**
	 * Sync manager instance.
	 *
	 * @var Intufind_Sync
	 */
	private $sync;

	/**
	 * Admin page hook suffix.
	 *
	 * @var string
	 */
	private $page_hook;

	/**
	 * Constructor.
	 *
	 * @param Intufind_API  $api  API client instance.
	 * @param Intufind_Sync $sync Sync manager instance.
	 */
	public function __construct( Intufind_API $api, Intufind_Sync $sync = null ) {
		$this->api  = $api;
		$this->sync = $sync;
	}

	/**
	 * Set the sync manager instance.
	 *
	 * @param Intufind_Sync $sync Sync manager instance.
	 * @return void
	 */
	public function set_sync( Intufind_Sync $sync ) {
		$this->sync = $sync;
	}

	/**
	 * Initialize admin functionality.
	 *
	 * @return void
	 */
	public function init() {
		add_action( 'admin_menu', array( $this, 'add_menu_pages' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_init', array( $this, 'verify_connection_on_plugin_pages' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );

		// AJAX handlers.
		add_action( 'wp_ajax_intufind_validate_key', array( $this, 'ajax_validate_key' ) );
		add_action( 'wp_ajax_intufind_switch_workspace', array( $this, 'ajax_switch_workspace' ) );
		add_action( 'wp_ajax_intufind_disconnect', array( $this, 'ajax_disconnect' ) );
		add_action( 'wp_ajax_intufind_manual_sync', array( $this, 'ajax_manual_sync' ) );
		add_action( 'wp_ajax_intufind_sync_taxonomies', array( $this, 'ajax_sync_taxonomies' ) );
		add_action( 'wp_ajax_intufind_get_sync_stats', array( $this, 'ajax_get_sync_stats' ) );
		add_action( 'wp_ajax_intufind_get_cloud_stats', array( $this, 'ajax_get_cloud_stats' ) );
		add_action( 'wp_ajax_intufind_retry_failed', array( $this, 'ajax_retry_failed' ) );
		add_action( 'wp_ajax_intufind_save_post_types', array( $this, 'ajax_save_post_types' ) );
		add_action( 'wp_ajax_intufind_save_taxonomies', array( $this, 'ajax_save_taxonomies' ) );
		add_action( 'wp_ajax_intufind_save_search_option', array( $this, 'ajax_save_search_option' ) );
		add_action( 'wp_ajax_intufind_save_recommendations_option', array( $this, 'ajax_save_recommendations_option' ) );
		add_action( 'wp_ajax_intufind_save_chat_option', array( $this, 'ajax_save_chat_option' ) );
		add_action( 'wp_ajax_intufind_save_sync_option', array( $this, 'ajax_save_sync_option' ) );
		add_action( 'wp_ajax_intufind_retry_mcp_registration', array( $this, 'ajax_retry_mcp_registration' ) );
		add_action( 'wp_ajax_intufind_dismiss_notice', array( $this, 'ajax_dismiss_notice' ) );

		// WooCommerce activation/deactivation hooks for MCP domain registration.
		add_action( 'activated_plugin', array( $this, 'on_plugin_activated' ), 10, 1 );
		add_action( 'deactivated_plugin', array( $this, 'on_plugin_deactivated' ), 10, 1 );

		// Admin notices.
		add_action( 'admin_notices', array( $this, 'display_initial_sync_notice' ) );
	}

	/**
	 * Add admin menu pages.
	 *
	 * @return void
	 */
	public function add_menu_pages() {
		$this->page_hook = add_menu_page(
			__( 'Intufind', 'intufind' ),
			__( 'Intufind', 'intufind' ),
			'manage_options',
			'intufind',
			array( $this, 'render_settings_page' ),
			$this->get_menu_icon(),
			30
		);

		// Add submenu pages.
		add_submenu_page(
			'intufind',
			__( 'Settings', 'intufind' ),
			__( 'Settings', 'intufind' ),
			'manage_options',
			'intufind',
			array( $this, 'render_settings_page' )
		);

		add_submenu_page(
			'intufind',
			__( 'Knowledge', 'intufind' ),
			__( 'Knowledge', 'intufind' ),
			'manage_options',
			'intufind-sync',
			array( $this, 'render_sync_page' )
		);

		add_submenu_page(
			'intufind',
			__( 'Search', 'intufind' ),
			__( 'Search', 'intufind' ),
			'manage_options',
			'intufind-search',
			array( $this, 'render_search_page' )
		);

		add_submenu_page(
			'intufind',
			__( 'Chat', 'intufind' ),
			__( 'Chat', 'intufind' ),
			'manage_options',
			'intufind-chat',
			array( $this, 'render_chat_page' )
		);

		// Add Recommendations page only if WooCommerce is active.
		if ( class_exists( 'WooCommerce' ) ) {
			add_submenu_page(
				'intufind',
				__( 'Recommendations', 'intufind' ),
				__( 'Recommendations', 'intufind' ),
				'manage_options',
				'intufind-recommendations',
				array( $this, 'render_recommendations_page' )
			);
		}

		add_submenu_page(
			'intufind',
			__( 'Status', 'intufind' ),
			__( 'Status', 'intufind' ),
			'manage_options',
			'intufind-status',
			array( $this, 'render_status_page' )
		);
	}

	/**
	 * Get the menu icon SVG as base64.
	 *
	 * @return string
	 */
	private function get_menu_icon() {
		// AI sparkles icon - represents AI-powered features.
		$svg = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">'
			// Large 4-pointed star (center-right).
			. '<path d="M12 5l1.5 4.5L18 11l-4.5 1.5L12 17l-1.5-4.5L6 11l4.5-1.5L12 5z"/>'
			// Small 4-pointed star (top-left).
			. '<path d="M5 2l.75 2.25L8 5l-2.25.75L5 8l-.75-2.25L2 5l2.25-.75L5 2z"/>'
			. '</svg>';
		return 'data:image/svg+xml;base64,' . base64_encode( $svg );
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 * @return void
	 */
	public function enqueue_assets( $hook ) {
		// Only load on our pages.
		$our_pages = array(
			'toplevel_page_intufind',
			'intufind_page_intufind-sync',
			'intufind_page_intufind-search',
			'intufind_page_intufind-chat',
			'intufind_page_intufind-recommendations',
			'intufind_page_intufind-status',
		);

		if ( ! in_array( $hook, $our_pages, true ) ) {
			return;
		}

		// Enqueue styles.
		wp_enqueue_style(
			'intufind-admin',
			INTUFIND_PLUGIN_URL . 'admin/css/intufind-admin.css',
			array(),
			INTUFIND_VERSION
		);

		// Enqueue scripts.
		wp_enqueue_script(
			'intufind-admin',
			INTUFIND_PLUGIN_URL . 'admin/js/intufind-admin.js',
			array( 'jquery' ),
			INTUFIND_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'intufind-admin',
			'intufindAdmin',
			array(
				'ajaxUrl'     => admin_url( 'admin-ajax.php' ),
				'nonce'       => wp_create_nonce( 'intufind_admin' ),
				'workspaceId' => get_option( INTUFIND_OPTION_WORKSPACE_ID, '' ),
				'strings'     => array(
					'saving'              => __( 'Saving...', 'intufind' ),
					'saved'               => __( 'Saved', 'intufind' ),
					'error'               => __( 'Error', 'intufind' ),
					'validating'          => __( 'Validating...', 'intufind' ),
					'syncing'             => __( 'Syncing', 'intufind' ),
					'syncComplete'        => __( 'Sync complete', 'intufind' ),
					'syncStarted'         => __( 'Sync started...', 'intufind' ),
					'retrying'            => __( 'Retrying...', 'intufind' ),
					'confirmSwitch'       => __( 'Content synced to the current workspace will remain there. Continue?', 'intufind' ),
					'workspaceExists'     => __( 'A workspace with this ID already exists.', 'intufind' ),
					'reconnect'           => __( 'Reconnect to it', 'intufind' ),
					'createNew'           => __( 'Create new workspace', 'intufind' ),
					'showPassword'        => __( 'Show password', 'intufind' ),
					'hidePassword'        => __( 'Hide password', 'intufind' ),
					'connectingAndSyncing' => __( 'Connecting and syncing content...', 'intufind' ),
					/* translators: %s: Number of terms */
					'totalEnabled'        => __( 'Total enabled: %s terms', 'intufind' ),
					'lastSynced'          => __( 'Last synced:', 'intufind' ),
					'synced'              => __( 'Synced', 'intufind' ),
					'notSynced'           => __( 'Not synced', 'intufind' ),
					'taxonomies'          => __( 'Taxonomies', 'intufind' ),
					'disconnect'          => __( 'Disconnect', 'intufind' ),
					'disconnecting'       => __( 'Disconnecting...', 'intufind' ),
					'confirmDisconnect'   => __( 'Are you sure you want to disconnect? This will permanently delete all your data from Intufind including indexed content, chat configuration, prompts, and styling. All local plugin settings will also be removed. This action cannot be undone.', 'intufind' ),
					'verifiedFromCloud'   => __( 'Verified from cloud', 'intufind' ),
					// Plan limit and error messaging.
					'planLimitTitle'      => __( 'Plan Limit Reached', 'intufind' ),
					/* translators: {current} and {limit} are placeholders replaced by JavaScript */
					'planLimitMessage'    => __( 'You have reached your plan\'s document limit ({current} of {limit}). Upgrade your plan to sync more content.', 'intufind' ),
					'upgradePlan'         => __( 'Upgrade Plan', 'intufind' ),
					'planLimitReached'    => __( 'Plan limit reached', 'intufind' ),
					'viewErrorDetails'    => __( 'View error details', 'intufind' ),
				),
			)
		);
	}

	/**
	 * Register plugin settings.
	 *
	 * @return void
	 */
	public function register_settings() {
		// Register settings.
		register_setting(
			'intufind_settings',
			INTUFIND_OPTION_API_KEY,
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			)
		);

		register_setting(
			'intufind_settings',
			INTUFIND_OPTION_WORKSPACE_ID,
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			)
		);

		register_setting(
			'intufind_sync_settings',
			INTUFIND_OPTION_SYNC_POST_TYPES,
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_post_types' ),
			)
		);

		// Search settings.
		register_setting(
			'intufind_search_settings',
			Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH,
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => false,
			)
		);

		register_setting(
			'intufind_search_settings',
			Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH,
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => false,
			)
		);

		register_setting(
			'intufind_search_settings',
			Intufind_Search_Override::OPTION_ENABLE_MIXED_SEARCH,
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			'intufind_search_settings',
			Intufind_Search_Override::OPTION_SEARCH_LIMIT,
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 20,
			)
		);

		register_setting(
			'intufind_search_settings',
			Intufind_Search_Override::OPTION_SEARCH_CACHE_DURATION,
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 3600,
			)
		);

		// Chat settings.
		register_setting(
			'intufind_chat_settings',
			Intufind_Chat_Widget::OPTION_ENABLE_CHAT,
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		// Recommendations settings (WooCommerce only).
		if ( class_exists( 'WooCommerce' ) ) {
			register_setting(
				'intufind_recommendations_settings',
				Intufind_Recommendations_Override::OPTION_ENABLE_RELATED,
				array(
					'type'              => 'boolean',
					'sanitize_callback' => 'rest_sanitize_boolean',
					'default'           => false,
				)
			);

			register_setting(
				'intufind_recommendations_settings',
				Intufind_Recommendations_Override::OPTION_ENABLE_UPSELLS,
				array(
					'type'              => 'boolean',
					'sanitize_callback' => 'rest_sanitize_boolean',
					'default'           => false,
				)
			);

			register_setting(
				'intufind_recommendations_settings',
				Intufind_Recommendations_Override::OPTION_ENABLE_CROSS_SELLS,
				array(
					'type'              => 'boolean',
					'sanitize_callback' => 'rest_sanitize_boolean',
					'default'           => false,
				)
			);

			register_setting(
				'intufind_recommendations_settings',
				Intufind_Recommendations_Override::OPTION_MAX_RECOMMENDATIONS,
				array(
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
					'default'           => 4,
				)
			);

			register_setting(
				'intufind_recommendations_settings',
				Intufind_Recommendations_Override::OPTION_CACHE_DURATION,
				array(
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
					'default'           => 3600,
				)
			);
		}

		// Auto-sync settings.
		register_setting(
			'intufind_sync_settings',
			Intufind_Sync::OPTION_SYNC_ON_CONNECT,
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			'intufind_sync_settings',
			Intufind_Sync::OPTION_AUTO_SYNC_ENABLED,
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			'intufind_sync_settings',
			Intufind_Sync::OPTION_SYNC_INTERVAL,
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => Intufind_Sync::DEFAULT_SYNC_INTERVAL,
			)
		);

		register_setting(
			'intufind_sync_settings',
			Intufind_Sync::OPTION_SYNC_HOUR,
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => Intufind_Sync::DEFAULT_SYNC_HOUR,
			)
		);
	}

	/**
	 * Sanitize post types array.
	 *
	 * @param mixed $value Input value.
	 * @return array Sanitized array of post types.
	 */
	public function sanitize_post_types( $value ) {
		if ( ! is_array( $value ) ) {
			return array();
		}
		return array_map( 'sanitize_key', $value );
	}

	/**
	 * Render the settings page.
	 *
	 * @return void
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		require_once INTUFIND_PLUGIN_DIR . 'admin/partials/settings-display.php';
	}

	/**
	 * Render the sync page.
	 *
	 * @return void
	 */
	public function render_sync_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		require_once INTUFIND_PLUGIN_DIR . 'admin/partials/sync-display.php';
	}

	/**
	 * Render the search settings page.
	 *
	 * @return void
	 */
	public function render_search_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		require_once INTUFIND_PLUGIN_DIR . 'admin/partials/search-display.php';
	}

	/**
	 * Render the chat settings page.
	 *
	 * @return void
	 */
	public function render_chat_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		require_once INTUFIND_PLUGIN_DIR . 'admin/partials/chat-display.php';
	}

	/**
	 * Render the recommendations settings page.
	 *
	 * @return void
	 */
	public function render_recommendations_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		require_once INTUFIND_PLUGIN_DIR . 'admin/partials/recommendations-display.php';
	}

	/**
	 * Render the status page.
	 *
	 * @return void
	 */
	public function render_status_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		require_once INTUFIND_PLUGIN_DIR . 'admin/partials/status-display.php';
	}

	/**
	 * AJAX handler: Validate API key.
	 *
	 * @return void
	 */
	public function ajax_validate_key() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$api_key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';

		if ( empty( $api_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key is required.', 'intufind' ) ) );
		}

		// Create temporary API instance with the new key.
		$temp_api = new Intufind_API();

		// Temporarily set the key for validation.
		update_option( INTUFIND_OPTION_API_KEY, $api_key );

		$result = $temp_api->validate_api_key( $api_key );

		if ( is_wp_error( $result ) ) {
			// Remove the key if validation failed.
			delete_option( INTUFIND_OPTION_API_KEY );
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		// Get tenant info from result.
		$tenant_id = isset( $result['data']['tenantId'] ) ? $result['data']['tenantId'] : '';
		update_option( INTUFIND_OPTION_TENANT_ID, $tenant_id );

		// Store publishable key for frontend widgets.
		$publishable_key = isset( $result['data']['publishableKey'] ) ? $result['data']['publishableKey'] : '';
		if ( ! empty( $publishable_key ) ) {
			update_option( INTUFIND_OPTION_PUBLISHABLE_KEY, $publishable_key );
		}

		// Store JWT secret for signing user tokens (enables authenticated chat).
		$jwt_secret = isset( $result['data']['jwtSecret'] ) ? $result['data']['jwtSecret'] : '';
		if ( ! empty( $jwt_secret ) ) {
			update_option( INTUFIND_OPTION_JWT_SECRET, $jwt_secret );
		}

		// Check if auto-generated workspace exists.
		$suggested_workspace_id = Intufind_Plugin::generate_workspace_id();
		$workspace_exists       = $temp_api->workspace_exists( $suggested_workspace_id );

		// Get list of workspaces for dropdown.
		$workspaces_result = $temp_api->list_workspaces();
		$workspaces        = array();
		if ( ! is_wp_error( $workspaces_result ) && isset( $workspaces_result['data'] ) ) {
			// API returns { data: [...] } where data is the array of workspaces directly.
			$workspaces = $workspaces_result['data'];
			// Handle nested format for compatibility.
			if ( isset( $workspaces['workspaces'] ) ) {
				$workspaces = $workspaces['workspaces'];
			}
		}

		wp_send_json_success(
			array(
				'tenantId'            => $tenant_id,
				'suggestedWorkspace'  => $suggested_workspace_id,
				'workspaceExists'     => $workspace_exists,
				'workspaces'          => $workspaces,
				'message'             => __( 'API key validated successfully.', 'intufind' ),
			)
		);
	}

	/**
	 * AJAX handler: Switch or create workspace.
	 *
	 * @return void
	 */
	public function ajax_switch_workspace() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$workspace_id = isset( $_POST['workspace_id'] ) ? sanitize_text_field( wp_unslash( $_POST['workspace_id'] ) ) : '';
		$action_type  = isset( $_POST['action_type'] ) ? sanitize_text_field( wp_unslash( $_POST['action_type'] ) ) : 'use';

		if ( empty( $workspace_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Workspace ID is required.', 'intufind' ) ) );
		}

		if ( 'create' === $action_type ) {
			// Create new workspace.
			$site_name = get_bloginfo( 'name' );
			$result    = $this->api->create_workspace( $workspace_id, $site_name );

			if ( is_wp_error( $result ) ) {
				// If workspace already exists (409), proceed to use it instead of failing.
				$error_data = $result->get_error_data();
				if ( isset( $error_data['status'] ) && 409 === $error_data['status'] ) {
					// Workspace exists - proceed to connect to it.
					$action_type = 'use';
				} else {
					wp_send_json_error( array( 'message' => $result->get_error_message() ) );
				}
			}
		}

		// Check if this is a first-time connection.
		$was_connected = get_option( INTUFIND_OPTION_CONNECTED, false );

		// Save workspace ID.
		update_option( INTUFIND_OPTION_WORKSPACE_ID, $workspace_id );
		update_option( INTUFIND_OPTION_CONNECTED, true );

		// Clear verification cache since we're now connected.
		Intufind_API::clear_verification_cache();

		// Enable features by default on first connection.
		if ( ! $was_connected || 'create' === $action_type ) {
			// Enable WordPress search override.
			if ( false === get_option( Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH ) ) {
				update_option( Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH, true );
			}

			// Enable WooCommerce search override if WooCommerce is active.
			if ( class_exists( 'WooCommerce' ) && false === get_option( Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH ) ) {
				update_option( Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH, true );
			}

			// Enable chat widget.
			if ( false === get_option( Intufind_Chat_Widget::OPTION_ENABLE_CHAT ) ) {
				update_option( Intufind_Chat_Widget::OPTION_ENABLE_CHAT, true );
			}
		}

		// If WooCommerce is active, register the orders MCP domain.
		$mcp_registered = false;
		$mcp_error      = null;
		if ( class_exists( 'WooCommerce' ) ) {
			$mcp_result = $this->api->register_orders_domain();
			if ( is_wp_error( $mcp_result ) ) {
				$mcp_error = $mcp_result->get_error_message();
				update_option( INTUFIND_OPTION_MCP_ERROR, $mcp_error );
			} else {
				$mcp_registered = true;
				update_option( INTUFIND_OPTION_MCP_REGISTERED, true );
				delete_option( INTUFIND_OPTION_MCP_ERROR );
			}
		}

		// Check if we should trigger initial sync.
		$sync_on_connect = isset( $_POST['sync_on_connect'] ) ? rest_sanitize_boolean( wp_unslash( $_POST['sync_on_connect'] ) ) : false;
		$initial_sync    = null;

		// Save the preference.
		update_option( Intufind_Sync::OPTION_SYNC_ON_CONNECT, $sync_on_connect );

		if ( $sync_on_connect && $this->sync ) {
			$initial_sync = $this->sync->trigger_initial_sync();
		}

		$response = array(
			'workspaceId'    => $workspace_id,
			'mcpRegistered'  => $mcp_registered,
			'woocommerce'    => class_exists( 'WooCommerce' ),
			'message'        => __( 'Workspace connected successfully.', 'intufind' ),
		);

		// Include MCP error if registration failed.
		if ( $mcp_error ) {
			$response['mcpError'] = $mcp_error;
		}

		if ( null !== $initial_sync ) {
			$response['initialSync'] = $initial_sync;
			$response['message']     = sprintf(
				/* translators: %d: Number of documents synced */
				__( 'Workspace connected! Initial sync complete: %d documents synced.', 'intufind' ),
				$initial_sync['synced']
			);
		}

		wp_send_json_success( $response );
	}

	/**
	 * AJAX handler: Disconnect from cloud.
	 *
	 * Performs a complete cleanup:
	 * 1. Notifies the cloud to delete all indexed data (deprovision)
	 * 2. Clears all plugin options
	 * 3. Deletes dynamic options (sync timestamps)
	 * 4. Deletes all transients
	 * 5. Clears scheduled cron jobs
	 * 6. Deletes all post meta
	 * 7. Deletes user meta (dismissed notices)
	 *
	 * @return void
	 */
	public function ajax_disconnect() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		// 1. Notify cloud to delete all data (best effort - don't fail if this errors).
		$api = new Intufind_API();
		$api->deprovision();

		// 2. Clear all plugin options.
		// NOTE: Keep this list in sync with uninstall.php (which can't use constants).
		$options = array(
			// Core connection options.
			INTUFIND_OPTION_API_KEY,
			INTUFIND_OPTION_JWT_SECRET,
			INTUFIND_OPTION_PUBLISHABLE_KEY,
			INTUFIND_OPTION_WORKSPACE_ID,
			INTUFIND_OPTION_TENANT_ID,
			INTUFIND_OPTION_CONNECTED,
			INTUFIND_OPTION_MCP_REGISTERED,
			INTUFIND_OPTION_MCP_ERROR,
			INTUFIND_OPTION_SYNC_POST_TYPES,
			Intufind_Exclusions::OPTION_SYNC_TAXONOMIES,
			'intufind_version',
			// Sync settings.
			Intufind_Sync::OPTION_AUTO_SYNC_ENABLED,
			Intufind_Sync::OPTION_SYNC_INTERVAL,
			Intufind_Sync::OPTION_SYNC_HOUR,
			Intufind_Sync::OPTION_SYNC_ON_CONNECT,
			Intufind_Sync::OPTION_SYNC_SCHEDULE_VER,
			// Search settings.
			Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH,
			Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH,
			Intufind_Search_Override::OPTION_ENABLE_MIXED_SEARCH,
			Intufind_Search_Override::OPTION_SEARCH_LIMIT,
			Intufind_Search_Override::OPTION_SEARCH_CACHE_DURATION,
			// Chat settings.
			Intufind_Chat_Widget::OPTION_ENABLE_CHAT,
			// Recommendations settings.
			Intufind_Recommendations_Override::OPTION_ENABLE_RELATED,
			Intufind_Recommendations_Override::OPTION_ENABLE_UPSELLS,
			Intufind_Recommendations_Override::OPTION_ENABLE_CROSS_SELLS,
			Intufind_Recommendations_Override::OPTION_MAX_RECOMMENDATIONS,
			Intufind_Recommendations_Override::OPTION_CACHE_DURATION,
		);

		foreach ( $options as $option ) {
			delete_option( $option );
		}

		// 3. Delete dynamic options (sync timestamps for post types and taxonomies).
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
				'intufind_last_%_sync',
				'intufind_taxonomy_sync_%'
			)
		);

		// 4. Delete all transients.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
				'%_transient_intufind_%',
				'%_transient_timeout_intufind_%'
			)
		);

		// 5. Clear scheduled cron jobs.
		wp_clear_scheduled_hook( 'intufind_scheduled_sync' );
		wp_clear_scheduled_hook( 'intufind_process_sync_batch' );
		wp_clear_scheduled_hook( 'intufind_process_sync_batch_taxonomy' );

		// 6. Delete all post meta.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
				'_intufind_%'
			)
		);

		// 7. Delete user meta (dismissed notices for all users).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE %s",
				'intufind_%'
			)
		);

		wp_send_json_success(
			array(
				'message' => __( 'Disconnected successfully. All data has been removed.', 'intufind' ),
			)
		);
	}

	/**
	 * Verify connection status on settings page load.
	 *
	 * Returns the full verification result including any error message
	 * for display on the settings page. The connected flag is automatically
	 * cleared if verification fails.
	 *
	 * @return array{valid: bool, error?: string}|null Verification result or null if not connected.
	 */
	public function verify_connection_on_load() {
		// Quick check - if not connected, nothing to verify.
		if ( ! get_option( INTUFIND_OPTION_CONNECTED, false ) ) {
			return null;
		}

		// Verify and get the full result (is_connected() handles flag clearing).
		$result = $this->api->verify_connection();

		// Ensure flag is cleared on failure (is_connected(true) also does this,
		// but we call verify_connection() directly here to get the full result).
		if ( ! $result['valid'] ) {
			delete_option( INTUFIND_OPTION_CONNECTED );
		}

		return $result;
	}

	/**
	 * Verify connection on Intufind admin pages.
	 *
	 * Called on admin_init to ensure connection validity is checked
	 * on all plugin pages, not just the Settings page. Uses cached
	 * verification to avoid excessive API calls.
	 *
	 * @return void
	 */
	public function verify_connection_on_plugin_pages() {
		// Only run on Intufind admin pages.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking page parameter.
		$page = isset( $_GET['page'] ) ? sanitize_key( $_GET['page'] ) : '';
		if ( strpos( $page, 'intufind' ) !== 0 ) {
			return;
		}

		// Check and verify connection (uses caching).
		$this->is_connected( true );
	}

	/**
	 * AJAX handler: Manual sync.
	 *
	 * @return void
	 */
	public function ajax_manual_sync() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$post_type = isset( $_POST['post_type'] ) ? sanitize_key( wp_unslash( $_POST['post_type'] ) ) : '';

		if ( empty( $post_type ) ) {
			wp_send_json_error( array( 'message' => __( 'Post type is required.', 'intufind' ) ) );
		}

		// Run sync using the sync manager.
		if ( ! $this->sync ) {
			wp_send_json_error( array( 'message' => __( 'Sync manager not available.', 'intufind' ) ) );
		}

		$results = $this->sync->manual_sync( $post_type );

		// Clear cloud stats cache so next page load shows fresh counts.
		Intufind_API::clear_cloud_stats_cache();

		$post_type_obj = get_post_type_object( $post_type );
		$label         = $post_type_obj ? $post_type_obj->labels->name : $post_type;

		// Build response with error details if present.
		$response = array(
			'message' => sprintf(
				/* translators: %1$s: Post type name, %2$d: Synced count, %3$d: Error count */
				__( '%1$s sync complete: %2$d synced, %3$d errors, %4$d skipped, %5$d cleaned up.', 'intufind' ),
				$label,
				$results['synced'],
				$results['errors'],
				$results['skipped'],
				$results['deleted']
			),
			'results' => $results,
			'stats'   => $this->sync->get_sync_stats( $post_type ),
		);

		// Include error details for user feedback.
		if ( ! empty( $results['error_messages'] ) ) {
			$response['error_messages'] = $results['error_messages'];
			$response['error_info']     = $this->parse_sync_error_info( $results['error_messages'] );
		}

		wp_send_json_success( $response );
	}

	/**
	 * Parse sync error messages to extract actionable info.
	 *
	 * Detects plan limit errors and provides upgrade information.
	 *
	 * @param array $error_messages Array of error message strings.
	 * @return array Parsed error info with type and details.
	 */
	private function parse_sync_error_info( $error_messages ) {
		$info = array(
			'type'        => 'generic',
			'is_limit'    => false,
			'upgrade_url' => '',
		);

		foreach ( $error_messages as $message ) {
			// Check for indexing limit exceeded errors.
			// Pattern: "post indexing limit exceeded. Current: X, Limit: Y, New documents: Z".
			if ( preg_match( '/indexing limit exceeded.*Current:\s*(\d+).*Limit:\s*(\d+)/i', $message, $matches ) ) {
				$info['type']          = 'limit_exceeded';
				$info['is_limit']      = true;
				$info['current_count'] = (int) $matches[1];
				$info['plan_limit']    = (int) $matches[2];

				// Build upgrade URL.
				$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
				if ( ! empty( $workspace_id ) ) {
					$info['upgrade_url'] = 'https://intufind.com/dashboard/workspaces/' . urlencode( $workspace_id ) . '/subscription';
				} else {
					$info['upgrade_url'] = 'https://intufind.com/dashboard/subscription';
				}

				break; // Found a limit error, no need to check further.
			}
		}

		return $info;
	}

	/**
	 * AJAX handler: Sync taxonomies.
	 *
	 * @return void
	 */
	public function ajax_sync_taxonomies() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		// Run taxonomy sync using the sync manager.
		if ( ! $this->sync ) {
			wp_send_json_error( array( 'message' => __( 'Sync manager not available.', 'intufind' ) ) );
		}

		$results = $this->sync->manual_sync_taxonomies();

		// Clear cloud stats cache so next page load shows fresh counts.
		Intufind_API::clear_cloud_stats_cache();

		// Get per-taxonomy sync status for UI update.
		$exclusions          = $this->sync->get_exclusions();
		$taxonomy_info       = $exclusions->get_taxonomy_info();
		$taxonomy_status_map = array();

		foreach ( $taxonomy_info as $taxonomy => $info ) {
			if ( $info['enabled'] ) {
				$taxonomy_status_map[ $taxonomy ] = $this->sync->get_taxonomy_sync_status( $taxonomy );
			}
		}

		// Build response with error details if present.
		$response = array(
			'message'         => sprintf(
				/* translators: %1$d: Synced count, %2$d: Error count, %3$d: Deleted count */
				__( 'Taxonomy sync complete: %1$d synced, %2$d errors, %3$d cleaned up.', 'intufind' ),
				$results['synced'],
				$results['errors'],
				$results['deleted']
			),
			'results'         => $results,
			'stats'           => $this->sync->get_taxonomy_sync_stats(),
			'taxonomy_status' => $taxonomy_status_map,
		);

		// Include error details for user feedback.
		if ( ! empty( $results['error_messages'] ) ) {
			$response['error_messages'] = $results['error_messages'];
			$response['error_info']     = $this->parse_sync_error_info( $results['error_messages'] );
		}

		wp_send_json_success( $response );
	}

	/**
	 * AJAX handler: Get sync stats.
	 *
	 * @return void
	 */
	public function ajax_get_sync_stats() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$post_type = isset( $_POST['post_type'] ) ? sanitize_key( wp_unslash( $_POST['post_type'] ) ) : '';

		if ( ! $this->sync ) {
			wp_send_json_error( array( 'message' => __( 'Sync manager not available.', 'intufind' ) ) );
		}

		if ( empty( $post_type ) ) {
			// Return stats for all enabled types.
			$enabled = $this->sync->get_exclusions()->get_enabled_post_types();
			$all_stats = array();
			foreach ( $enabled as $type ) {
				$all_stats[ $type ] = $this->sync->get_sync_stats( $type );
			}
			wp_send_json_success( array( 'stats' => $all_stats ) );
		} else {
			wp_send_json_success( array( 'stats' => $this->sync->get_sync_stats( $post_type ) ) );
		}
	}

	/**
	 * AJAX handler: Get cloud document stats.
	 *
	 * Fetches actual document counts from the cloud to verify sync status.
	 *
	 * @return void
	 */
	public function ajax_get_cloud_stats() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$result = $this->api->get_document_stats();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( $result['data'] );
	}

	/**
	 * AJAX handler: Retry failed syncs.
	 *
	 * @return void
	 */
	public function ajax_retry_failed() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$post_type = isset( $_POST['post_type'] ) ? sanitize_key( wp_unslash( $_POST['post_type'] ) ) : '';

		if ( empty( $post_type ) ) {
			wp_send_json_error( array( 'message' => __( 'Post type is required.', 'intufind' ) ) );
		}

		if ( ! $this->sync ) {
			wp_send_json_error( array( 'message' => __( 'Sync manager not available.', 'intufind' ) ) );
		}

		$results = $this->sync->retry_failed_syncs( $post_type );

		// Clear cloud stats cache so next page load shows fresh counts.
		Intufind_API::clear_cloud_stats_cache();

		wp_send_json_success(
			array(
				'message' => sprintf(
					/* translators: %1$d: Synced count, %2$d: Error count */
					__( 'Retry complete: %1$d synced, %2$d still failing.', 'intufind' ),
					$results['synced'],
					$results['errors']
				),
				'results' => $results,
				'stats'   => $this->sync->get_sync_stats( $post_type ),
			)
		);
	}

	/**
	 * AJAX handler: Save post types.
	 *
	 * @return void
	 */
	public function ajax_save_post_types() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$post_types = isset( $_POST['post_types'] ) ? array_map( 'sanitize_key', (array) $_POST['post_types'] ) : array();

		update_option( INTUFIND_OPTION_SYNC_POST_TYPES, $post_types );

		wp_send_json_success(
			array(
				'message' => __( 'Post types saved.', 'intufind' ),
			)
		);
	}

	/**
	 * AJAX handler: Save enabled taxonomies.
	 *
	 * @return void
	 */
	public function ajax_save_taxonomies() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$taxonomies = isset( $_POST['taxonomies'] ) ? array_map( 'sanitize_key', (array) $_POST['taxonomies'] ) : array();

		$exclusions = new Intufind_Exclusions();
		$exclusions->save_enabled_taxonomies( $taxonomies );

		wp_send_json_success(
			array(
				'message' => __( 'Taxonomies saved.', 'intufind' ),
			)
		);
	}

	/**
	 * AJAX handler: Save search option.
	 *
	 * @return void
	 */
	public function ajax_save_search_option() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$option_name  = isset( $_POST['option_name'] ) ? sanitize_key( wp_unslash( $_POST['option_name'] ) ) : '';
		$option_value = isset( $_POST['option_value'] ) ? wp_unslash( $_POST['option_value'] ) : '';

		// Validate option name is one of ours.
		$allowed_options = array(
			Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH,
			Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH,
			Intufind_Search_Override::OPTION_ENABLE_MIXED_SEARCH,
			Intufind_Search_Override::OPTION_SEARCH_LIMIT,
			Intufind_Search_Override::OPTION_SEARCH_CACHE_DURATION,
		);

		if ( ! in_array( $option_name, $allowed_options, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid option.', 'intufind' ) ) );
		}

		// Sanitize based on option type.
		if ( in_array( $option_name, array(
			Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH,
			Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH,
			Intufind_Search_Override::OPTION_ENABLE_MIXED_SEARCH,
		), true ) ) {
			$option_value = rest_sanitize_boolean( $option_value );
		} else {
			$option_value = absint( $option_value );
		}

		update_option( $option_name, $option_value );

		wp_send_json_success(
			array(
				'message' => __( 'Setting saved.', 'intufind' ),
			)
		);
	}

	/**
	 * AJAX handler: Save recommendations option.
	 *
	 * @return void
	 */
	public function ajax_save_recommendations_option() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$option_name  = isset( $_POST['option_name'] ) ? sanitize_key( wp_unslash( $_POST['option_name'] ) ) : '';
		$option_value = isset( $_POST['option_value'] ) ? wp_unslash( $_POST['option_value'] ) : '';

		// Validate option name is one of ours.
		$allowed_options = array(
			Intufind_Recommendations_Override::OPTION_ENABLE_RELATED,
			Intufind_Recommendations_Override::OPTION_ENABLE_UPSELLS,
			Intufind_Recommendations_Override::OPTION_ENABLE_CROSS_SELLS,
			Intufind_Recommendations_Override::OPTION_MAX_RECOMMENDATIONS,
			Intufind_Recommendations_Override::OPTION_CACHE_DURATION,
		);

		if ( ! in_array( $option_name, $allowed_options, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid option.', 'intufind' ) ) );
		}

		// Sanitize based on option type.
		if ( in_array( $option_name, array(
			Intufind_Recommendations_Override::OPTION_ENABLE_RELATED,
			Intufind_Recommendations_Override::OPTION_ENABLE_UPSELLS,
			Intufind_Recommendations_Override::OPTION_ENABLE_CROSS_SELLS,
		), true ) ) {
			$option_value = rest_sanitize_boolean( $option_value );
		} else {
			$option_value = absint( $option_value );
		}

		update_option( $option_name, $option_value );

		wp_send_json_success(
			array(
				'message' => __( 'Setting saved.', 'intufind' ),
			)
		);
	}

	/**
	 * AJAX handler: Save chat option.
	 *
	 * @return void
	 */
	public function ajax_save_chat_option() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$option_name  = isset( $_POST['option_name'] ) ? sanitize_key( wp_unslash( $_POST['option_name'] ) ) : '';
		$option_value = isset( $_POST['option_value'] ) ? wp_unslash( $_POST['option_value'] ) : '';

		// Validate option name is one of ours.
		$allowed_options = array(
			Intufind_Chat_Widget::OPTION_ENABLE_CHAT,
		);

		if ( ! in_array( $option_name, $allowed_options, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid option.', 'intufind' ) ) );
		}

		// Sanitize as boolean.
		$option_value = rest_sanitize_boolean( $option_value );

		update_option( $option_name, $option_value );

		wp_send_json_success(
			array(
				'message' => __( 'Setting saved.', 'intufind' ),
			)
		);
	}

	/**
	 * AJAX handler: Save sync option.
	 *
	 * @return void
	 */
	public function ajax_save_sync_option() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$option_name  = isset( $_POST['option_name'] ) ? sanitize_key( wp_unslash( $_POST['option_name'] ) ) : '';
		$option_value = isset( $_POST['option_value'] ) ? wp_unslash( $_POST['option_value'] ) : '';

		// Validate option name is one of ours.
		$allowed_options = array(
			Intufind_Sync::OPTION_AUTO_SYNC_ENABLED,
			Intufind_Sync::OPTION_SYNC_INTERVAL,
			Intufind_Sync::OPTION_SYNC_HOUR,
		);

		if ( ! in_array( $option_name, $allowed_options, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid option.', 'intufind' ) ) );
		}

		// Sanitize based on option type.
		if ( Intufind_Sync::OPTION_AUTO_SYNC_ENABLED === $option_name ) {
			$option_value = rest_sanitize_boolean( $option_value );
		} else {
			$option_value = absint( $option_value );
		}

		update_option( $option_name, $option_value );

		// Reschedule sync when settings change.
		if ( $this->sync ) {
			$this->sync->reschedule_sync();
		}

		// Get next scheduled time for response.
		$next_sync = wp_next_scheduled( Intufind_Sync::SCHEDULED_SYNC_HOOK );

		wp_send_json_success(
			array(
				'message'   => __( 'Setting saved.', 'intufind' ),
				'next_sync' => $next_sync ? wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $next_sync ) : null,
			)
		);
	}

	/**
	 * Check if plugin is connected.
	 *
	 * @param bool $verify Whether to verify the connection with the API (uses caching).
	 * @return bool
	 */
	public function is_connected( $verify = false ) {
		$is_connected = (bool) get_option( INTUFIND_OPTION_CONNECTED, false );

		if ( ! $is_connected ) {
			return false;
		}

		// Optionally verify connection with the API (cached for 10 min).
		if ( $verify ) {
			$result = $this->api->verify_connection();
			if ( ! $result['valid'] ) {
				// Clear the connected flag since connection is invalid.
				delete_option( INTUFIND_OPTION_CONNECTED );
				return false;
			}
		}

		return true;
	}

	/**
	 * Get current workspace ID.
	 *
	 * @return string|null
	 */
	public function get_workspace_id() {
		$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		return ! empty( $workspace_id ) ? $workspace_id : null;
	}

	/**
	 * Handle plugin activation.
	 *
	 * When WooCommerce is activated on an already-connected site,
	 * register the orders MCP domain automatically.
	 *
	 * @param string $plugin Path to the plugin file relative to plugins directory.
	 * @return void
	 */
	public function on_plugin_activated( $plugin ) {
		// Check if WooCommerce was just activated.
		if ( 'woocommerce/woocommerce.php' !== $plugin ) {
			return;
		}

		// Only proceed if we're connected.
		if ( ! $this->is_connected() ) {
			return;
		}

		// Register the orders MCP domain.
		$result = $this->api->register_orders_domain();
		if ( is_wp_error( $result ) ) {
			update_option( INTUFIND_OPTION_MCP_ERROR, $result->get_error_message() );
		} else {
			update_option( INTUFIND_OPTION_MCP_REGISTERED, true );
			delete_option( INTUFIND_OPTION_MCP_ERROR );
		}
	}

	/**
	 * Handle plugin deactivation.
	 *
	 * When WooCommerce is deactivated, unregister the orders MCP domain.
	 *
	 * @param string $plugin Path to the plugin file relative to plugins directory.
	 * @return void
	 */
	public function on_plugin_deactivated( $plugin ) {
		// Check if WooCommerce was just deactivated.
		if ( 'woocommerce/woocommerce.php' !== $plugin ) {
			return;
		}

		// Only proceed if we're connected and had MCP registered.
		if ( ! $this->is_connected() || ! get_option( INTUFIND_OPTION_MCP_REGISTERED, false ) ) {
			return;
		}

		// Unregister the orders MCP domain.
		$result = $this->api->unregister_orders_domain();
		if ( ! is_wp_error( $result ) ) {
			delete_option( INTUFIND_OPTION_MCP_REGISTERED );
			delete_option( INTUFIND_OPTION_MCP_ERROR );
		}
	}

	/**
	 * Display admin notice for initial sync completion.
	 *
	 * @return void
	 */
	public function display_initial_sync_notice() {
		$sync_status = Intufind_Sync::get_initial_sync_status();

		if ( ! $sync_status ) {
			return;
		}

		// Clear the notice after displaying.
		Intufind_Sync::clear_initial_sync_status();

		$sync_url = admin_url( 'admin.php?page=intufind-sync' );
		$message  = sprintf(
			/* translators: 1: Number synced, 2: Number of errors, 3: Link to sync page */
			__( 'Initial content sync complete! %1$d documents synced, %2$d errors. %3$s', 'intufind' ),
			$sync_status['synced'],
			$sync_status['errors'],
			'<a href="' . esc_url( $sync_url ) . '">' . __( 'View sync status →', 'intufind' ) . '</a>'
		);

		$class = $sync_status['errors'] > 0 ? 'notice-warning' : 'notice-success';
		?>
		<div class="notice <?php echo esc_attr( $class ); ?> is-dismissible">
			<p><strong><?php echo esc_html__( 'Intufind', 'intufind' ); ?>:</strong> <?php echo wp_kses_post( $message ); ?></p>
		</div>
		<?php
	}

	/**
	 * AJAX handler: Retry MCP domain registration.
	 *
	 * Allows users to retry registering the orders domain after fixing issues
	 * like missing HTTPS/SSL certificate.
	 *
	 * @return void
	 */
	public function ajax_retry_mcp_registration() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		// Check if connected.
		if ( ! $this->is_connected() ) {
			wp_send_json_error( array( 'message' => __( 'Not connected. Please connect your account first.', 'intufind' ) ) );
		}

		// Check if WooCommerce is active.
		if ( ! class_exists( 'WooCommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'WooCommerce is not active. Please activate WooCommerce first.', 'intufind' ) ) );
		}

		// Attempt to register the orders domain.
		$result = $this->api->register_orders_domain();

		if ( is_wp_error( $result ) ) {
			$error_message = $result->get_error_message();
			update_option( INTUFIND_OPTION_MCP_ERROR, $error_message );
			wp_send_json_error(
				array(
					'message' => sprintf(
						/* translators: %s: Error message */
						__( 'Registration failed: %s', 'intufind' ),
						$error_message
					),
				)
			);
		}

		// Success - update options.
		update_option( INTUFIND_OPTION_MCP_REGISTERED, true );
		delete_option( INTUFIND_OPTION_MCP_ERROR );

		wp_send_json_success(
			array(
				'message' => __( 'Orders domain registered successfully! Your AI assistant can now help customers with order inquiries.', 'intufind' ),
			)
		);
	}

	/**
	 * Handle AJAX request to dismiss an intro notice.
	 *
	 * @return void
	 */
	public function ajax_dismiss_notice() {
		check_ajax_referer( 'intufind_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		$notice_id = isset( $_POST['notice_id'] ) ? sanitize_key( $_POST['notice_id'] ) : '';

		if ( empty( $notice_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid notice ID.', 'intufind' ) ) );
		}

		$user_id           = get_current_user_id();
		$dismissed_notices = get_user_meta( $user_id, 'intufind_dismissed_notices', true );

		if ( ! is_array( $dismissed_notices ) ) {
			$dismissed_notices = array();
		}

		if ( ! in_array( $notice_id, $dismissed_notices, true ) ) {
			$dismissed_notices[] = $notice_id;
			update_user_meta( $user_id, 'intufind_dismissed_notices', $dismissed_notices );
		}

		wp_send_json_success();
	}
}
