<?php
/**
 * Reusable UI components.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * UI Components class.
 *
 * Provides static methods for rendering consistent, reusable UI elements
 * throughout the admin interface. All output is properly escaped.
 */
class Intufind_Components {

	/**
	 * Render a button.
	 *
	 * @param string $text    Button text.
	 * @param array  $args    {
	 *     Optional. Button arguments.
	 *     @type string $variant  Button variant: primary, secondary, danger, ghost. Default 'primary'.
	 *     @type string $type     Button type: button, submit. Default 'button'.
	 *     @type string $name     Button name attribute.
	 *     @type string $id       Button ID attribute.
	 *     @type string $class    Additional CSS classes.
	 *     @type bool   $disabled Whether button is disabled. Default false.
	 *     @type string $icon     Dashicon name (without dashicons- prefix).
	 * }
	 * @return void
	 */
	public static function button( $text, $args = array() ) {
		$defaults = array(
			'variant'  => 'primary',
			'type'     => 'button',
			'name'     => '',
			'id'       => '',
			'class'    => '',
			'size'     => '',
			'disabled' => false,
			'icon'     => '',
			'data'     => array(),
		);
		$args     = wp_parse_args( $args, $defaults );

		$classes = array( 'intufind-button', 'intufind-button--' . $args['variant'] );
		if ( ! empty( $args['size'] ) ) {
			$classes[] = 'intufind-button--' . $args['size'];
		}
		if ( ! empty( $args['class'] ) ) {
			$classes[] = $args['class'];
		}

		$attrs = array(
			'type'  => $args['type'],
			'class' => implode( ' ', $classes ),
		);

		if ( ! empty( $args['name'] ) ) {
			$attrs['name'] = $args['name'];
		}
		if ( ! empty( $args['id'] ) ) {
			$attrs['id'] = $args['id'];
		}
		if ( $args['disabled'] ) {
			$attrs['disabled'] = 'disabled';
		}

		// Add data attributes.
		if ( ! empty( $args['data'] ) && is_array( $args['data'] ) ) {
			foreach ( $args['data'] as $key => $value ) {
				$attrs[ 'data-' . $key ] = $value;
			}
		}

		$attr_string = self::build_attrs( $attrs );

		echo '<button ' . $attr_string . '>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		if ( ! empty( $args['icon'] ) ) {
			echo '<span class="dashicons dashicons-' . esc_attr( $args['icon'] ) . '"></span> ';
		}
		echo esc_html( $text );
		echo '</button>';
	}

	/**
	 * Render a link styled as a button.
	 *
	 * @param string $text Button text.
	 * @param string $url  Link URL.
	 * @param array  $args Same as button() plus 'target' for link target.
	 * @return void
	 */
	public static function link_button( $text, $url, $args = array() ) {
		$defaults = array(
			'variant' => 'secondary',
			'class'   => '',
			'target'  => '',
			'icon'    => '',
		);
		$args     = wp_parse_args( $args, $defaults );

		$classes = array( 'intufind-button', 'intufind-button--' . $args['variant'] );
		if ( ! empty( $args['class'] ) ) {
			$classes[] = $args['class'];
		}

		$attrs = array(
			'href'  => esc_url( $url ),
			'class' => implode( ' ', $classes ),
		);

		if ( ! empty( $args['target'] ) ) {
			$attrs['target'] = $args['target'];
			if ( '_blank' === $args['target'] ) {
				$attrs['rel'] = 'noopener noreferrer';
			}
		}

		$attr_string = self::build_attrs( $attrs );

		echo '<a ' . $attr_string . '>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		if ( ! empty( $args['icon'] ) ) {
			echo '<span class="dashicons dashicons-' . esc_attr( $args['icon'] ) . '"></span> ';
		}
		echo esc_html( $text );
		echo '</a>';
	}

	/**
	 * Render a text input field.
	 *
	 * @param string $name  Input name attribute.
	 * @param string $value Input value.
	 * @param array  $args  {
	 *     Optional. Input arguments.
	 *     @type string   $type                 Input type. Default 'text'.
	 *     @type string   $id                   Input ID. Defaults to name.
	 *     @type string   $label                Label text.
	 *     @type string   $description          Help text below input. Can include HTML if $description_html is true.
	 *     @type bool     $description_html     Whether description contains HTML. Default false.
	 *     @type string   $placeholder          Placeholder text.
	 *     @type string   $class                Additional CSS classes.
	 *     @type bool     $required             Whether field is required.
	 *     @type bool     $readonly             Whether field is readonly.
	 *     @type bool     $disabled             Whether field is disabled.
	 *     @type bool     $show_password_toggle Whether to show password visibility toggle. Default false.
	 *     @type callable $suffix               Callback to render suffix element (e.g., button).
	 *     @type array    $data                 Data attributes (key => value, without 'data-' prefix).
	 * }
	 * @return void
	 */
	public static function text_input( $name, $value, $args = array() ) {
		$defaults = array(
			'type'                 => 'text',
			'id'                   => $name,
			'label'                => '',
			'description'          => '',
			'description_html'     => false,
			'placeholder'          => '',
			'class'                => '',
			'required'             => false,
			'readonly'             => false,
			'disabled'             => false,
			'show_password_toggle' => false,
			'suffix'               => null,
			'data'                 => array(),
		);
		$args     = wp_parse_args( $args, $defaults );

		$wrapper_classes = array( 'intufind-field' );
		$input_classes   = array( 'intufind-input' );
		if ( ! empty( $args['class'] ) ) {
			$input_classes[] = $args['class'];
		}

		$has_suffix          = is_callable( $args['suffix'] );
		$has_password_toggle = $args['show_password_toggle'] && 'password' === $args['type'];

		echo '<div class="' . esc_attr( implode( ' ', $wrapper_classes ) ) . '">';

		if ( ! empty( $args['label'] ) ) {
			echo '<label class="intufind-label" for="' . esc_attr( $args['id'] ) . '">';
			echo esc_html( $args['label'] );
			if ( $args['required'] ) {
				echo ' <span class="intufind-required">*</span>';
			}
			echo '</label>';
		}

		// Wrap in input-group if there's a suffix or password toggle.
		if ( $has_suffix || $has_password_toggle ) {
			echo '<div class="intufind-input-group">';
		}

		// Wrap input in password wrapper if toggle is enabled.
		if ( $has_password_toggle ) {
			echo '<div class="intufind-password-wrapper">';
		}

		$input_attrs = array(
			'type'  => $args['type'],
			'id'    => $args['id'],
			'name'  => $name,
			'value' => $value,
			'class' => implode( ' ', $input_classes ),
		);

		if ( ! empty( $args['placeholder'] ) ) {
			$input_attrs['placeholder'] = $args['placeholder'];
		}
		if ( $args['required'] ) {
			$input_attrs['required'] = 'required';
		}
		if ( $args['readonly'] ) {
			$input_attrs['readonly'] = 'readonly';
		}
		if ( $args['disabled'] ) {
			$input_attrs['disabled'] = 'disabled';
		}

		// Add data attributes.
		foreach ( $args['data'] as $key => $data_value ) {
			$input_attrs[ 'data-' . $key ] = $data_value;
		}

		echo '<input ' . self::build_attrs( $input_attrs ) . ' />'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

		// Render password toggle button.
		if ( $has_password_toggle ) {
			echo '<button type="button" class="intufind-password-toggle" aria-label="' . esc_attr__( 'Toggle password visibility', 'intufind' ) . '" data-target="' . esc_attr( $args['id'] ) . '">';
			echo '<span class="dashicons dashicons-visibility" aria-hidden="true"></span>';
			echo '</button>';
			echo '</div>'; // Close password-wrapper.
		}

		// Render suffix if provided.
		if ( $has_suffix ) {
			call_user_func( $args['suffix'] );
		}

		// Close input-group if we opened it.
		if ( $has_suffix || $has_password_toggle ) {
			echo '</div>'; // Close input-group.
		}

		if ( ! empty( $args['description'] ) ) {
			echo '<p class="intufind-description">';
			if ( $args['description_html'] ) {
				echo wp_kses_post( $args['description'] );
			} else {
				echo esc_html( $args['description'] );
			}
			echo '</p>';
		}

		echo '</div>';
	}

	/**
	 * Render a select dropdown (legacy signature).
	 *
	 * @param string $name    Select name attribute.
	 * @param array  $options Array of value => label options.
	 * @param string $value   Currently selected value.
	 * @param array  $args    Optional. Select arguments.
	 * @return void
	 */
	public static function select( $name, $options, $value, $args = array() ) {
		$args['options'] = $options;
		self::select_input( $name, $value, $args );
	}

	/**
	 * Render a select dropdown.
	 *
	 * @param string $name  Select name attribute.
	 * @param string $value Currently selected value.
	 * @param array  $args  {
	 *     Optional. Select arguments.
	 *     @type array  $options     Array of value => label options.
	 *     @type string $id          Select ID. Defaults to name.
	 *     @type string $label       Label text.
	 *     @type string $description Help text below select.
	 *     @type string $class       Additional CSS classes.
	 *     @type string $placeholder Placeholder option text.
	 * }
	 * @return void
	 */
	public static function select_input( $name, $value, $args = array() ) {
		$defaults = array(
			'options'     => array(),
			'id'          => $name,
			'label'       => '',
			'description' => '',
			'class'       => '',
			'placeholder' => '',
		);
		$args     = wp_parse_args( $args, $defaults );

		$select_classes = array( 'intufind-select' );
		if ( ! empty( $args['class'] ) ) {
			$select_classes[] = $args['class'];
		}

		echo '<div class="intufind-field">';

		if ( ! empty( $args['label'] ) ) {
			echo '<label class="intufind-label" for="' . esc_attr( $args['id'] ) . '">';
			echo esc_html( $args['label'] );
			echo '</label>';
		}

		echo '<select id="' . esc_attr( $args['id'] ) . '" name="' . esc_attr( $name ) . '" class="' . esc_attr( implode( ' ', $select_classes ) ) . '">';

		if ( ! empty( $args['placeholder'] ) ) {
			echo '<option value="">' . esc_html( $args['placeholder'] ) . '</option>';
		}

		foreach ( $args['options'] as $option_value => $option_label ) {
			$selected = selected( $value, $option_value, false );
			echo '<option value="' . esc_attr( $option_value ) . '"' . $selected . '>' . esc_html( $option_label ) . '</option>';
		}

		echo '</select>';

		if ( ! empty( $args['description'] ) ) {
			echo '<p class="intufind-description">' . esc_html( $args['description'] ) . '</p>';
		}

		echo '</div>';
	}

	/**
	 * Render a number input field.
	 *
	 * @param string $name  Input name attribute.
	 * @param mixed  $value Input value.
	 * @param array  $args  {
	 *     Optional. Input arguments.
	 *     @type string $id          Input ID. Defaults to name.
	 *     @type string $label       Label text.
	 *     @type string $description Help text below input.
	 *     @type string $class       Additional CSS classes.
	 *     @type int    $min         Minimum value.
	 *     @type int    $max         Maximum value.
	 *     @type int    $step        Step increment.
	 * }
	 * @return void
	 */
	public static function number_input( $name, $value, $args = array() ) {
		$defaults = array(
			'id'          => $name,
			'label'       => '',
			'description' => '',
			'class'       => '',
			'min'         => null,
			'max'         => null,
			'step'        => 1,
		);
		$args     = wp_parse_args( $args, $defaults );

		$input_classes = array( 'intufind-input', 'intufind-input--number' );
		if ( ! empty( $args['class'] ) ) {
			$input_classes[] = $args['class'];
		}

		echo '<div class="intufind-field">';

		if ( ! empty( $args['label'] ) ) {
			echo '<label class="intufind-label" for="' . esc_attr( $args['id'] ) . '">';
			echo esc_html( $args['label'] );
			echo '</label>';
		}

		$input_attrs = array(
			'type'  => 'number',
			'id'    => $args['id'],
			'name'  => $name,
			'value' => $value,
			'class' => implode( ' ', $input_classes ),
		);

		if ( null !== $args['min'] ) {
			$input_attrs['min'] = $args['min'];
		}
		if ( null !== $args['max'] ) {
			$input_attrs['max'] = $args['max'];
		}
		if ( $args['step'] ) {
			$input_attrs['step'] = $args['step'];
		}

		echo '<input ' . self::build_attrs( $input_attrs ) . ' />'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

		if ( ! empty( $args['description'] ) ) {
			echo '<p class="intufind-description">' . esc_html( $args['description'] ) . '</p>';
		}

		echo '</div>';
	}

	/**
	 * Render a card container.
	 *
	 * @param array $args {
	 *     Card arguments.
	 *     @type string   $title       Card title.
	 *     @type string   $description Card description below title.
	 *     @type string   $content     Card content (can be HTML).
	 *     @type callable $render      Callback function to render content.
	 *     @type string   $class       Additional CSS classes.
	 *     @type string   $id          Card ID attribute.
	 * }
	 * @return void
	 */
	public static function card( $args = array() ) {
		$defaults = array(
			'title'       => '',
			'description' => '',
			'content'     => '',
			'render'      => null,
			'class'       => '',
			'id'          => '',
		);
		$args     = wp_parse_args( $args, $defaults );

		$classes = array( 'intufind-card' );
		if ( ! empty( $args['class'] ) ) {
			$classes[] = $args['class'];
		}

		$id_attr = ! empty( $args['id'] ) ? ' id="' . esc_attr( $args['id'] ) . '"' : '';

		echo '<div class="' . esc_attr( implode( ' ', $classes ) ) . '"' . $id_attr . '>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

		if ( ! empty( $args['title'] ) || ! empty( $args['description'] ) ) {
			echo '<div class="intufind-card__header">';
			if ( ! empty( $args['title'] ) ) {
				echo '<h3 class="intufind-card__title">' . esc_html( $args['title'] ) . '</h3>';
			}
			if ( ! empty( $args['description'] ) ) {
				echo '<p class="intufind-card__description">' . esc_html( $args['description'] ) . '</p>';
			}
			echo '</div>';
		}

		echo '<div class="intufind-card__content">';
		if ( is_callable( $args['render'] ) ) {
			call_user_func( $args['render'] );
		} elseif ( ! empty( $args['content'] ) ) {
			echo wp_kses_post( $args['content'] );
		}
		echo '</div>';

		echo '</div>';
	}

	/**
	 * Render a collapsible section.
	 *
	 * @param string          $title   Section title.
	 * @param string|callable $content Content string or render callback.
	 * @param array           $args    {
	 *     Optional. Collapsible arguments.
	 *     @type bool   $open  Whether section starts open. Default false.
	 *     @type string $id    Section ID.
	 *     @type string $class Additional CSS classes.
	 * }
	 * @return void
	 */
	public static function collapsible( $title, $content, $args = array() ) {
		$defaults = array(
			'open'  => false,
			'id'    => '',
			'class' => '',
		);
		$args     = wp_parse_args( $args, $defaults );

		$classes   = array( 'intufind-collapsible' );
		$classes[] = $args['open'] ? 'intufind-collapsible--open' : '';
		if ( ! empty( $args['class'] ) ) {
			$classes[] = $args['class'];
		}

		$id_attr = ! empty( $args['id'] ) ? ' id="' . esc_attr( $args['id'] ) . '"' : '';

		echo '<div class="' . esc_attr( implode( ' ', array_filter( $classes ) ) ) . '"' . $id_attr . '>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo '<button type="button" class="intufind-collapsible__trigger" aria-expanded="' . ( $args['open'] ? 'true' : 'false' ) . '">';
		echo '<span class="intufind-collapsible__icon"></span>';
		echo '<span class="intufind-collapsible__title">' . esc_html( $title ) . '</span>';
		echo '</button>';
		echo '<div class="intufind-collapsible__content"' . ( $args['open'] ? '' : ' hidden' ) . '>';

		if ( is_callable( $content ) ) {
			call_user_func( $content );
		} else {
			echo wp_kses_post( $content );
		}

		echo '</div>';
		echo '</div>';
	}

	/**
	 * Render a badge.
	 *
	 * @param string $text    Badge text.
	 * @param string $variant Badge variant: success, warning, error, info, neutral. Default 'neutral'.
	 * @param array  $args    {
	 *     Optional. Badge arguments.
	 *     @type string $icon  Dashicon name.
	 *     @type string $class Additional CSS classes.
	 * }
	 * @return void
	 */
	public static function badge( $text, $variant = 'neutral', $args = array() ) {
		$defaults = array(
			'icon'  => '',
			'class' => '',
		);
		$args     = wp_parse_args( $args, $defaults );

		$classes = array( 'intufind-badge', 'intufind-badge--' . $variant );
		if ( ! empty( $args['class'] ) ) {
			$classes[] = $args['class'];
		}

		echo '<span class="' . esc_attr( implode( ' ', $classes ) ) . '">';
		if ( ! empty( $args['icon'] ) ) {
			echo '<span class="dashicons dashicons-' . esc_attr( $args['icon'] ) . '"></span> ';
		}
		echo esc_html( $text );
		echo '</span>';
	}

	/**
	 * Render an alert message.
	 *
	 * @param string $message Alert message.
	 * @param string $variant Alert variant: success, warning, error, info. Default 'info'.
	 * @param array  $args    {
	 *     Optional. Alert arguments.
	 *     @type bool   $dismissible Whether alert can be dismissed. Default false.
	 *     @type string $class       Additional CSS classes.
	 * }
	 * @return void
	 */
	public static function alert( $message, $variant = 'info', $args = array() ) {
		$defaults = array(
			'dismissible' => false,
			'class'       => '',
		);
		$args     = wp_parse_args( $args, $defaults );

		$classes = array( 'intufind-alert', 'intufind-alert--' . $variant );
		if ( $args['dismissible'] ) {
			$classes[] = 'intufind-alert--dismissible';
		}
		if ( ! empty( $args['class'] ) ) {
			$classes[] = $args['class'];
		}

		$icons = array(
			'success' => 'yes-alt',
			'warning' => 'warning',
			'error'   => 'dismiss',
			'info'    => 'info',
		);
		$icon  = isset( $icons[ $variant ] ) ? $icons[ $variant ] : 'info';

		echo '<div class="' . esc_attr( implode( ' ', $classes ) ) . '" role="alert">';
		echo '<span class="dashicons dashicons-' . esc_attr( $icon ) . ' intufind-alert__icon"></span>';
		echo '<span class="intufind-alert__message">' . esc_html( $message ) . '</span>';
		if ( $args['dismissible'] ) {
			echo '<button type="button" class="intufind-alert__dismiss" aria-label="' . esc_attr__( 'Dismiss', 'intufind' ) . '">';
			echo '<span class="dashicons dashicons-no-alt"></span>';
			echo '</button>';
		}
		echo '</div>';
	}

	/**
	 * Render a progress bar.
	 *
	 * @param int    $percent Progress percentage (0-100).
	 * @param string $label   Optional label text.
	 * @param array  $args    Optional additional arguments.
	 * @return void
	 */
	public static function progress( $percent, $label = '', $args = array() ) {
		$percent = max( 0, min( 100, intval( $percent ) ) );

		echo '<div class="intufind-progress">';
		if ( ! empty( $label ) ) {
			echo '<div class="intufind-progress__label">';
			echo '<span>' . esc_html( $label ) . '</span>';
			echo '<span>' . esc_html( $percent ) . '%</span>';
			echo '</div>';
		}
		echo '<div class="intufind-progress__bar">';
		echo '<div class="intufind-progress__fill" style="width: ' . esc_attr( $percent ) . '%;"></div>';
		echo '</div>';
		echo '</div>';
	}

	/**
	 * Render a status indicator.
	 *
	 * @param string $status Status: connected, syncing, error, disconnected.
	 * @param string $text   Optional status text. If empty, uses default text.
	 * @return void
	 */
	public static function status_indicator( $status, $text = '' ) {
		$statuses = array(
			'connected'    => array(
				'class' => 'success',
				'text'  => __( 'Connected', 'intufind' ),
				'icon'  => 'yes-alt',
			),
			'syncing'      => array(
				'class' => 'info',
				'text'  => __( 'Syncing...', 'intufind' ),
				'icon'  => 'update',
			),
			'error'        => array(
				'class' => 'error',
				'text'  => __( 'Error', 'intufind' ),
				'icon'  => 'warning',
			),
			'disconnected' => array(
				'class' => 'neutral',
				'text'  => __( 'Not connected', 'intufind' ),
				'icon'  => 'minus',
			),
		);

		$config       = isset( $statuses[ $status ] ) ? $statuses[ $status ] : $statuses['disconnected'];
		$display_text = ! empty( $text ) ? $text : $config['text'];

		echo '<span class="intufind-status intufind-status--' . esc_attr( $config['class'] ) . '">';
		echo '<span class="dashicons dashicons-' . esc_attr( $config['icon'] ) . '"></span> ';
		echo esc_html( $display_text );
		echo '</span>';
	}

	/**
	 * Render a branded page header with logo, title, and optional subtitle.
	 *
	 * @param string $title    Page title.
	 * @param array  $args     {
	 *     Optional. Header arguments.
	 *     @type string $subtitle  Subtitle/description text.
	 *     @type string $icon      Custom icon name (dashicon without prefix). Default uses logo.
	 * }
	 * @return void
	 */
	public static function page_header( $title, $args = array() ) {
		$defaults = array(
			'subtitle' => '',
			'icon'     => '',
		);
		$args     = wp_parse_args( $args, $defaults );

		echo '<div class="intufind-page-header">';

		// Logo mark with gradient background.
		echo '<div class="intufind-page-logo">';
		if ( ! empty( $args['icon'] ) ) {
			echo '<span class="dashicons dashicons-' . esc_attr( $args['icon'] ) . '" style="color: white; font-size: 20px; width: 20px; height: 20px;"></span>';
		} else {
			// Stylized "i" logo mark - centered in viewBox.
			echo '<svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
			echo '<circle cx="12" cy="6" r="2.5" fill="white"/>';
			echo '<rect x="9.5" y="10" width="5" height="12" rx="1" fill="white"/>';
			echo '</svg>';
		}
		echo '</div>';

		echo '<div class="intufind-page-title-group">';
		echo '<h1 class="intufind-page-title">' . esc_html( $title ) . '</h1>';
		if ( ! empty( $args['subtitle'] ) ) {
			echo '<p class="intufind-page-subtitle">' . esc_html( $args['subtitle'] ) . '</p>';
		}
		echo '</div>';

		echo '</div>';
	}

	/**
	 * Render a dismissible intro notice that persists dismissal in user meta.
	 *
	 * @param string $notice_id Unique identifier for this notice.
	 * @param string $message   Notice message (can contain HTML if message_html is true).
	 * @param array  $args      {
	 *     Optional. Notice arguments.
	 *     @type string $title        Optional title for the notice.
	 *     @type string $icon         Dashicon name (without prefix). Default 'lightbulb'.
	 *     @type string $class        Additional CSS classes.
	 *     @type bool   $message_html Whether message contains HTML. Default false.
	 * }
	 * @return void
	 */
	public static function intro_notice( $notice_id, $message, $args = array() ) {
		// Check if user has dismissed this notice.
		$user_id           = get_current_user_id();
		$dismissed_notices = get_user_meta( $user_id, 'intufind_dismissed_notices', true );

		if ( ! is_array( $dismissed_notices ) ) {
			$dismissed_notices = array();
		}

		if ( in_array( $notice_id, $dismissed_notices, true ) ) {
			return;
		}

		$defaults = array(
			'title'        => '',
			'icon'         => 'lightbulb',
			'class'        => '',
			'message_html' => false,
		);
		$args     = wp_parse_args( $args, $defaults );

		$classes = array( 'intufind-intro-notice' );
		if ( ! empty( $args['class'] ) ) {
			$classes[] = $args['class'];
		}

		echo '<div class="' . esc_attr( implode( ' ', $classes ) ) . '" data-notice-id="' . esc_attr( $notice_id ) . '">';
		echo '<div class="intufind-intro-notice__icon">';
		echo '<span class="dashicons dashicons-' . esc_attr( $args['icon'] ) . '"></span>';
		echo '</div>';
		echo '<div class="intufind-intro-notice__content">';
		if ( ! empty( $args['title'] ) ) {
			echo '<strong class="intufind-intro-notice__title">' . esc_html( $args['title'] ) . '</strong>';
		}
		echo '<p class="intufind-intro-notice__message">';
		if ( $args['message_html'] ) {
			echo wp_kses_post( $message );
		} else {
			echo esc_html( $message );
		}
		echo '</p>';
		echo '</div>';
		echo '<button type="button" class="intufind-intro-notice__dismiss" aria-label="' . esc_attr__( 'Dismiss', 'intufind' ) . '">';
		echo '<span class="dashicons dashicons-no-alt"></span>';
		echo '</button>';
		echo '</div>';
	}

	/**
	 * Build HTML attributes string from array.
	 *
	 * @param array $attrs Array of attribute => value pairs.
	 * @return string HTML attributes string (already escaped).
	 */
	private static function build_attrs( $attrs ) {
		$parts = array();
		foreach ( $attrs as $attr => $value ) {
			if ( is_bool( $value ) ) {
				if ( $value ) {
					$parts[] = esc_attr( $attr );
				}
			} else {
				$parts[] = esc_attr( $attr ) . '="' . esc_attr( $value ) . '"';
			}
		}
		return implode( ' ', $parts );
	}
}
