/**
 * Intufind Admin JavaScript
 *
 * Handles admin UI interactivity including API key validation,
 * workspace selection, and sync operations.
 *
 * @package Intufind
 */

/* global intufindAdmin */

(($) => {
	/**
	 * Intufind Admin module.
	 */
	const IntufindAdmin = {
		/**
		 * Cached DOM elements.
		 */
		elements: {},

		/**
		 * State.
		 */
		state: {
			workspaces: [],
			suggestedWorkspace: '',
			isConnecting: false,
		},

		/**
		 * Initialize the module.
		 */
		init: function () {
			this.cacheElements();
			this.bindEvents();
		},

		/**
		 * Cache DOM elements.
		 */
		cacheElements: function () {
			this.elements = {
				apiKeyInput: $('#intufind-api-key'),
				connectBtn: $('#intufind-connect-btn'),
				connectActions: $('#intufind-connect-actions'),
				connectStatus: $('#intufind-connect-status'),
				connectionStatus: $('.intufind-connection-status'),
				changeKeyBtn: $('#intufind-change-key'),
				cancelChangeBtn: $('#intufind-cancel-change-btn'),
				disconnectBtn: $('#intufind-disconnect-btn'),
				workspacePending: $('#intufind-workspace-pending'),
				workspaceSelector: $('#intufind-workspace-selector'),
				workspaceSelect: $('#intufind-workspace-select'),
				workspaceActions: $('#intufind-workspace-actions'),
				switchWorkspaceBtn: $('#intufind-switch-workspace-btn'),
				existsDialog: $('#intufind-workspace-exists-dialog'),
				existsWorkspaceId: $('#intufind-workspace-exists-id'),
				reconnectBtn: $('#intufind-reconnect-btn'),
				createNewBtn: $('#intufind-create-new-btn'),
			};
			// Store original key for cancel functionality.
			this.originalApiKey = this.elements.apiKeyInput.val();
		},

		/**
		 * Bind event handlers.
		 */
		bindEvents: function () {
			// Connect button.
			this.elements.connectBtn.on('click', this.handleConnect.bind(this));

			// Change key button.
			this.elements.changeKeyBtn.on('click', this.handleChangeKey.bind(this));

			// Cancel change button.
			this.elements.cancelChangeBtn.on('click', this.handleCancelChange.bind(this));

			// Disconnect button.
			this.elements.disconnectBtn.on('click', this.handleDisconnect.bind(this));

			// Workspace selection change.
			this.elements.workspaceSelect.on('change', this.handleWorkspaceChange.bind(this));

			// Switch workspace button.
			this.elements.switchWorkspaceBtn.on('click', this.handleSwitchWorkspace.bind(this));

			// Dialog buttons.
			this.elements.reconnectBtn.on('click', this.handleReconnect.bind(this));
			this.elements.createNewBtn.on('click', this.handleCreateNew.bind(this));

			// Collapsible sections.
			$('.intufind-collapsible__trigger').on('click', this.handleCollapsibleToggle);

			// Dismissible alerts.
			$('.intufind-alert__dismiss').on('click', this.handleAlertDismiss);

			// Dismissible intro notices.
			$(document).on('click', '.intufind-intro-notice__dismiss', this.handleIntroNoticeDismiss);

			// Password visibility toggle.
			$('.intufind-password-toggle').on('click', this.handlePasswordToggle);
		},

		/**
		 * Handle connect button click.
		 */
		handleConnect: function (e) {
			e.preventDefault();

			if (this.state.isConnecting) {
				return;
			}

			const apiKey = this.elements.apiKeyInput.val().trim();

			if (!apiKey) {
				this.showStatus(intufindAdmin.strings.error, 'error');
				return;
			}

			this.state.isConnecting = true;
			this.elements.connectBtn.prop('disabled', true);
			this.showStatus(intufindAdmin.strings.validating, 'loading');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_validate_key',
					nonce: intufindAdmin.nonce,
					api_key: apiKey,
				},
				success: this.handleValidateSuccess.bind(this),
				error: this.handleValidateError.bind(this),
				complete: function () {
					this.state.isConnecting = false;
					this.elements.connectBtn.prop('disabled', false);
				}.bind(this),
			});
		},

		/**
		 * Handle validation success.
		 */
		handleValidateSuccess: function (response) {
			if (!response.success) {
				this.showStatus(response.data.message || intufindAdmin.strings.error, 'error');
				return;
			}

			const data = response.data;
			this.state.workspaces = data.workspaces || [];
			this.state.suggestedWorkspace = data.suggestedWorkspace;

			// Populate workspace dropdown.
			this.populateWorkspaces();

			if (data.workspaceExists) {
				// Check if user is re-connecting to the same workspace (e.g., re-saving key).
				const currentWorkspace = intufindAdmin.workspaceId || '';
				if (currentWorkspace && currentWorkspace === data.suggestedWorkspace) {
					// Auto-reconnect to same workspace without dialog.
					this.switchToWorkspace(data.suggestedWorkspace, 'use');
				} else {
					// Show exists dialog for different workspace.
					this.elements.existsWorkspaceId.text(data.suggestedWorkspace);
					this.showDialog();
				}
			} else {
				// Auto-create workspace.
				this.createWorkspace(data.suggestedWorkspace);
			}
		},

		/**
		 * Handle validation error.
		 */
		handleValidateError: function (_xhr, _status, error) {
			this.showStatus(`${intufindAdmin.strings.error}: ${error}`, 'error');
		},

		/**
		 * Show status message.
		 */
		showStatus: function (message, type) {
			this.elements.connectStatus.text(message).removeClass('is-loading is-success is-error').addClass(`is-${type}`);
		},

		/**
		 * Populate workspaces dropdown.
		 */
		populateWorkspaces: function () {
			const $select = this.elements.workspaceSelect;
			$select.find('option:not(:first)').remove();

			this.state.workspaces.forEach((workspace) => {
				$select.append(
					$('<option>')
						.val(workspace.workspaceId)
						.text(workspace.name || workspace.workspaceId),
				);
			});

			// Hide the pending message and show the workspace selector.
			this.elements.workspacePending.hide();
			this.elements.workspaceSelector.show();
		},

		/**
		 * Handle workspace selection change.
		 */
		handleWorkspaceChange: function () {
			const selected = this.elements.workspaceSelect.val();
			if (selected) {
				this.elements.workspaceActions.show();
			} else {
				this.elements.workspaceActions.hide();
			}
		},

		/**
		 * Handle switch workspace button.
		 */
		handleSwitchWorkspace: function (e) {
			e.preventDefault();

			const workspaceId = this.elements.workspaceSelect.val();
			if (!workspaceId) {
				return;
			}

			if (!confirm(intufindAdmin.strings.confirmSwitch)) {
				return;
			}

			this.switchToWorkspace(workspaceId, 'use');
		},

		/**
		 * Handle change key button.
		 */
		handleChangeKey: function (e) {
			e.preventDefault();

			// Store original value for cancel.
			this.originalApiKey = this.elements.apiKeyInput.val();

			// Make input editable.
			this.elements.apiKeyInput.prop('readonly', false).prop('type', 'password').val('').focus();

			// Hide change button and connection status, show save actions.
			this.elements.changeKeyBtn.hide();
			this.elements.connectionStatus.hide();
			this.elements.connectActions.show();
		},

		/**
		 * Handle cancel change button.
		 */
		handleCancelChange: function (e) {
			e.preventDefault();

			// Restore original key.
			this.elements.apiKeyInput.prop('readonly', true).prop('type', 'password').val(this.originalApiKey);

			// Restore UI state.
			this.elements.connectActions.hide();
			this.elements.connectionStatus.show();
			this.elements.changeKeyBtn.show();
			this.elements.connectStatus.text('');
		},

		/**
		 * Handle disconnect button click.
		 */
		handleDisconnect: function (e) {
			e.preventDefault();

			const confirmMessage =
				intufindAdmin.strings.confirmDisconnect ||
				'Are you sure you want to disconnect? Your synced content will remain in the cloud, but you will need to reconnect to manage it.';

			if (!confirm(confirmMessage)) {
				return;
			}

			this.elements.disconnectBtn
				.prop('disabled', true)
				.text(intufindAdmin.strings.disconnecting || 'Disconnecting...');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_disconnect',
					nonce: intufindAdmin.nonce,
				},
				success: function (response) {
					if (response.success) {
						// Reload page to show disconnected state.
						window.location.reload();
					} else {
						alert(response.data.message || intufindAdmin.strings.error);
						this.elements.disconnectBtn.prop('disabled', false).text(intufindAdmin.strings.disconnect || 'Disconnect');
					}
				}.bind(this),
				error: function () {
					alert(intufindAdmin.strings.error);
					this.elements.disconnectBtn.prop('disabled', false).text(intufindAdmin.strings.disconnect || 'Disconnect');
				}.bind(this),
			});
		},

		/**
		 * Show workspace exists dialog.
		 */
		showDialog: function () {
			// Add backdrop.
			$('body').append('<div class="intufind-dialog-backdrop"></div>');
			this.elements.existsDialog.show();

			// Handle backdrop click.
			$('.intufind-dialog-backdrop').on('click', this.hideDialog.bind(this));
		},

		/**
		 * Hide workspace exists dialog.
		 */
		hideDialog: function () {
			$('.intufind-dialog-backdrop').remove();
			this.elements.existsDialog.hide();
		},

		/**
		 * Handle reconnect button.
		 */
		handleReconnect: function (e) {
			e.preventDefault();
			this.hideDialog();
			this.switchToWorkspace(this.state.suggestedWorkspace, 'use');
		},

		/**
		 * Handle create new button.
		 */
		handleCreateNew: function (e) {
			e.preventDefault();
			this.hideDialog();

			// Generate a new unique ID.
			const newId = `${this.state.suggestedWorkspace}-${Date.now().toString(36).slice(-4)}`;
			this.createWorkspace(newId);
		},

		/**
		 * Create a new workspace.
		 */
		createWorkspace: function (workspaceId) {
			this.switchToWorkspace(workspaceId, 'create');
		},

		/**
		 * Switch to workspace.
		 */
		switchToWorkspace: function (workspaceId, actionType) {
			// Get sync on connect preference.
			const syncOnConnect = $('#intufind-sync-on-connect').is(':checked');

			this.showStatus(
				syncOnConnect
					? intufindAdmin.strings.connectingAndSyncing || 'Connecting and syncing...'
					: intufindAdmin.strings.saving,
				'loading',
			);

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_switch_workspace',
					nonce: intufindAdmin.nonce,
					workspace_id: workspaceId,
					action_type: actionType,
					sync_on_connect: syncOnConnect ? '1' : '0',
				},
				success: function (response) {
					if (response.success) {
						this.showStatus(response.data.message || intufindAdmin.strings.saved, 'success');
						// Reload page to show connected state.
						setTimeout(() => window.location.reload(), 1000);
					} else {
						this.showStatus(response.data.message || intufindAdmin.strings.error, 'error');
					}
				}.bind(this),
				error: function (_xhr, _status, error) {
					this.showStatus(`${intufindAdmin.strings.error}: ${error}`, 'error');
				}.bind(this),
			});
		},

		/**
		 * Handle collapsible toggle.
		 */
		handleCollapsibleToggle: function () {
			const $trigger = $(this);
			const $collapsible = $trigger.closest('.intufind-collapsible');
			const $content = $collapsible.find('.intufind-collapsible__content');
			const isOpen = $collapsible.hasClass('intufind-collapsible--open');

			$collapsible.toggleClass('intufind-collapsible--open');
			$trigger.attr('aria-expanded', !isOpen);

			if (isOpen) {
				$content.attr('hidden', 'hidden');
			} else {
				$content.removeAttr('hidden');
			}
		},

		/**
		 * Handle alert dismiss.
		 */
		handleAlertDismiss: function () {
			$(this)
				.closest('.intufind-alert')
				.fadeOut(200, function () {
					$(this).remove();
				});
		},

		/**
		 * Handle intro notice dismiss - saves to user meta.
		 */
		handleIntroNoticeDismiss: function () {
			const $notice = $(this).closest('.intufind-intro-notice');
			const noticeId = $notice.data('notice-id');

			// Fade out immediately for responsive feel.
			$notice.fadeOut(200, function () {
				$(this).remove();
			});

			// Save dismissal to user meta.
			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_dismiss_notice',
					nonce: intufindAdmin.nonce,
					notice_id: noticeId,
				},
			});
		},

		/**
		 * Handle password visibility toggle.
		 */
		handlePasswordToggle: function () {
			const $toggle = $(this);
			const targetId = $toggle.data('target');
			const $input = $(`#${targetId}`);

			if ($input.length) {
				const isPassword = $input.attr('type') === 'password';
				$input.attr('type', isPassword ? 'text' : 'password');
				$toggle.toggleClass('is-visible', isPassword);
				$toggle.attr(
					'aria-label',
					isPassword ? intufindAdmin.strings.hidePassword : intufindAdmin.strings.showPassword,
				);
			}
		},
	};

	/**
	 * Intufind Sync module.
	 */
	const IntufindSync = {
		/**
		 * State.
		 */
		state: {
			isSyncing: false,
			syncQueue: [],
			limitError: null, // Tracks plan limit errors during sync all.
		},

		/**
		 * Initialize the module.
		 */
		init: function () {
			this.bindEvents();
		},

		/**
		 * Bind event handlers.
		 */
		bindEvents: function () {
			// Individual sync buttons.
			$(document).on('click', '.intufind-sync-btn', this.handleSync.bind(this));

			// Retry buttons.
			$(document).on('click', '.intufind-retry-btn', this.handleRetry.bind(this));

			// Sync all button.
			$('#intufind-sync-all').on('click', this.handleSyncAll.bind(this));

			// Taxonomy sync button.
			$('#intufind-sync-taxonomies-btn').on('click', this.handleSyncTaxonomies.bind(this));

			// Content type toggle.
			$(document).on('change', '.intufind-type-toggle', this.handleTypeToggle.bind(this));

			// Taxonomy toggle.
			$(document).on('change', '.intufind-taxonomy-toggle', this.handleTaxonomyToggle.bind(this));
		},

		/**
		 * Handle individual sync button click.
		 */
		handleSync: function (e) {
			e.preventDefault();

			const $btn = $(e.currentTarget);
			const postType = $btn.data('post-type');

			if (!postType || $btn.prop('disabled')) {
				return;
			}

			this.syncPostType(postType, $btn);
		},

		/**
		 * Handle retry button click.
		 */
		handleRetry: function (e) {
			e.preventDefault();

			const $btn = $(e.currentTarget);
			const postType = $btn.data('post-type');

			if (!postType || $btn.prop('disabled')) {
				return;
			}

			this.retryFailed(postType, $btn);
		},

		/**
		 * Handle sync all button click.
		 */
		handleSyncAll: function (e) {
			e.preventDefault();

			if (this.state.isSyncing) {
				return;
			}

			// Collect all post types.
			const postTypes = [];
			$('.intufind-sync-btn').each(function () {
				const postType = $(this).data('post-type');
				if (postType) {
					postTypes.push(postType);
				}
			});

			if (postTypes.length === 0) {
				return;
			}

			this.state.isSyncing = true;
			this.state.syncQueue = [...postTypes];
			this.state.limitError = null; // Reset limit error tracking.

			// Show progress and log.
			$('#intufind-sync-all').prop('disabled', true);
			$('#intufind-sync-progress').show();
			$('#intufind-sync-log').show();
			$('.intufind-sync-log__list').empty();

			// Process queue.
			this.processSyncQueue();
		},

		/**
		 * Handle taxonomy sync button click.
		 */
		handleSyncTaxonomies: function (e) {
			e.preventDefault();

			const $btn = $(e.currentTarget);

			if ($btn.prop('disabled')) {
				return;
			}

			$btn.prop('disabled', true).addClass('is-syncing');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_sync_taxonomies',
					nonce: intufindAdmin.nonce,
				},
				success: (response) => {
					if (response.success) {
						// Update taxonomy status badges.
						this.updateTaxonomyStatus(response.data.taxonomy_status);
						// Update taxonomy overview stat card.
						this.updateTaxonomyOverviewCard(response.data.taxonomy_status);
						// Update last sync time display.
						if (response.data.stats?.last_sync_ago) {
							$('.intufind-taxonomy-last-sync').html(
								`&bull; ${intufindAdmin.strings.lastSynced || 'Last synced:'} ${response.data.stats.last_sync_ago}`,
							);
						}
						this.handleSyncNotice(response);
					} else {
						this.showNotice(response.data.message || 'Taxonomy sync failed', 'error');
					}
				},
				error: (_xhr, _status, error) => {
					this.showNotice(`Taxonomy sync failed: ${error}`, 'error');
				},
				complete: () => {
					$btn.prop('disabled', false).removeClass('is-syncing');
				},
			});
		},

		/**
		 * Update taxonomy status badges after sync.
		 *
		 * @param {Object} taxonomyStatus - Map of taxonomy slug to sync status.
		 */
		updateTaxonomyStatus: (taxonomyStatus) => {
			if (!taxonomyStatus) {
				return;
			}

			for (const taxonomy in taxonomyStatus) {
				const status = taxonomyStatus[taxonomy];
				const $row = $(`tr[data-taxonomy="${taxonomy}"]`);

				if ($row.length === 0) {
					continue;
				}

				const $statusCell = $row.find('.intufind-sync-table__status');

				if (status.synced) {
					$statusCell.html(
						`<span class="intufind-badge intufind-badge--success" title="${status.last_sync_ago || ''}">${intufindAdmin.strings.synced || 'Synced'
						}</span>`,
					);
				} else {
					$statusCell.html(
						`<span class="intufind-badge intufind-badge--muted">${intufindAdmin.strings.notSynced || 'Not synced'}</span>`,
					);
				}
			}
		},

		/**
		 * Update taxonomy overview stat cards after sync.
		 *
		 * @param {Object} taxonomyStatus - Map of taxonomy slug to sync status.
		 */
		updateTaxonomyOverviewCard: (taxonomyStatus) => {
			if (!taxonomyStatus) {
				return;
			}

			let synced = 0;
			let total = 0;

			for (const taxonomy in taxonomyStatus) {
				total++;
				if (taxonomyStatus[taxonomy].synced) {
					synced++;
				}
			}

			const pending = total - synced;
			const $taxonomySection = $('.intufind-taxonomy-overview');
			$taxonomySection.find('.intufind-stat-card--synced .intufind-stat-number').text(synced);
			$taxonomySection.find('.intufind-stat-card--pending .intufind-stat-number').text(pending);
		},

		/**
		 * Process sync queue sequentially.
		 */
		processSyncQueue: function () {
			if (this.state.syncQueue.length === 0) {
				this.finishSyncAll();
				return;
			}

			const postType = this.state.syncQueue.shift();
			const label = $(`tr[data-post-type="${postType}"] strong`).text() || postType;

			$('#intufind-sync-progress .intufind-sync-progress__text').text(`${intufindAdmin.strings.syncing} ${label}...`);

			this.addLogEntry(`${label}...`, 'pending');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_manual_sync',
					nonce: intufindAdmin.nonce,
					post_type: postType,
				},
				success: (response) => {
					if (response.success) {
						this.updateRowStats(postType, response.data.stats);

						// Check for plan limit errors.
						if (response.data.error_info?.is_limit) {
							// Store limit error info for display after sync all completes.
							this.state.limitError = response.data.error_info;
							this.updateLogEntry(
								`${label}: ${intufindAdmin.strings.planLimitReached || 'Plan limit reached'} (${response.data.error_info.current_count}/${response.data.error_info.plan_limit})`,
								'error',
							);
						} else if (response.data.error_messages?.length > 0) {
							this.updateLogEntry(`${label}: ${response.data.message}`, 'error');
						} else {
							this.updateLogEntry(`${label}: ${response.data.message}`, 'success');
						}
					} else {
						this.updateLogEntry(`${label}: ${response.data.message || 'Error'}`, 'error');
					}
					this.processSyncQueue();
				},
				error: (_xhr, _status, error) => {
					this.updateLogEntry(`${label}: ${error}`, 'error');
					this.processSyncQueue();
				},
			});
		},

		/**
		 * Finish sync all operation - sync taxonomies then complete.
		 */
		finishSyncAll: function () {
			// Sync taxonomies as part of "Sync All".
			$('#intufind-sync-progress .intufind-sync-progress__text').text(
				`${intufindAdmin.strings.syncing} ${intufindAdmin.strings.taxonomies || 'Taxonomies'}...`,
			);
			this.addLogEntry(`${intufindAdmin.strings.taxonomies || 'Taxonomies'}...`, 'pending');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_sync_taxonomies',
					nonce: intufindAdmin.nonce,
				},
				success: (response) => {
					if (response.success) {
						this.updateLogEntry(
							`${intufindAdmin.strings.taxonomies || 'Taxonomies'}: ${response.data.message}`,
							'success',
						);
						this.updateTaxonomyStatus(response.data.taxonomy_status);
						this.updateTaxonomyOverviewCard(response.data.taxonomy_status);
					} else {
						this.updateLogEntry(
							`${intufindAdmin.strings.taxonomies || 'Taxonomies'}: ${response.data.message || 'Error'}`,
							'error',
						);
					}
				},
				error: (_xhr, _status, error) => {
					this.updateLogEntry(`${intufindAdmin.strings.taxonomies || 'Taxonomies'}: ${error}`, 'error');
				},
				complete: () => {
					this.completeSyncAll();
				},
			});
		},

		/**
		 * Complete the sync all operation after taxonomies are done.
		 */
		completeSyncAll: function () {
			this.state.isSyncing = false;
			$('#intufind-sync-all').prop('disabled', false);
			$('#intufind-sync-progress').hide();
			$('#intufind-sync-progress .intufind-sync-progress__text').text(intufindAdmin.strings.syncing);

			// Show plan limit notice if a limit error occurred during sync.
			if (this.state.limitError) {
				this.showPlanLimitNotice(this.state.limitError);
				this.state.limitError = null; // Clear after showing.
			}

			// Refresh stats.
			this.refreshStats();
		},

		/**
		 * Sync a single post type.
		 */
		syncPostType: function (postType, $btn) {
			$btn.prop('disabled', true).addClass('is-syncing');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_manual_sync',
					nonce: intufindAdmin.nonce,
					post_type: postType,
				},
				success: (response) => {
					if (response.success) {
						this.updateRowStats(postType, response.data.stats);
						this.handleSyncNotice(response);
					} else {
						this.showNotice(response.data.message || intufindAdmin.strings.error, 'error');
					}
				},
				error: (_xhr, _status, error) => {
					this.showNotice(`${intufindAdmin.strings.error}: ${error}`, 'error');
				},
				complete: () => {
					$btn.prop('disabled', false).removeClass('is-syncing');
					this.refreshOverviewStats();
				},
			});
		},

		/**
		 * Retry failed syncs for a post type.
		 */
		retryFailed: function (postType, $btn) {
			$btn.prop('disabled', true).addClass('is-syncing');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_retry_failed',
					nonce: intufindAdmin.nonce,
					post_type: postType,
				},
				success: (response) => {
					if (response.success) {
						this.updateRowStats(postType, response.data.stats);
						this.showNotice(response.data.message, 'success');
					} else {
						this.showNotice(response.data.message || intufindAdmin.strings.error, 'error');
					}
				},
				error: (_xhr, _status, error) => {
					this.showNotice(`${intufindAdmin.strings.error}: ${error}`, 'error');
				},
				complete: () => {
					$btn.prop('disabled', false).removeClass('is-syncing');
					this.refreshOverviewStats();
				},
			});
		},

		/**
		 * Handle content type toggle change.
		 */
		handleTypeToggle: function (e) {
			const $toggle = $(e.currentTarget);
			const isEnabled = $toggle.is(':checked');
			const $row = $toggle.closest('tr');

			// Disable toggle during save.
			$toggle.prop('disabled', true);

			// Collect all enabled post types.
			const enabledTypes = [];
			$('.intufind-type-toggle:checked').each(function () {
				enabledTypes.push($(this).data('post-type'));
			});

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_save_post_types',
					nonce: intufindAdmin.nonce,
					post_types: enabledTypes,
				},
				success: (response) => {
					if (response.success) {
						// Update row styling.
						if (isEnabled) {
							$row.removeClass('intufind-sync-row--disabled');
							// Reload page to get fresh stats for newly enabled type.
							window.location.reload();
						} else {
							$row.addClass('intufind-sync-row--disabled');
							// Update the row UI for disabled state.
							$row.find('.intufind-sync-table__status').html('<span class="intufind-text-muted">&mdash;</span>');
							$row.find('.intufind-sync-table__progress').html('<span class="intufind-text-muted">&mdash;</span>');
							$row.find('.intufind-sync-table__last').html('<span class="intufind-text-muted">&mdash;</span>');
							$row.find('.intufind-sync-table__actions').empty();
							// Update overview stats.
							this.refreshOverviewStats();
						}
					} else {
						// Revert toggle on error.
						$toggle.prop('checked', !isEnabled);
						this.showNotice(response.data.message || intufindAdmin.strings.error, 'error');
					}
				},
				error: (_xhr, _status, error) => {
					// Revert toggle on error.
					$toggle.prop('checked', !isEnabled);
					this.showNotice(`${intufindAdmin.strings.error}: ${error}`, 'error');
				},
				complete: () => {
					$toggle.prop('disabled', false);
				},
			});
		},

		/**
		 * Handle taxonomy toggle change.
		 */
		handleTaxonomyToggle: function (e) {
			const $toggle = $(e.currentTarget);
			const isEnabled = $toggle.is(':checked');
			const $row = $toggle.closest('tr');

			// Disable toggle during save.
			$toggle.prop('disabled', true);

			// Collect all enabled taxonomies.
			const enabledTaxonomies = [];
			$('.intufind-taxonomy-toggle:checked').each(function () {
				enabledTaxonomies.push($(this).data('taxonomy'));
			});

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_save_taxonomies',
					nonce: intufindAdmin.nonce,
					taxonomies: enabledTaxonomies,
				},
				success: (response) => {
					if (response.success) {
						// Update row styling.
						if (isEnabled) {
							$row.removeClass('intufind-sync-row--disabled');
						} else {
							$row.addClass('intufind-sync-row--disabled');
						}
						// Update total count in footer.
						this.updateTaxonomyTotalCount();
					} else {
						// Revert toggle on error.
						$toggle.prop('checked', !isEnabled);
						this.showNotice(response.data.message || intufindAdmin.strings.error, 'error');
					}
				},
				error: (_xhr, _status, error) => {
					// Revert toggle on error.
					$toggle.prop('checked', !isEnabled);
					this.showNotice(`${intufindAdmin.strings.error}: ${error}`, 'error');
				},
				complete: () => {
					$toggle.prop('disabled', false);
				},
			});
		},

		/**
		 * Update taxonomy total count in footer.
		 */
		updateTaxonomyTotalCount: () => {
			let total = 0;
			$('.intufind-taxonomy-toggle:checked').each(function () {
				const $row = $(this).closest('tr');
				const count = parseInt($row.find('.intufind-sync-table__terms').text().replace(/,/g, ''), 10);
				if (!Number.isNaN(count)) {
					total += count;
				}
			});
			$('.intufind-taxonomy-summary .intufind-text-muted')
				.first()
				.text(
					intufindAdmin.strings.totalEnabled.replace('%s', total.toLocaleString()) ||
					`Total enabled: ${total.toLocaleString()} terms`,
				);
		},

		/**
		 * Update row stats after sync.
		 */
		updateRowStats: (postType, stats) => {
			if (!stats || !stats.counts) {
				return;
			}

			const $row = $(`tr[data-post-type="${postType}"]`);
			if ($row.length === 0) {
				return;
			}

			const counts = stats.counts;
			const total = Math.max(1, counts.total);
			const synced = counts.synced || 0;
			const pending = (counts.pending || 0) + (counts.not_synced || 0);
			const errors = counts.error || 0;
			const percentage = Math.round((synced / total) * 100);

			// Update badges.
			let badgesHtml = '';
			if (synced > 0) {
				badgesHtml += `<span class="intufind-badge intufind-badge--success" title="Synced">${synced}</span>`;
			}
			if (pending > 0) {
				badgesHtml += `<span class="intufind-badge intufind-badge--warning" title="Pending">${pending}</span>`;
			}
			if (errors > 0) {
				badgesHtml += `<span class="intufind-badge intufind-badge--error" title="Errors">${errors}</span>`;
			}
			$row.find('.intufind-sync-status-badges').html(badgesHtml);

			// Update progress bar.
			$row.find('.intufind-progress__bar').css('width', `${percentage}%`);
			$row.find('.intufind-progress__text').text(`${percentage}%`);

			// Update last sync time.
			if (stats.last_sync_ago) {
				$row.find('.intufind-sync-table__last span').text(stats.last_sync_ago);
			}

			// Show/hide retry button.
			const $retryBtn = $row.find('.intufind-retry-btn');
			if (errors > 0 && $retryBtn.length === 0) {
				$row.find('.intufind-button-group').append(`
          <button type="button" class="intufind-btn intufind-btn--ghost intufind-btn--small intufind-retry-btn" data-post-type="${postType}">
            Retry
          </button>
        `);
			} else if (errors === 0) {
				$retryBtn.remove();
			}
		},

		/**
		 * Refresh overview stats.
		 */
		refreshOverviewStats: () => {
			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_get_sync_stats',
					nonce: intufindAdmin.nonce,
				},
				success: (response) => {
					if (!response.success || !response.data.stats) {
						return;
					}

					let totalSynced = 0;
					let totalPending = 0;
					let totalErrors = 0;
					let totalEligible = 0;

					for (const postType in response.data.stats) {
						const stats = response.data.stats[postType];
						const counts = stats.counts || {};

						totalSynced += counts.synced || 0;
						totalPending += (counts.pending || 0) + (counts.not_synced || 0);
						totalErrors += counts.error || 0;
						totalEligible += counts.total || 0;
					}

					// Scope to content overview only (not taxonomy overview).
					const $contentSection = $('.intufind-content-overview');
					$contentSection.find('.intufind-stat-card--synced .intufind-stat-number').text(totalSynced.toLocaleString());
					$contentSection.find('.intufind-stat-card--pending .intufind-stat-number').text(totalPending.toLocaleString());
					$contentSection.find('.intufind-stat-card--errors .intufind-stat-number').text(totalErrors.toLocaleString());
					$contentSection.find('.intufind-stat-card--total .intufind-stat-number').text(totalEligible.toLocaleString());
				},
			});
		},

		/**
		 * Refresh all stats.
		 */
		refreshStats: function () {
			this.refreshOverviewStats();

			// Update individual rows.
			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_get_sync_stats',
					nonce: intufindAdmin.nonce,
				},
				success: (response) => {
					if (!response.success || !response.data.stats) {
						return;
					}

					for (const postType in response.data.stats) {
						this.updateRowStats(postType, response.data.stats[postType]);
					}
				},
			});
		},

		/**
		 * Add log entry.
		 */
		addLogEntry: (message, status) => {
			const $list = $('.intufind-sync-log__list');
			$list.append(`<li class="${status}" data-pending="true">${message}</li>`);
			$list.scrollTop($list[0].scrollHeight);
		},

		/**
		 * Update last log entry.
		 */
		updateLogEntry: (message, status) => {
			const $lastEntry = $('.intufind-sync-log__list li[data-pending="true"]:last');
			if ($lastEntry.length) {
				$lastEntry.removeClass('pending').addClass(status).text(message).removeAttr('data-pending');
			}
		},

		/**
		 * Handle sync response and show appropriate notice.
		 *
		 * @param {Object} response - AJAX response object.
		 */
		handleSyncNotice: function (response) {
			if (response.data.error_info?.is_limit) {
				this.showPlanLimitNotice(response.data.error_info);
			} else if (response.data.error_messages?.length > 0) {
				this.showSyncErrorNotice(response.data.error_messages, response.data.message);
			} else {
				this.showNotice(response.data.message, 'success');
			}
		},

		/**
		 * Create dismiss button for notices.
		 *
		 * @returns {jQuery} Dismiss button element.
		 */
		createDismissButton: () =>
			$('<button>')
				.attr('type', 'button')
				.addClass('intufind-notice__dismiss')
				.html('&times;')
				.on('click', function () {
					$(this)
						.closest('.intufind-notice')
						.fadeOut(200, function () {
							$(this).remove();
						});
				}),

		/**
		 * Insert notice into the page.
		 *
		 * @param {jQuery} $notice - Notice element to insert.
		 */
		insertNotice: ($notice) => {
			$('.intufind-notice').remove();
			$('.intufind-page-header').after($notice);
		},

		/**
		 * Show notice.
		 *
		 * @param {string} message - Notice message (will be escaped)
		 * @param {string} type - Notice type ('success', 'error', 'warning', 'info')
		 */
		showNotice: function (message, type) {
			const $notice = $('<div>')
				.addClass(`intufind-notice intufind-notice--${type}`)
				.append($('<p>').text(message))
				.append(this.createDismissButton());

			this.insertNotice($notice);

			// Auto-dismiss after 5 seconds.
			setTimeout(() => {
				$notice.fadeOut(200, function () {
					$(this).remove();
				});
			}, 5000);
		},

		/**
		 * Show plan limit exceeded notice with upgrade CTA.
		 *
		 * @param {Object} errorInfo - Error info object with plan limit details.
		 */
		showPlanLimitNotice: function (errorInfo) {
			const currentCount = errorInfo.current_count || 0;
			const planLimit = errorInfo.plan_limit || 0;
			const upgradeUrl = errorInfo.upgrade_url || 'https://intufind.com/dashboard/subscription';

			const $notice = $('<div>')
				.addClass('intufind-notice intufind-notice--error intufind-notice--limit')
				.append(
					$('<div>')
						.addClass('intufind-notice__content')
						.append(
							$('<p>')
								.addClass('intufind-notice__title')
								.html(`<strong>${intufindAdmin.strings.planLimitTitle || 'Plan Limit Reached'}</strong>`),
						)
						.append(
							$('<p>').text(
								(intufindAdmin.strings.planLimitMessage ||
									"You have reached your plan's document limit ({current} of {limit}). Upgrade your plan to sync more content.")
									.replace('{current}', currentCount)
									.replace('{limit}', planLimit),
							),
						)
						.append(
							$('<p>')
								.addClass('intufind-notice__actions')
								.append(
									$('<a>')
										.attr('href', upgradeUrl)
										.attr('target', '_blank')
										.attr('rel', 'noopener')
										.addClass('button button-primary')
										.text(intufindAdmin.strings.upgradePlan || 'Upgrade Plan'),
								),
						),
				)
				.append(this.createDismissButton());

			this.insertNotice($notice);
			// Don't auto-dismiss limit notices - user needs to see them.
		},

		/**
		 * Show sync error notice with error details.
		 *
		 * @param {Array} errorMessages - Array of error message strings.
		 * @param {string} summaryMessage - Summary message from sync operation.
		 */
		showSyncErrorNotice: function (errorMessages, summaryMessage) {
			const $notice = $('<div>')
				.addClass('intufind-notice intufind-notice--error')
				.append(
					$('<div>')
						.addClass('intufind-notice__content')
						.append($('<p>').text(summaryMessage))
						.append(
							$('<details>')
								.addClass('intufind-notice__details')
								.append($('<summary>').text(intufindAdmin.strings.viewErrorDetails || 'View error details'))
								.append(
									$('<ul>')
										.addClass('intufind-notice__error-list')
										.append(errorMessages.slice(0, 5).map((msg) => $('<li>').text(msg))),
								),
						),
				)
				.append(this.createDismissButton());

			this.insertNotice($notice);
			// Don't auto-dismiss error notices with details.
		},
	};

	/**
	 * Intufind List Columns module.
	 * Handles sync/searchable toggles in post list tables.
	 */
	const IntufindListColumns = {
		/**
		 * Initialize the module.
		 */
		init: function () {
			this.bindEvents();
		},

		/**
		 * Bind event handlers.
		 */
		bindEvents: function () {
			$(document).on('change', '.intufind-sync-toggle', this.handleSyncToggle.bind(this));
			$(document).on('change', '.intufind-searchable-toggle', this.handleSearchableToggle.bind(this));
		},

		/**
		 * Handle sync toggle change.
		 */
		handleSyncToggle: (e) => {
			const $checkbox = $(e.currentTarget);
			const postId = $checkbox.data('post-id');
			const nonce = $checkbox.data('nonce');
			const $container = $checkbox.closest('.intufind-col-toggle');
			const $row = $checkbox.closest('tr');

			$container.addClass('is-loading');
			$checkbox.prop('disabled', true);

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_toggle_sync',
					post_id: postId,
					nonce: nonce,
				},
				success: (response) => {
					if (response.success) {
						// Find the searchable column container (by column class, not by checkbox).
						const $searchableContainer = $row.find('.column-intufind_searchable .intufind-col-toggle');

						if (response.data.synced) {
							// Sync enabled - update status indicator based on sync result.
							const statusClass = response.data.syncStatus === 'success' ? 'success' : 'error';
							const statusTitle =
								response.data.syncStatus === 'success'
									? intufindAdmin.strings?.synced || 'Synced'
									: intufindAdmin.strings?.syncError || 'Sync error';

							const $existingStatus = $container.find('.intufind-col-status');
							if ($existingStatus.length) {
								$existingStatus
									.removeClass('intufind-col-status--pending intufind-col-status--success intufind-col-status--error')
									.addClass(`intufind-col-status--${statusClass}`)
									.attr('title', statusTitle);
							} else {
								$container.append(
									`<span class="intufind-col-status intufind-col-status--${statusClass}" title="${statusTitle}"></span>`,
								);
							}

							// Re-enable searchable toggle if it was disabled.
							if ($searchableContainer.length && $searchableContainer.hasClass('intufind-col-toggle--disabled') && response.data.syncStatus === 'success') {
								// Restore searchable toggle (default to checked/searchable for published posts).
								const searchableNonce = response.data.searchableNonce || '';
								$searchableContainer
									.removeClass('intufind-col-toggle--disabled')
									.removeAttr('title')
									.html(`
										<label class="intufind-mini-toggle">
											<input type="checkbox" class="intufind-searchable-toggle" data-post-id="${postId}" data-nonce="${searchableNonce}" checked />
											<span class="intufind-mini-toggle__slider"></span>
										</label>
									`);
							}
						} else {
							// Sync disabled - remove status and disable searchable.
							$container.find('.intufind-col-status').remove();
							if ($searchableContainer.length) {
								$searchableContainer
									.html('<span class="intufind-col-dash">&mdash;</span>')
									.addClass('intufind-col-toggle--disabled')
									.attr('title', 'Enable sync first');
							}
						}
					} else {
						// Revert on error.
						$checkbox.prop('checked', !$checkbox.prop('checked'));
						alert(response.data.message || 'Error toggling sync.');
					}
				},
				error: () => {
					$checkbox.prop('checked', !$checkbox.prop('checked'));
					alert('Request failed.');
				},
				complete: () => {
					$container.removeClass('is-loading');
					$checkbox.prop('disabled', false);
				},
			});
		},

		/**
		 * Handle searchable toggle change.
		 */
		handleSearchableToggle: (e) => {
			const $checkbox = $(e.currentTarget);
			const postId = $checkbox.data('post-id');
			const nonce = $checkbox.data('nonce');
			const $container = $checkbox.closest('.intufind-col-toggle');

			$container.addClass('is-loading');
			$checkbox.prop('disabled', true);

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_toggle_searchable',
					post_id: postId,
					nonce: nonce,
				},
				success: (response) => {
					if (!response.success) {
						$checkbox.prop('checked', !$checkbox.prop('checked'));
						alert(response.data.message || 'Error toggling searchable.');
					}
				},
				error: () => {
					$checkbox.prop('checked', !$checkbox.prop('checked'));
					alert('Request failed.');
				},
				complete: () => {
					$container.removeClass('is-loading');
					$checkbox.prop('disabled', false);
				},
			});
		},
	};

	/**
	 * Settings Module Factory
	 *
	 * Creates auto-save settings modules with consistent behavior.
	 * Reduces code duplication across search, recommendations, chat, and sync settings.
	 *
	 * @param {Object} config - Module configuration
	 * @param {string} config.optionClass - CSS class for option inputs (e.g., 'intufind-search-option')
	 * @param {string} config.action - AJAX action name
	 * @param {Function} [config.onSuccess] - Optional callback on successful save
	 * @param {Function} [config.onInit] - Optional callback after init
	 * @returns {Object} Settings module instance
	 */
	const createSettingsModule = (config) => ({
		saveTimeout: null,

		init: function () {
			if ($(`.${config.optionClass}`).length === 0) {
				return;
			}
			this.bindEvents();
			if (config.onInit) {
				config.onInit.call(this);
			}
		},

		bindEvents: function () {
			$(document).on('change', `.${config.optionClass}`, this.handleOptionChange.bind(this));
		},

		handleOptionChange: function (e) {
			const $input = $(e.target);
			const name = $input.attr('name');
			const value = $input.is(':checkbox') ? ($input.is(':checked') ? '1' : '0') : $input.val();

			// Debounce the save.
			if (this.saveTimeout) {
				clearTimeout(this.saveTimeout);
			}

			this.saveTimeout = setTimeout(() => {
				this.saveOption(name, value, $input);
			}, 300);
		},

		saveOption: (name, value, $input) => {
			const $container = $input.closest('.intufind-field, .intufind-search-toggle');
			$container.addClass('is-saving');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: config.action,
					nonce: intufindAdmin.nonce,
					option_name: name,
					option_value: value,
				},
				success: (response) => {
					if (response.success) {
						$container.addClass('is-saved');
						setTimeout(() => $container.removeClass('is-saved'), 1000);
						if (config.onSuccess) {
							config.onSuccess(response, $input);
						}
					} else {
						alert(response.data?.message || intufindAdmin.strings.error);
					}
				},
				error: () => {
					alert(intufindAdmin.strings.error);
				},
				complete: () => {
					$container.removeClass('is-saving');
				},
			});
		},
	});

	/**
	 * Search Settings Module
	 */
	const IntufindSearch = createSettingsModule({
		optionClass: 'intufind-search-option',
		action: 'intufind_save_search_option',
	});

	/**
	 * Recommendations Settings Module
	 */
	const IntufindRecommendations = createSettingsModule({
		optionClass: 'intufind-recommendations-option',
		action: 'intufind_save_recommendations_option',
	});

	/**
	 * Auto-Sync Settings Module
	 */
	const IntufindAutoSync = createSettingsModule({
		optionClass: 'intufind-sync-option',
		action: 'intufind_save_sync_option',
		onInit: () => {
			// Toggle visibility of interval settings based on enabled state.
			$(document).on('change', 'input[name="intufind_auto_sync_enabled"]', (e) => {
				const isEnabled = $(e.target).is(':checked');
				const $fields = $('.intufind-auto-sync-fields');
				if (isEnabled) {
					$fields.slideDown(200);
				} else {
					$fields.slideUp(200);
				}
			});
		},
		onSuccess: (response) => {
			// Update next sync time display if provided.
			if (response.data?.next_sync) {
				$('.intufind-next-sync-time').html(
					`<span class="dashicons dashicons-clock"></span> ${response.data.next_sync}`,
				);
			}
		},
	});

	/**
	 * Chat Settings Module
	 */
	const IntufindChat = createSettingsModule({
		optionClass: 'intufind-chat-option',
		action: 'intufind_save_chat_option',
	});

	/**
	 * MCP Registration Retry Handler.
	 *
	 * Handles retry of MCP domain registration from the status page.
	 */
	const IntufindMcpRetry = {
		$btn: null,
		$status: null,

		init: function () {
			this.$btn = $('#intufind-retry-mcp');
			if (!this.$btn.length) {
				return;
			}
			this.$status = $('#intufind-retry-mcp-status');
			this.$btn.on('click', this.handleRetry.bind(this));
		},

		handleRetry: function (e) {
			e.preventDefault();

			// Prevent double-click.
			if (this.$btn.prop('disabled')) {
				return;
			}

			this.$btn.prop('disabled', true).addClass('is-busy');
			this.$status.html('<span class="spinner is-active" style="float: none; margin: 0;"></span>');

			$.ajax({
				url: intufindAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'intufind_retry_mcp_registration',
					nonce: intufindAdmin.nonce,
				},
				success: (response) => {
					if (response.success) {
						this.$status.empty().append($('<span>').css('color', '#00a32a').text(response.data.message));
						// Reload after brief delay to show updated status.
						setTimeout(() => window.location.reload(), 1500);
					} else {
						this.$status.empty().append(
							$('<span>')
								.css('color', '#d63638')
								.text(response.data?.message || 'Registration failed.'),
						);
					}
				},
				error: () => {
					this.$status.empty().append($('<span>').css('color', '#d63638').text('Network error. Please try again.'));
				},
				complete: () => {
					this.$btn.prop('disabled', false).removeClass('is-busy');
				},
			});
		},
	};

	// Initialize on document ready.
	$(document).ready(() => {
		IntufindAdmin.init();
		IntufindSync.init();
		IntufindListColumns.init();
		IntufindSearch.init();
		IntufindChat.init();
		IntufindRecommendations.init();
		IntufindAutoSync.init();
		IntufindMcpRetry.init();
	});
})(jQuery);
