<?php
/**
 * Search Settings page template.
 *
 * Configure search override options for WordPress and WooCommerce.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$is_connected       = get_option( INTUFIND_OPTION_CONNECTED, false );
$is_woo_active      = class_exists( 'WooCommerce' );
$wp_search_enabled  = get_option( Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH, false );
$woo_search_enabled = get_option( Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH, false );
$mixed_enabled      = get_option( Intufind_Search_Override::OPTION_ENABLE_MIXED_SEARCH, true );
$search_limit       = get_option( Intufind_Search_Override::OPTION_SEARCH_LIMIT, 20 );
$cache_duration     = get_option( Intufind_Search_Override::OPTION_SEARCH_CACHE_DURATION, 3600 );
?>

<div class="wrap intufind-wrap">
	<?php
	Intufind_Components::page_header(
		__( 'Search', 'intufind' ),
		array(
			'subtitle' => __( 'Replace default search with AI-powered semantic search.', 'intufind' ),
			'icon'     => 'search',
		)
	);

	// Intro notice for new users.
	Intufind_Components::intro_notice(
		'search_intro',
		sprintf(
			/* translators: %s: link to documentation */
			__( 'AI-powered search understands meaning, not just keywords, so visitors get relevant results even when they don\'t use exact terms. Enable it to replace your default WordPress or WooCommerce search. <a href="%s" target="_blank" rel="noopener">Learn more about AI search</a>.', 'intufind' ),
			'https://intufind.com/docs/wordpress/search'
		),
		array(
			'title'        => __( 'About AI Search', 'intufind' ),
			'message_html' => true,
		)
	);
	?>

	<?php
	// Show link to web dashboard.
	$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
	if ( $is_connected && ! empty( $workspace_id ) ) :
		$dashboard_url = 'https://intufind.com/dashboard/workspaces/' . urlencode( $workspace_id ) . '/search';
		?>
		<p class="intufind-dashboard-link">
			<span class="dashicons dashicons-external"></span>
			<?php
			printf(
				/* translators: %s: Link to dashboard */
				esc_html__( 'Configure advanced search settings on the %s', 'intufind' ),
				'<a href="' . esc_url( $dashboard_url ) . '" target="_blank" rel="noopener">' . esc_html__( 'Intufind Dashboard', 'intufind' ) . '</a>'
			);
			?>
		</p>
		<?php
	endif;
	?>

	<?php if ( ! $is_connected ) : ?>
		<?php
		Intufind_Components::alert(
			__( 'Please connect your account in Settings before configuring search.', 'intufind' ),
			'warning'
		);
		?>
	<?php else : ?>
		<div class="intufind-settings">
			<!-- Search Override Card -->
			<?php
			Intufind_Components::card(
				array(
					'title'       => __( 'Search Override', 'intufind' ),
					'description' => __( 'Replace default WordPress and WooCommerce search with Intufind semantic search for more relevant results.', 'intufind' ),
					'render'      => function () use ( $wp_search_enabled, $woo_search_enabled, $is_woo_active ) {
						?>
						<form id="intufind-search-form" method="post" action="">
							<?php wp_nonce_field( 'intufind_search_settings', 'intufind_search_nonce' ); ?>

							<div class="intufind-search-toggles">
								<!-- WordPress Search Toggle -->
								<div class="intufind-search-toggle">
									<label class="intufind-toggle-row">
										<span class="intufind-toggle-info">
											<span class="intufind-toggle-title">
												<span class="dashicons dashicons-wordpress"></span>
												<?php esc_html_e( 'WordPress Search', 'intufind' ); ?>
											</span>
											<span class="intufind-toggle-description">
												<?php esc_html_e( 'Override the default WordPress search with semantic search for posts and pages.', 'intufind' ); ?>
											</span>
										</span>
										<span class="intufind-toggle">
											<input
												type="checkbox"
												name="<?php echo esc_attr( Intufind_Search_Override::OPTION_ENABLE_WP_SEARCH ); ?>"
												class="intufind-toggle__input intufind-search-option"
												value="1"
												<?php checked( $wp_search_enabled ); ?>
											/>
											<span class="intufind-toggle__slider"></span>
										</span>
									</label>
								</div>

								<!-- WooCommerce Search Toggle -->
								<?php if ( $is_woo_active ) : ?>
								<div class="intufind-search-toggle">
									<label class="intufind-toggle-row">
										<span class="intufind-toggle-info">
											<span class="intufind-toggle-title">
												<span class="dashicons dashicons-cart"></span>
												<?php esc_html_e( 'WooCommerce Search', 'intufind' ); ?>
											</span>
											<span class="intufind-toggle-description">
												<?php esc_html_e( 'Override WooCommerce product search on shop pages with semantic search.', 'intufind' ); ?>
											</span>
										</span>
										<span class="intufind-toggle">
											<input
												type="checkbox"
												name="<?php echo esc_attr( Intufind_Search_Override::OPTION_ENABLE_WOO_SEARCH ); ?>"
												class="intufind-toggle__input intufind-search-option"
												value="1"
												<?php checked( $woo_search_enabled ); ?>
											/>
											<span class="intufind-toggle__slider"></span>
										</span>
									</label>
								</div>
								<?php else : ?>
								<div class="intufind-search-toggle intufind-search-toggle--disabled">
									<div class="intufind-toggle-row">
										<span class="intufind-toggle-info">
											<span class="intufind-toggle-title">
												<span class="dashicons dashicons-cart"></span>
												<?php esc_html_e( 'WooCommerce Search', 'intufind' ); ?>
											</span>
											<span class="intufind-toggle-description">
												<?php esc_html_e( 'WooCommerce is not active. Install and activate WooCommerce to enable product search override.', 'intufind' ); ?>
											</span>
										</span>
										<span class="intufind-toggle intufind-toggle--disabled">
											<input type="checkbox" class="intufind-toggle__input" disabled />
											<span class="intufind-toggle__slider"></span>
										</span>
									</div>
								</div>
								<?php endif; ?>
							</div>
						</form>
						<?php
					},
				)
			);
			?>

			<!-- Advanced Settings Card -->
			<?php
			Intufind_Components::card(
				array(
					'title'       => __( 'Advanced Settings', 'intufind' ),
					'description' => __( 'Fine-tune search behavior and performance.', 'intufind' ),
					'render'      => function () use ( $mixed_enabled, $search_limit, $cache_duration, $is_woo_active ) {
						?>
						<form id="intufind-search-advanced-form" method="post" action="">
							<?php wp_nonce_field( 'intufind_search_settings', 'intufind_search_advanced_nonce' ); ?>

							<div class="intufind-form-fields">
								<?php if ( $is_woo_active ) : ?>
								<!-- Mixed Content Search -->
								<div class="intufind-field">
									<label class="intufind-toggle-row intufind-toggle-row--compact">
										<span class="intufind-toggle-info">
											<span class="intufind-label"><?php esc_html_e( 'Mixed Content Search', 'intufind' ); ?></span>
											<span class="intufind-description">
												<?php esc_html_e( 'Include both products and posts in general search results.', 'intufind' ); ?>
											</span>
										</span>
										<span class="intufind-toggle">
											<input
												type="checkbox"
												name="<?php echo esc_attr( Intufind_Search_Override::OPTION_ENABLE_MIXED_SEARCH ); ?>"
												class="intufind-toggle__input intufind-search-option"
												value="1"
												<?php checked( $mixed_enabled ); ?>
											/>
											<span class="intufind-toggle__slider"></span>
										</span>
									</label>
								</div>
								<?php endif; ?>

								<!-- Search Limit -->
								<?php
								Intufind_Components::number_input(
									Intufind_Search_Override::OPTION_SEARCH_LIMIT,
									$search_limit,
									array(
										'label'       => __( 'Maximum Results', 'intufind' ),
										'description' => __( 'Maximum number of results to return from semantic search (1-100).', 'intufind' ),
										'min'         => 1,
										'max'         => 100,
										'step'        => 1,
										'class'       => 'intufind-search-option',
									)
								);
								?>

								<!-- Cache Duration -->
								<?php
								Intufind_Components::select_input(
									Intufind_Search_Override::OPTION_SEARCH_CACHE_DURATION,
									$cache_duration,
									array(
										'label'       => __( 'Cache Duration', 'intufind' ),
										'description' => __( 'How long to cache search results for improved performance.', 'intufind' ),
										'options'     => array(
											0     => __( 'No caching', 'intufind' ),
											300   => __( '5 minutes', 'intufind' ),
											900   => __( '15 minutes', 'intufind' ),
											1800  => __( '30 minutes', 'intufind' ),
											3600  => __( '1 hour', 'intufind' ),
											7200  => __( '2 hours', 'intufind' ),
											21600 => __( '6 hours', 'intufind' ),
											86400 => __( '24 hours', 'intufind' ),
										),
										'class'       => 'intufind-search-option',
									)
								);
								?>
							</div>
						</form>
						<?php
					},
				)
			);
			?>

			<!-- How It Works Card -->
			<?php
			Intufind_Components::card(
				array(
					'title'  => __( 'How It Works', 'intufind' ),
					'render' => function () {
						?>
						<div class="intufind-info-content">
							<div class="intufind-info-section">
								<h4><?php esc_html_e( 'Semantic Search', 'intufind' ); ?></h4>
								<p>
									<?php esc_html_e( 'Unlike traditional keyword matching, Intufind uses AI to understand the meaning behind search queries. This means users can find relevant content even when they don\'t use exact keywords.', 'intufind' ); ?>
								</p>
							</div>

							<div class="intufind-info-section">
								<h4><?php esc_html_e( 'Automatic Fallback', 'intufind' ); ?></h4>
								<p>
									<?php esc_html_e( 'If the Intufind API is unavailable, search automatically falls back to WordPress native search, ensuring your site always works.', 'intufind' ); ?>
								</p>
							</div>

							<div class="intufind-info-section">
								<h4><?php esc_html_e( 'Searchable Content', 'intufind' ); ?></h4>
								<p>
									<?php esc_html_e( 'Only synced content marked as "Searchable" appears in search results. Use the Knowledge page and post list columns to control what\'s searchable.', 'intufind' ); ?>
								</p>
							</div>
						</div>
						<?php
					},
				)
			);
			?>
		</div>
	<?php endif; ?>
</div>
