<?php
/**
 * Settings page template.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$api_key      = get_option( INTUFIND_OPTION_API_KEY, '' );
$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
$is_connected = get_option( INTUFIND_OPTION_CONNECTED, false );

// Verify connection with the cloud if we think we're connected.
$connection_error = null;
if ( $is_connected ) {
	$api               = new Intufind_API();
	$verification      = $api->verify_connection();
	if ( ! $verification['valid'] ) {
		$connection_error = isset( $verification['error'] ) ? $verification['error'] : __( 'Connection invalid.', 'intufind' );
		$is_connected     = false; // Update local state for template.
		// Note: INTUFIND_OPTION_CONNECTED is cleared by verify_connection() if invalid.
	}
}

// Mask API key for display.
$api_key_display = '';
if ( ! empty( $api_key ) ) {
	$api_key_display = substr( $api_key, 0, 10 ) . str_repeat( '•', 20 );
}
?>

<div class="wrap intufind-wrap">
	<?php
	Intufind_Components::page_header(
		__( 'Settings', 'intufind' ),
		array(
			'subtitle' => __( 'Connect and configure your Intufind workspace.', 'intufind' ),
			'icon'     => 'admin-generic',
		)
	);

	// Intro notice for new users.
	Intufind_Components::intro_notice(
		'settings_intro',
		sprintf(
			/* translators: %1$s: link to create account, %2$s: link to dashboard */
			__( 'To connect, paste your Secret Key below. Don\'t have one yet? <a href="%1$s" target="_blank" rel="noopener">Create a free Intufind account</a> to get your key. Already signed up? You can find your key in your <a href="%2$s" target="_blank" rel="noopener">Intufind dashboard</a>.', 'intufind' ),
			'https://intufind.com/signup',
			'https://intufind.com/dashboard/api-keys'
		),
		array(
			'title'        => __( 'Welcome to Intufind', 'intufind' ),
			'message_html' => true,
		)
	);
	?>

	<div class="intufind-settings">
		<?php
		// Show connection error notice if verification failed.
		if ( $connection_error ) :
			?>
			<div class="notice notice-error inline" style="margin-bottom: 16px;">
				<p>
					<strong><?php esc_html_e( 'Connection Error:', 'intufind' ); ?></strong>
					<?php echo esc_html( $connection_error ); ?>
					<?php esc_html_e( 'Please reconnect with a valid API key.', 'intufind' ); ?>
				</p>
			</div>
			<?php
		endif;

		Intufind_Components::card(
			array(
				'title'  => __( 'Connection', 'intufind' ),
				'render' => function () use ( $api_key, $api_key_display, $workspace_id, $is_connected, $connection_error ) {
					?>
					<form id="intufind-settings-form" method="post" action="">
						<?php wp_nonce_field( 'intufind_settings', 'intufind_settings_nonce' ); ?>

						<?php
						Intufind_Components::text_input(
							'api_key',
							$is_connected ? $api_key_display : $api_key,
							array(
								'type'                 => $is_connected ? 'text' : 'password',
								'id'                   => 'intufind-api-key',
								'label'                => __( 'Secret Key', 'intufind' ),
								'placeholder'          => __( 'Enter your secret key', 'intufind' ),
								'class'                => 'intufind-input--monospace',
								'readonly'             => $is_connected,
								'show_password_toggle' => ! $is_connected,
								'description'          => sprintf(
									/* translators: %s: Link to dashboard */
									__( 'Get your secret key from the %s.', 'intufind' ),
									'<a href="https://intufind.com/dashboard" target="_blank" rel="noopener noreferrer">' . __( 'Intufind Dashboard', 'intufind' ) . '</a>'
								),
								'description_html'     => true,
								'suffix'               => $is_connected ? function () {
									Intufind_Components::button(
										__( 'Change', 'intufind' ),
										array(
											'type'    => 'button',
											'id'      => 'intufind-change-key',
											'variant' => 'ghost',
										)
									);
								} : null,
							)
						);
						?>

						<?php if ( $is_connected ) : ?>
							<div class="intufind-connection-status">
								<?php Intufind_Components::status_indicator( 'connected' ); ?>
								<span class="intufind-workspace-info">
									<?php
									printf(
										/* translators: %s: Workspace ID */
										esc_html__( 'Workspace: %s', 'intufind' ),
										'<strong>' . esc_html( $workspace_id ) . '</strong>'
									);
									?>
								</span>
								<?php
								Intufind_Components::link_button(
									__( 'Manage in Dashboard', 'intufind' ),
									'https://intufind.com/dashboard/workspaces/' . urlencode( $workspace_id ),
									array(
										'variant' => 'ghost',
										'target'  => '_blank',
										'icon'    => 'external',
									)
								);
								Intufind_Components::button(
									__( 'Disconnect', 'intufind' ),
									array(
										'type'    => 'button',
										'id'      => 'intufind-disconnect-btn',
										'variant' => 'ghost',
										'class'   => 'intufind-disconnect-btn',
									)
								);
								?>
							</div>
							<!-- Hidden save actions for key change -->
							<div id="intufind-connect-actions" style="display: none;">
								<?php
								Intufind_Components::button(
									__( 'Save', 'intufind' ),
									array(
										'type'    => 'button',
										'id'      => 'intufind-connect-btn',
										'variant' => 'primary',
									)
								);
								Intufind_Components::button(
									__( 'Cancel', 'intufind' ),
									array(
										'type'    => 'button',
										'id'      => 'intufind-cancel-change-btn',
										'variant' => 'ghost',
									)
								);
								?>
								<span id="intufind-connect-status" class="intufind-status-text"></span>
							</div>
						<?php else : ?>
							<?php $sync_on_connect = get_option( Intufind_Sync::OPTION_SYNC_ON_CONNECT, true ); ?>
							<?php if ( ! empty( $api_key ) && ! $connection_error ) : ?>
								<div class="intufind-connection-status intufind-connection-status--pending">
									<?php Intufind_Components::status_indicator( 'disconnected', __( 'Not connected', 'intufind' ) ); ?>
									<span class="intufind-muted-text">
										<?php esc_html_e( 'Secret key saved. Click Connect to activate.', 'intufind' ); ?>
									</span>
								</div>
							<?php elseif ( ! empty( $api_key ) && $connection_error ) : ?>
								<div class="intufind-connection-status intufind-connection-status--error">
									<?php Intufind_Components::status_indicator( 'error', __( 'Connection failed', 'intufind' ) ); ?>
									<span class="intufind-muted-text">
										<?php esc_html_e( 'Your key may be invalid or expired. Please enter a new key.', 'intufind' ); ?>
									</span>
								</div>
							<?php endif; ?>
							<div class="intufind-connect-options">
								<label class="intufind-checkbox-label">
									<input
										type="checkbox"
										id="intufind-sync-on-connect"
										name="sync_on_connect"
										value="1"
										<?php checked( $sync_on_connect ); ?>
									/>
									<span><?php esc_html_e( 'Automatically sync content after connecting', 'intufind' ); ?></span>
								</label>
							</div>
							<div id="intufind-connect-actions">
								<?php
								Intufind_Components::button(
									__( 'Connect', 'intufind' ),
									array(
										'type'    => 'button',
										'id'      => 'intufind-connect-btn',
										'variant' => 'primary',
									)
								);
								?>
								<span id="intufind-connect-status" class="intufind-status-text"></span>
							</div>
						<?php endif; ?>
					</form>

					<?php
					// Advanced section - workspace selection.
					$advanced_content = function () use ( $is_connected, $workspace_id, $api_key ) {
						?>
						<div class="intufind-advanced-section">
							<?php if ( $is_connected ) : ?>
								<p class="intufind-warning-text">
									<span class="dashicons dashicons-warning"></span>
									<?php esc_html_e( 'Switching workspaces? Your synced content stays with the current workspace. You\'ll need to sync again to add content to a new workspace.', 'intufind' ); ?>
								</p>
							<?php endif; ?>

							<!-- Shown before API key is validated -->
							<div id="intufind-workspace-pending" class="intufind-field" <?php echo ( $is_connected || ! empty( $api_key ) ) ? 'style="display:none;"' : ''; ?>>
								<p class="intufind-muted-text">
									<span class="dashicons dashicons-info"></span>
									<?php esc_html_e( 'Enter and validate your Secret Key first to access workspace options.', 'intufind' ); ?>
								</p>
							</div>

							<div id="intufind-workspace-selector" class="intufind-field" <?php echo ! $is_connected ? 'style="display:none;"' : ''; ?>>
								<label class="intufind-label" for="intufind-workspace-select">
									<?php echo $is_connected ? esc_html__( 'Switch to a different workspace', 'intufind' ) : esc_html__( 'Use an existing workspace', 'intufind' ); ?>
								</label>
								<select id="intufind-workspace-select" class="intufind-select" name="workspace_id">
									<option value=""><?php esc_html_e( 'Select a workspace...', 'intufind' ); ?></option>
								</select>
								<div id="intufind-workspace-actions" style="display:none; margin-top: 12px;">
									<?php
									Intufind_Components::button(
										__( 'Switch Workspace', 'intufind' ),
										array(
											'type'    => 'button',
											'id'      => 'intufind-switch-workspace-btn',
											'variant' => 'secondary',
										)
									);
									?>
								</div>
							</div>
						</div>
						<?php
					};

					Intufind_Components::collapsible(
						__( 'Advanced', 'intufind' ),
						$advanced_content,
						array(
							'id'   => 'intufind-advanced',
							'open' => false,
						)
					);
					?>
					<?php
				},
			)
		);
		?>

		<?php
		// Show dashboard info card when connected.
		if ( $is_connected ) :
			?>
			<div class="intufind-info-card">
				<div class="intufind-info-card__icon">
					<span class="dashicons dashicons-external"></span>
				</div>
				<div class="intufind-info-card__content">
					<h3 class="intufind-info-card__title">
						<?php esc_html_e( 'Manage Your AI Assistant', 'intufind' ); ?>
					</h3>
				<p class="intufind-info-card__text">
					<?php esc_html_e( 'Visit your Intufind Dashboard to set up AI-powered search, customize your chat assistant, and see how visitors are engaging with your content.', 'intufind' ); ?>
				</p>
				<ul class="intufind-info-card__list">
					<li><?php esc_html_e( 'Configure AI search and chat widgets', 'intufind' ); ?></li>
					<li><?php esc_html_e( 'View search queries and conversation insights', 'intufind' ); ?></li>
					<li><?php esc_html_e( 'Customize AI personality and responses', 'intufind' ); ?></li>
					<li><?php esc_html_e( 'Monitor usage and performance', 'intufind' ); ?></li>
				</ul>
					<?php
					Intufind_Components::link_button(
						__( 'Open Dashboard', 'intufind' ),
						'https://intufind.com/dashboard/workspaces/' . urlencode( $workspace_id ),
						array(
							'variant' => 'primary',
							'target'  => '_blank',
							'icon'    => 'external',
						)
					);
					?>
				</div>
			</div>
		<?php endif; ?>

		<!-- Workspace exists dialog (hidden by default) -->
		<div id="intufind-workspace-exists-dialog" class="intufind-dialog" style="display: none;">
			<?php
			Intufind_Components::card(
				array(
					'title'  => __( 'Workspace Found', 'intufind' ),
					'render' => function () {
						?>
						<p class="intufind-dialog-message">
							<?php esc_html_e( 'A workspace with this ID already exists in your account.', 'intufind' ); ?>
						</p>
						<p id="intufind-workspace-exists-id" class="intufind-workspace-id"></p>
						<div class="intufind-dialog-actions">
							<?php
							Intufind_Components::button(
								__( 'Reconnect to it', 'intufind' ),
								array(
									'type'    => 'button',
									'id'      => 'intufind-reconnect-btn',
									'variant' => 'primary',
								)
							);
							Intufind_Components::button(
								__( 'Create new workspace', 'intufind' ),
								array(
									'type'    => 'button',
									'id'      => 'intufind-create-new-btn',
									'variant' => 'secondary',
								)
							);
							?>
						</div>
						<?php
					},
				)
			);
			?>
		</div>
	</div>
</div>
