<?php
/**
 * Document Sync page template.
 *
 * Provides document management dashboard with sync status,
 * progress indicators, and enable/disable toggles.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$is_connected = get_option( INTUFIND_OPTION_CONNECTED, false );

// Initialize managers.
$exclusions     = new Intufind_Exclusions();
$status_tracker = new Intufind_Sync_Status();

// Fetch fresh cloud stats on page load (force refresh to show current counts).
$cloud_stats        = null;
$cloud_total        = 0;
$status_reconciled  = false;

if ( $is_connected ) {
	$api         = new Intufind_API();
	$cloud_stats = $api->get_cached_document_stats( true ); // Force fresh fetch

	if ( ! is_wp_error( $cloud_stats ) ) {
		// Only use totalDocuments if it's explicitly present in the response.
		// Missing field should not be treated as 0 (could be partial response or schema change).
		$cloud_total_present = array_key_exists( 'totalDocuments', $cloud_stats );
		$cloud_total         = $cloud_total_present ? (int) $cloud_stats['totalDocuments'] : 0;

		// Reconcile local status with cloud reality.
		// If cloud explicitly shows 0 documents but we have local "synced" records, reset local status.
		// Only reset when totalDocuments is explicitly present and is 0 - not when the field is missing.
		if ( $cloud_total_present && 0 === $cloud_total ) {
			global $wpdb;
			// Check if we have any posts marked as synced locally.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$local_synced_count = (int) $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = %s",
					'_intufind_sync_status',
					'synced'
				)
			);

			if ( $local_synced_count > 0 ) {
				// Cloud is empty but local shows synced items - reset local status.
				$status_tracker->reset_all_sync_status();
				$status_reconciled = true;
			}
		}
	}
}

// Get auto-sync settings.
$auto_sync_enabled = get_option( Intufind_Sync::OPTION_AUTO_SYNC_ENABLED, true );
$sync_interval     = get_option( Intufind_Sync::OPTION_SYNC_INTERVAL, Intufind_Sync::DEFAULT_SYNC_INTERVAL );
$sync_hour         = get_option( Intufind_Sync::OPTION_SYNC_HOUR, Intufind_Sync::DEFAULT_SYNC_HOUR );
$next_sync         = wp_next_scheduled( Intufind_Sync::SCHEDULED_SYNC_HOOK );

$available_post_types = $exclusions->get_available_post_types();
$enabled_post_types   = $exclusions->get_enabled_post_types();

// Get stats for ALL available types (enabled and disabled).
$all_type_stats = array();
foreach ( $available_post_types as $post_type => $label ) {
	$is_enabled = in_array( $post_type, $enabled_post_types, true );
	// Pass exclusions to get accurate counts (excludes system pages, etc.).
	$counts     = $is_enabled ? $status_tracker->get_status_counts( $post_type, $exclusions ) : array(
		'synced'     => 0,
		'pending'    => 0,
		'error'      => 0,
		'not_synced' => 0,
		'total'      => 0,
	);
	$last_sync  = $is_enabled ? $status_tracker->get_last_sync_time( $post_type ) : null;

	// Get total post count for disabled types (approximate, doesn't filter exclusions).
	if ( ! $is_enabled ) {
		$count_obj       = wp_count_posts( $post_type );
		$counts['total'] = isset( $count_obj->publish ) ? (int) $count_obj->publish : 0;
	}

	$all_type_stats[ $post_type ] = array(
		'label'     => $label,
		'enabled'   => $is_enabled,
		'counts'    => $counts,
		'last_sync' => $last_sync,
	);
}

// Calculate totals for enabled types only.
$total_synced   = 0;
$total_pending  = 0;
$total_errors   = 0;
$total_eligible = 0;

foreach ( $all_type_stats as $stats ) {
	if ( $stats['enabled'] ) {
		$total_synced   += $stats['counts']['synced'];
		$total_pending  += $stats['counts']['pending'] + $stats['counts']['not_synced'];
		$total_errors   += $stats['counts']['error'];
		$total_eligible += $stats['counts']['total'];
	}
}

// Calculate taxonomy sync stats for overview card.
$taxonomy_info_early   = $exclusions->get_taxonomy_info();
$api_early             = new Intufind_API();
$sync_manager_early    = new Intufind_Sync( $api_early, null, $exclusions );
$taxonomies_synced     = 0;
$taxonomies_enabled    = 0;

foreach ( $taxonomy_info_early as $tax => $info ) {
	if ( $info['enabled'] ) {
		$taxonomies_enabled++;
		$tax_status = $sync_manager_early->get_taxonomy_sync_status( $tax );
		if ( $tax_status['synced'] ) {
			$taxonomies_synced++;
		}
	}
}
?>

<div class="wrap intufind-wrap">
	<?php
	Intufind_Components::page_header(
		__( 'Knowledge', 'intufind' ),
		array(
			'subtitle' => __( 'Manage your AI knowledge base synchronization.', 'intufind' ),
			'icon'     => 'cloud-upload',
		)
	);

	// Intro notice for new users.
	Intufind_Components::intro_notice(
		'knowledge_intro',
		sprintf(
			/* translators: %s: link to documentation */
			__( 'This is your AI knowledge base. Enable the content types and taxonomies you want your AI to learn from, then sync them. Synced content becomes searchable and available for chat responses. <a href="%s" target="_blank" rel="noopener">Learn more about syncing content</a>.', 'intufind' ),
			'https://intufind.com/docs/wordpress/syncing-content'
		),
		array(
			'title'        => __( 'Getting Started', 'intufind' ),
			'message_html' => true,
		)
	);

	// Show notice if local status was reconciled with cloud.
	if ( $status_reconciled ) :
		Intufind_Components::alert(
			__( 'Local sync status has been reset to match cloud state. Your documents need to be re-synced.', 'intufind' ),
			'info'
		);
	endif;

	// Show link to web dashboard for KB management.
	$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
	if ( $is_connected && ! empty( $workspace_id ) ) :
		$dashboard_url = 'https://intufind.com/dashboard/workspaces/' . urlencode( $workspace_id ) . '/knowledge';
		?>
		<p class="intufind-dashboard-link">
			<span class="dashicons dashicons-external"></span>
			<?php
			printf(
				/* translators: %s: Link to dashboard */
				esc_html__( 'Review and manage your knowledge base on the %s', 'intufind' ),
				'<a href="' . esc_url( $dashboard_url ) . '" target="_blank" rel="noopener">' . esc_html__( 'Intufind Dashboard', 'intufind' ) . '</a>'
			);
			?>
		</p>
		<?php
	endif;
	?>

	<?php if ( ! $is_connected ) : ?>
		<?php
		Intufind_Components::alert(
			__( 'Please connect your account in Settings before syncing documents.', 'intufind' ),
			'warning'
		);
		?>
	<?php else : ?>
		<div class="intufind-settings">
			<!-- Content Types & Sync Status Card -->
			<?php
			Intufind_Components::card(
				array(
					'title'  => __( 'Content Types', 'intufind' ),
					'render' => function () use ( $all_type_stats, $status_tracker, $total_synced, $total_pending, $total_errors, $total_eligible ) {
						?>
					<div class="intufind-sync-overview intufind-sync-overview--inline intufind-content-overview">
						<div class="intufind-stat-cards">
							<div class="intufind-stat-card intufind-stat-card--synced">
								<span class="intufind-stat-number"><?php echo esc_html( number_format_i18n( $total_synced ) ); ?></span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Synced', 'intufind' ); ?></span>
							</div>
							<div class="intufind-stat-card intufind-stat-card--pending">
								<span class="intufind-stat-number"><?php echo esc_html( number_format_i18n( $total_pending ) ); ?></span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Pending', 'intufind' ); ?></span>
							</div>
							<div class="intufind-stat-card intufind-stat-card--errors">
								<span class="intufind-stat-number"><?php echo esc_html( number_format_i18n( $total_errors ) ); ?></span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Errors', 'intufind' ); ?></span>
							</div>
							<div class="intufind-stat-card intufind-stat-card--total">
								<span class="intufind-stat-number"><?php echo esc_html( number_format_i18n( $total_eligible ) ); ?></span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Total', 'intufind' ); ?></span>
							</div>
						</div>
					</div>
						<p class="intufind-description">
							<?php esc_html_e( 'Enable content types to sync for AI-powered search and chat.', 'intufind' ); ?>
						</p>
						<div class="intufind-sync-table-wrapper">
							<table class="intufind-sync-table widefat">
								<thead>
									<tr>
										<th class="intufind-sync-table__toggle"><?php esc_html_e( 'Enabled', 'intufind' ); ?></th>
										<th><?php esc_html_e( 'Content Type', 'intufind' ); ?></th>
										<th class="intufind-sync-table__status"><?php esc_html_e( 'Status', 'intufind' ); ?></th>
										<th class="intufind-sync-table__progress"><?php esc_html_e( 'Progress', 'intufind' ); ?></th>
										<th class="intufind-sync-table__last"><?php esc_html_e( 'Last Sync', 'intufind' ); ?></th>
										<th class="intufind-sync-table__actions"><?php esc_html_e( 'Actions', 'intufind' ); ?></th>
									</tr>
								</thead>
								<tbody>
									<?php foreach ( $all_type_stats as $post_type => $stats ) : ?>
										<?php
										$is_enabled = $stats['enabled'];
										$counts     = $stats['counts'];
										$total      = max( 1, $counts['total'] );
										$synced     = $counts['synced'];
										$errors     = $counts['error'];
										$pending    = $counts['pending'] + $counts['not_synced'];
										$percentage = $is_enabled ? round( ( $synced / $total ) * 100 ) : 0;
										$last_sync  = $stats['last_sync'];
										$row_class  = $is_enabled ? '' : 'intufind-sync-row--disabled';
										?>
										<tr data-post-type="<?php echo esc_attr( $post_type ); ?>" class="<?php echo esc_attr( $row_class ); ?>">
											<td class="intufind-sync-table__toggle">
												<label class="intufind-toggle">
													<input
														type="checkbox"
														class="intufind-toggle__input intufind-type-toggle"
														data-post-type="<?php echo esc_attr( $post_type ); ?>"
														<?php checked( $is_enabled ); ?>
													/>
													<span class="intufind-toggle__slider"></span>
												</label>
											</td>
											<td>
												<strong><?php echo esc_html( $stats['label'] ); ?></strong>
												<span class="intufind-sync-table__count">
													<?php
													printf(
														/* translators: %d: Number of items */
														esc_html( _n( '%d document', '%d documents', $counts['total'], 'intufind' ) ),
														intval( $counts['total'] )
													);
													?>
												</span>
											</td>
											<td class="intufind-sync-table__status">
												<?php if ( $is_enabled ) : ?>
													<div class="intufind-sync-status-badges">
														<?php if ( $synced > 0 ) : ?>
															<span class="intufind-badge intufind-badge--success" title="<?php esc_attr_e( 'Synced', 'intufind' ); ?>">
																<?php echo esc_html( $synced ); ?>
															</span>
														<?php endif; ?>
														<?php if ( $pending > 0 ) : ?>
															<span class="intufind-badge intufind-badge--warning" title="<?php esc_attr_e( 'Pending', 'intufind' ); ?>">
																<?php echo esc_html( $pending ); ?>
															</span>
														<?php endif; ?>
														<?php if ( $errors > 0 ) : ?>
															<span class="intufind-badge intufind-badge--error" title="<?php esc_attr_e( 'Errors', 'intufind' ); ?>">
																<?php echo esc_html( $errors ); ?>
															</span>
														<?php endif; ?>
														<?php if ( 0 === $synced && 0 === $pending && 0 === $errors ) : ?>
															<span class="intufind-badge intufind-badge--muted"><?php esc_html_e( 'Not synced', 'intufind' ); ?></span>
														<?php endif; ?>
													</div>
												<?php else : ?>
													<span class="intufind-text-muted">&mdash;</span>
												<?php endif; ?>
											</td>
											<td class="intufind-sync-table__progress">
												<?php if ( $is_enabled ) : ?>
													<div class="intufind-progress">
														<div class="intufind-progress__bar" style="width: <?php echo esc_attr( $percentage ); ?>%"></div>
													</div>
													<span class="intufind-progress__text"><?php echo esc_html( $percentage ); ?>%</span>
												<?php else : ?>
													<span class="intufind-text-muted">&mdash;</span>
												<?php endif; ?>
											</td>
											<td class="intufind-sync-table__last">
												<?php if ( $is_enabled && $last_sync ) : ?>
													<span title="<?php echo esc_attr( wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $last_sync ) ); ?>">
														<?php echo esc_html( $status_tracker->format_time_ago( $last_sync ) ); ?>
													</span>
												<?php else : ?>
													<span class="intufind-text-muted"><?php echo $is_enabled ? esc_html__( 'Never', 'intufind' ) : '&mdash;'; ?></span>
												<?php endif; ?>
											</td>
											<td class="intufind-sync-table__actions">
												<?php if ( $is_enabled ) : ?>
													<div class="intufind-button-group">
														<?php
														Intufind_Components::button(
															__( 'Sync', 'intufind' ),
															array(
																'type'    => 'button',
																'variant' => 'primary',
																'size'    => 'small',
																'class'   => 'intufind-sync-btn',
																'icon'    => 'update',
																'data'    => array( 'post-type' => $post_type ),
															)
														);
														?>
														<?php if ( $errors > 0 ) : ?>
															<?php
															Intufind_Components::button(
																__( 'Retry', 'intufind' ),
																array(
																	'type'    => 'button',
																	'variant' => 'ghost',
																	'size'    => 'small',
																	'class'   => 'intufind-retry-btn',
																	'data'    => array( 'post-type' => $post_type ),
																)
															);
															?>
														<?php endif; ?>
													</div>
												<?php endif; ?>
											</td>
										</tr>
									<?php endforeach; ?>
								</tbody>
							</table>
					</div>
					<?php
				},
			)
		);
		?>

		<!-- Taxonomy Sync Card -->
		<?php
		// Get taxonomy info from exclusions.
		$taxonomy_info     = $exclusions->get_taxonomy_info();
		$enabled_tax       = $exclusions->get_enabled_taxonomies();
		$api               = new Intufind_API();
		$sync_manager      = new Intufind_Sync( $api, null, $exclusions );
		$taxonomy_stats    = $sync_manager->get_taxonomy_sync_stats();
		$taxonomy_last     = $taxonomy_stats['last_sync'];
		$total_term_count  = 0;

		// Calculate total enabled term count.
		foreach ( $taxonomy_info as $tax => $info ) {
			if ( $info['enabled'] ) {
				$total_term_count += $info['count'];
			}
		}

		Intufind_Components::card(
			array(
				'title'  => __( 'Taxonomies', 'intufind' ),
				'render' => function () use ( $taxonomy_info, $taxonomy_last, $status_tracker, $total_term_count, $sync_manager, $taxonomies_synced, $taxonomies_enabled ) {
					?>
					<div class="intufind-sync-overview intufind-sync-overview--inline intufind-taxonomy-overview">
						<div class="intufind-stat-cards">
							<div class="intufind-stat-card intufind-stat-card--synced">
								<span class="intufind-stat-number"><?php echo esc_html( $taxonomies_synced ); ?></span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Synced', 'intufind' ); ?></span>
							</div>
							<div class="intufind-stat-card intufind-stat-card--pending">
								<span class="intufind-stat-number"><?php echo esc_html( $taxonomies_enabled - $taxonomies_synced ); ?></span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Pending', 'intufind' ); ?></span>
							</div>
							<div class="intufind-stat-card intufind-stat-card--errors">
								<span class="intufind-stat-number">0</span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Errors', 'intufind' ); ?></span>
							</div>
							<div class="intufind-stat-card intufind-stat-card--terms">
								<span class="intufind-stat-number"><?php echo esc_html( number_format_i18n( $total_term_count ) ); ?></span>
								<span class="intufind-stat-label"><?php esc_html_e( 'Total Terms', 'intufind' ); ?></span>
							</div>
						</div>
					</div>
					<p class="intufind-description">
						<?php esc_html_e( 'Enable taxonomies to sync for AI-powered filtering and search refinement.', 'intufind' ); ?>
					</p>
					<div class="intufind-sync-table-wrapper">
						<table class="intufind-sync-table widefat">
							<thead>
								<tr>
									<th class="intufind-sync-table__toggle"><?php esc_html_e( 'Enabled', 'intufind' ); ?></th>
									<th><?php esc_html_e( 'Taxonomy', 'intufind' ); ?></th>
									<th class="intufind-sync-table__status"><?php esc_html_e( 'Status', 'intufind' ); ?></th>
									<th class="intufind-sync-table__terms"><?php esc_html_e( 'Terms', 'intufind' ); ?></th>
									<th class="intufind-sync-table__type"><?php esc_html_e( 'Post Types', 'intufind' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ( $taxonomy_info as $taxonomy => $info ) : ?>
									<?php
									$is_enabled   = $info['enabled'];
									$row_class    = $is_enabled ? '' : 'intufind-sync-row--disabled';
									$sync_status  = $sync_manager->get_taxonomy_sync_status( $taxonomy );
									$post_labels  = $info['post_labels'];
									$post_count   = count( $post_labels );
									?>
									<tr data-taxonomy="<?php echo esc_attr( $taxonomy ); ?>" class="<?php echo esc_attr( $row_class ); ?>">
										<td class="intufind-sync-table__toggle">
											<label class="intufind-toggle">
												<input
													type="checkbox"
													class="intufind-toggle__input intufind-taxonomy-toggle"
													data-taxonomy="<?php echo esc_attr( $taxonomy ); ?>"
													<?php checked( $is_enabled ); ?>
												/>
												<span class="intufind-toggle__slider"></span>
											</label>
										</td>
										<td>
											<strong><?php echo esc_html( $info['label'] ); ?></strong>
											<span class="intufind-sync-table__slug"><?php echo esc_html( $taxonomy ); ?></span>
										</td>
										<td class="intufind-sync-table__status">
											<?php if ( $is_enabled ) : ?>
												<?php if ( $sync_status['synced'] ) : ?>
													<span class="intufind-badge intufind-badge--success" title="<?php echo esc_attr( $sync_status['last_sync_ago'] ); ?>">
														<?php esc_html_e( 'Synced', 'intufind' ); ?>
													</span>
												<?php else : ?>
													<span class="intufind-badge intufind-badge--muted">
														<?php esc_html_e( 'Not synced', 'intufind' ); ?>
													</span>
												<?php endif; ?>
											<?php else : ?>
												<span class="intufind-text-muted">&mdash;</span>
											<?php endif; ?>
										</td>
										<td class="intufind-sync-table__terms">
											<?php echo esc_html( number_format_i18n( $info['count'] ) ); ?>
										</td>
										<td class="intufind-sync-table__type">
											<?php if ( $post_count <= 2 ) : ?>
												<span class="intufind-text-muted">
													<?php echo esc_html( implode( ', ', $post_labels ) ); ?>
												</span>
											<?php else : ?>
												<span class="intufind-text-muted" title="<?php echo esc_attr( implode( ', ', $post_labels ) ); ?>" style="cursor: help; border-bottom: 1px dotted currentColor;">
													<?php
													/* translators: %d: number of post types */
													printf( esc_html__( '%d types', 'intufind' ), $post_count );
													?>
												</span>
											<?php endif; ?>
										</td>
									</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
					</div>

					<div class="intufind-taxonomy-footer">
						<div class="intufind-taxonomy-summary">
							<span class="intufind-text-muted">
								<?php
								printf(
									/* translators: %s: Total term count */
									esc_html__( 'Total enabled: %s terms', 'intufind' ),
									number_format_i18n( $total_term_count )
								);
								?>
							</span>
							<?php if ( $taxonomy_last ) : ?>
								<span class="intufind-text-muted intufind-taxonomy-last-sync">
									&bull;
									<?php
									printf(
										/* translators: %s: Time ago */
										esc_html__( 'Last synced: %s', 'intufind' ),
										esc_html( $status_tracker->format_time_ago( $taxonomy_last ) )
									);
									?>
								</span>
							<?php endif; ?>
						</div>
						<div class="intufind-taxonomy-actions">
							<?php
							Intufind_Components::button(
								__( 'Sync', 'intufind' ),
								array(
									'type'    => 'button',
									'variant' => 'primary',
									'id'      => 'intufind-sync-taxonomies-btn',
									'icon'    => 'update',
								)
							);
							?>
						</div>
					</div>
					<?php
				},
			)
		);
		?>

		<!-- Auto-Sync Settings Card -->
			<?php
			Intufind_Components::card(
				array(
					'title'       => __( 'Automatic Sync', 'intufind' ),
					'description' => __( 'Automatically sync content on a schedule to keep your AI search up to date.', 'intufind' ),
					'render'      => function () use ( $auto_sync_enabled, $sync_interval, $sync_hour, $next_sync, $status_tracker ) {
						?>
						<form id="intufind-auto-sync-form" method="post" action="">
							<?php wp_nonce_field( 'intufind_sync_settings', 'intufind_sync_nonce' ); ?>

							<div class="intufind-search-toggles">
								<!-- Enable Auto-Sync Toggle -->
								<div class="intufind-search-toggle">
									<label class="intufind-toggle-row">
										<span class="intufind-toggle-info">
											<span class="intufind-toggle-title">
												<span class="dashicons dashicons-update"></span>
												<?php esc_html_e( 'Enable Automatic Sync', 'intufind' ); ?>
											</span>
											<span class="intufind-toggle-description">
												<?php esc_html_e( 'Automatically sync all enabled content types on a schedule.', 'intufind' ); ?>
											</span>
										</span>
										<span class="intufind-toggle">
											<input
												type="checkbox"
												name="<?php echo esc_attr( Intufind_Sync::OPTION_AUTO_SYNC_ENABLED ); ?>"
												class="intufind-toggle__input intufind-sync-option"
												value="1"
												<?php checked( $auto_sync_enabled ); ?>
											/>
											<span class="intufind-toggle__slider"></span>
										</span>
									</label>
								</div>
							</div>

							<div class="intufind-form-fields intufind-auto-sync-fields" <?php echo $auto_sync_enabled ? '' : 'style="display: none;"'; ?>>
								<!-- Sync Interval -->
								<?php
								Intufind_Components::select_input(
									Intufind_Sync::OPTION_SYNC_INTERVAL,
									$sync_interval,
									array(
										'label'       => __( 'Sync Interval', 'intufind' ),
										'description' => __( 'How often to automatically sync content.', 'intufind' ),
										'options'     => array(
											21600  => __( 'Every 6 hours', 'intufind' ),
											43200  => __( 'Every 12 hours', 'intufind' ),
											86400  => __( 'Once daily', 'intufind' ),
											172800 => __( 'Every 2 days', 'intufind' ),
										),
										'class'       => 'intufind-sync-option',
									)
								);
								?>

								<!-- Sync Hour -->
								<?php
								$hour_options = array();
								for ( $h = 0; $h < 24; $h++ ) {
									$hour_options[ $h ] = sprintf( '%02d:00', $h );
								}
								Intufind_Components::select_input(
									Intufind_Sync::OPTION_SYNC_HOUR,
									$sync_hour,
									array(
										'label'       => __( 'Preferred Time', 'intufind' ),
										'description' => __( 'Preferred hour to start the sync (in your site timezone).', 'intufind' ),
										'options'     => $hour_options,
										'class'       => 'intufind-sync-option',
									)
								);
								?>

								<!-- Next Scheduled Sync -->
								<?php if ( $next_sync ) : ?>
								<div class="intufind-field">
									<span class="intufind-label"><?php esc_html_e( 'Next Scheduled Sync', 'intufind' ); ?></span>
									<p class="intufind-next-sync-time">
										<span class="dashicons dashicons-clock"></span>
										<?php
										echo esc_html(
											sprintf(
												/* translators: 1: Date, 2: Time ago */
												__( '%1$s (%2$s)', 'intufind' ),
												wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $next_sync ),
												$status_tracker->format_time_ago( $next_sync, true )
											)
										);
										?>
									</p>
								</div>
								<?php endif; ?>
							</div>
						</form>
						<?php
					},
				)
			);
			?>

			<!-- Bulk Actions Card -->
			<?php
			$has_enabled_types = ! empty( $enabled_post_types );
			if ( $has_enabled_types ) :
				Intufind_Components::card(
					array(
						'title'  => __( 'Bulk Actions', 'intufind' ),
						'render' => function () {
							?>
							<p class="intufind-description" style="margin: 0 0 12px 0;">
								<?php esc_html_e( 'Sync all enabled content types at once. This may take a while for large sites.', 'intufind' ); ?>
							</p>
							<div class="intufind-bulk-actions">
								<?php
								Intufind_Components::button(
									__( 'Sync All', 'intufind' ),
									array(
										'type'    => 'button',
										'variant' => 'primary',
										'id'      => 'intufind-sync-all',
										'icon'    => 'update',
									)
								);
								?>
								<div id="intufind-sync-progress" class="intufind-sync-progress" style="display: none;">
									<div class="intufind-sync-progress__spinner"></div>
									<span class="intufind-sync-progress__text"><?php esc_html_e( 'Syncing...', 'intufind' ); ?></span>
								</div>
							</div>
							<div id="intufind-sync-log" class="intufind-sync-log" style="display: none;">
								<h4><?php esc_html_e( 'Sync Log', 'intufind' ); ?></h4>
								<ul class="intufind-sync-log__list"></ul>
							</div>
							<?php
						},
					)
				);
			endif;
			?>
		</div>
	<?php endif; ?>
</div>
