<?php
/**
 * API client for cloud communication.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Intufind API client.
 *
 * Handles all communication with the Intufind cloud API using
 * WordPress HTTP API functions for WordPress.org compliance.
 */
class Intufind_API {

	/**
	 * API base URL.
	 *
	 * @var string
	 */
	private $api_url;

	/**
	 * Request timeout in seconds.
	 *
	 * @var int
	 */
	private $timeout = 30;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->api_url = INTUFIND_API_URL;
	}

	/**
	 * Get the API key from options.
	 *
	 * @return string|null
	 */
	public function get_api_key() {
		$key = get_option( INTUFIND_OPTION_API_KEY, '' );
		return ! empty( $key ) ? $key : null;
	}

	/**
	 * Get the workspace ID from options.
	 *
	 * @return string|null
	 */
	public function get_workspace_id() {
		$id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		return ! empty( $id ) ? $id : null;
	}

	/**
	 * Make an authenticated API request.
	 *
	 * @param string $method   HTTP method (GET, POST, PUT, DELETE).
	 * @param string $endpoint API endpoint (without base URL).
	 * @param array  $data     Request body data (for POST/PUT).
	 * @param array  $args     Additional wp_remote_request args.
	 * @return array|WP_Error Response data or WP_Error on failure.
	 */
	public function request( $method, $endpoint, $data = array(), $args = array() ) {
		$api_key = $this->get_api_key();

		if ( empty( $api_key ) ) {
			return new WP_Error( 'no_api_key', __( 'API key not configured.', 'intufind' ) );
		}

		$url = trailingslashit( $this->api_url ) . ltrim( $endpoint, '/' );

		$headers = array(
			'Authorization' => 'Bearer ' . $api_key,
			'Content-Type'  => 'application/json',
			'Accept'        => 'application/json',
		);

		// Add workspace ID header if available.
		$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		if ( ! empty( $workspace_id ) ) {
			$headers['X-Workspace-ID'] = $workspace_id;
		}

		$default_args = array(
			'method'  => strtoupper( $method ),
			'timeout' => $this->timeout,
			'headers' => $headers,
		);

		// Add body for POST/PUT requests.
		if ( in_array( strtoupper( $method ), array( 'POST', 'PUT', 'PATCH' ), true ) && ! empty( $data ) ) {
			$default_args['body'] = wp_json_encode( $data );
		}

		// Merge with additional args.
		$request_args = wp_parse_args( $args, $default_args );

		// Make the request.
		$response = wp_remote_request( $url, $request_args );

		// Check for WP_Error.
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse response.
		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );
		$data        = json_decode( $body, true );

		// Check for HTTP errors.
		if ( $status_code >= 400 ) {
			$error_message = isset( $data['error'] ) ? $data['error'] : __( 'API request failed.', 'intufind' );
			return new WP_Error( 'api_error', $error_message, array( 'status' => $status_code ) );
		}

		return $data;
	}

	/**
	 * Validate an API key and get account info.
	 *
	 * @param string $api_key The API key to validate.
	 * @return array|WP_Error Account info or WP_Error on failure.
	 */
	public function validate_api_key( $api_key ) {
		$url = trailingslashit( $this->api_url ) . 'tenant/status';

		$response = wp_remote_post(
			$url,
			array(
				'timeout' => $this->timeout,
				'headers' => array(
					'Authorization' => 'Bearer ' . $api_key,
					'Content-Type'  => 'application/json',
					'Accept'        => 'application/json',
				),
				'body'    => '{}',
			)
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );
		$data        = json_decode( $body, true );

		if ( 401 === $status_code || 403 === $status_code ) {
			return new WP_Error( 'invalid_api_key', __( 'Invalid API key.', 'intufind' ) );
		}

		if ( $status_code >= 400 ) {
			$error_message = isset( $data['error'] ) ? $data['error'] : __( 'Failed to validate API key.', 'intufind' );
			return new WP_Error( 'validation_failed', $error_message );
		}

		return $data;
	}

	/**
	 * List workspaces for the authenticated tenant.
	 *
	 * @return array|WP_Error Array of workspaces or WP_Error.
	 */
	public function list_workspaces() {
		return $this->request( 'GET', 'tenant/workspaces' );
	}

	/**
	 * Create a new workspace.
	 *
	 * @param string $workspace_id   The workspace ID (slug).
	 * @param string $name           Display name for the workspace.
	 * @param string $description    Optional description.
	 * @return array|WP_Error Created workspace data or WP_Error.
	 */
	public function create_workspace( $workspace_id, $name = '', $description = '' ) {
		$data = array(
			'workspaceId' => $workspace_id,
			'name'        => ! empty( $name ) ? $name : $workspace_id,
			'description' => $description,
			'source'      => 'wordpress_plugin',
			'siteUrl'     => home_url(),
		);

		return $this->request( 'POST', 'tenant/workspaces', $data );
	}

	/**
	 * Check if a workspace exists.
	 *
	 * @param string $workspace_id The workspace ID to check.
	 * @return bool|WP_Error True if exists, false if not, WP_Error on failure.
	 */
	public function workspace_exists( $workspace_id ) {
		$workspaces = $this->list_workspaces();

		if ( is_wp_error( $workspaces ) ) {
			return $workspaces;
		}

		// API returns { data: [...] } where data is the array of workspaces.
		$workspace_list = $workspaces['data'] ?? array();

		// Handle nested workspaces format for compatibility.
		if ( isset( $workspace_list['workspaces'] ) ) {
			$workspace_list = $workspace_list['workspaces'];
		}

		foreach ( $workspace_list as $workspace ) {
			if ( isset( $workspace['workspaceId'] ) && $workspace['workspaceId'] === $workspace_id ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Verify the current connection is still valid.
	 *
	 * Checks that the API key is valid and the workspace still exists.
	 * Results are cached in a transient for performance.
	 *
	 * @param bool $force_check Skip the cache and force a fresh check.
	 * @return array{valid: bool, error?: string} Verification result.
	 */
	public function verify_connection( $force_check = false ) {
		$cache_key = 'intufind_connection_verified';

		// Check cache first (unless forced).
		if ( ! $force_check ) {
			$cached = get_transient( $cache_key );
			if ( false !== $cached ) {
				return $cached;
			}
		}

		$api_key      = $this->get_api_key();
		$workspace_id = $this->get_workspace_id();

		// No credentials stored.
		if ( empty( $api_key ) || empty( $workspace_id ) ) {
			$result = array(
				'valid' => false,
				'error' => __( 'Not configured.', 'intufind' ),
			);
			// Don't cache missing credentials - user might be setting up.
			return $result;
		}

		// Validate API key.
		$validation = $this->validate_api_key( $api_key );
		if ( is_wp_error( $validation ) ) {
			$result = array(
				'valid' => false,
				'error' => $validation->get_error_message(),
			);
			// Cache failures briefly (2 minutes) to avoid hammering API.
			set_transient( $cache_key, $result, 2 * MINUTE_IN_SECONDS );
			return $result;
		}

		// Check workspace still exists.
		$workspace_exists = $this->workspace_exists( $workspace_id );
		if ( is_wp_error( $workspace_exists ) ) {
			$result = array(
				'valid' => false,
				'error' => $workspace_exists->get_error_message(),
			);
			set_transient( $cache_key, $result, 2 * MINUTE_IN_SECONDS );
			return $result;
		}

		if ( ! $workspace_exists ) {
			$result = array(
				'valid' => false,
				'error' => __( 'Workspace no longer exists.', 'intufind' ),
			);
			set_transient( $cache_key, $result, 2 * MINUTE_IN_SECONDS );
			return $result;
		}

		// All good!
		$result = array( 'valid' => true );
		// Cache successful validation for 10 minutes.
		set_transient( $cache_key, $result, 10 * MINUTE_IN_SECONDS );

		return $result;
	}

	/**
	 * Clear the connection verification cache.
	 *
	 * Call this when connection settings change.
	 *
	 * @return void
	 */
	public static function clear_verification_cache() {
		delete_transient( 'intufind_connection_verified' );
	}

	/**
	 * Upsert posts to the cloud.
	 *
	 * @param array $posts Array of post data.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function upsert_posts( $posts ) {
		return $this->request( 'POST', 'posts/bulk', array( 'entities' => $posts ) );
	}

	/**
	 * Delete posts from the cloud.
	 *
	 * @param array $post_ids Array of post IDs to delete.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function delete_posts( $post_ids ) {
		return $this->request( 'DELETE', 'posts/bulk', array( 'ids' => $post_ids ) );
	}

	/**
	 * Upsert products to the cloud.
	 *
	 * @param array $products Array of product data.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function upsert_products( $products ) {
		return $this->request( 'POST', 'products/bulk', array( 'entities' => $products ) );
	}

	/**
	 * Delete products from the cloud.
	 *
	 * @param array $product_ids Array of product IDs to delete.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function delete_products( $product_ids ) {
		return $this->request( 'DELETE', 'products/bulk', array( 'ids' => $product_ids ) );
	}

	/**
	 * Get workspace settings including custom domains.
	 *
	 * @return array|WP_Error Settings data or WP_Error.
	 */
	public function get_workspace_settings() {
		return $this->request( 'GET', 'workspace/settings' );
	}

	/**
	 * Update workspace settings.
	 *
	 * @param array $settings Settings to update.
	 * @return array|WP_Error Updated settings or WP_Error.
	 */
	public function update_workspace_settings( $settings ) {
		return $this->request( 'PUT', 'workspace/settings', $settings );
	}

	/**
	 * Register a custom domain for MCP integration.
	 *
	 * Creates or updates a custom domain configuration in the workspace.
	 * Used by WordPress to register its MCP server endpoints with the cloud.
	 *
	 * @param array $domain_config Custom domain configuration.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function register_custom_domain( $domain_config ) {
		return $this->request( 'POST', 'workspace/settings/domains', $domain_config );
	}

	/**
	 * Delete a custom domain.
	 *
	 * @param string $domain_id The domain ID to delete.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function delete_custom_domain( $domain_id ) {
		return $this->request( 'DELETE', 'workspace/settings/domains/' . $domain_id );
	}

	/**
	 * Register the WordPress MCP server with the cloud.
	 *
	 * This is called automatically when:
	 * - Plugin is first connected and WooCommerce is active
	 * - WooCommerce is activated on a connected site
	 *
	 * The plugin's API key is passed as static auth, allowing the cloud
	 * to authenticate when calling the WordPress MCP endpoint.
	 *
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function register_orders_domain() {
		// Get the MCP endpoint URL.
		$mcp_url = rest_url( 'intufind/v1/mcp/domain/orders' );

		// Get the API key for MCP authentication.
		$api_key = $this->get_api_key();
		if ( empty( $api_key ) ) {
			return new WP_Error( 'no_api_key', __( 'API key not configured.', 'intufind' ) );
		}

		// Get site name for display.
		$site_name = get_bloginfo( 'name' );
		if ( empty( $site_name ) ) {
			$site_name = wp_parse_url( home_url(), PHP_URL_HOST );
		}

		$domain_config = array(
			'name'             => __( 'Order Support', 'intufind' ),
			'classifierPrompt' => __( "User's own purchase/order status, order history, delivery tracking, or order-related questions", 'intufind' ),
			'agentPrompt'      => "You are a customer service assistant that helps customers with their orders.\n\n" .
								"Use the available MCP tools to look up order information.\n" .
								"Always speak directly to the customer using \"your orders\", \"your purchase\".\n" .
								"Never invent order details - only use data from the tools.\n\n" .
								"IMPORTANT: For security, orders are user-scoped:\n" .
								"- For logged-in users: Use get_my_orders to see their orders.\n" .
								"- For guests: Use lookup_order with their order number AND email to verify ownership.\n" .
								"- Never assume access to orders without verification.\n" .
								"If the customer needs to take action (cancel, modify), explain the process clearly.",
			'progressMessage'  => __( 'Checking your orders...', 'intufind' ),
			'enabled'          => true,
			'mcpServers'       => array(
				array(
					'name'         => $site_name . ' ' . __( 'Orders', 'intufind' ),
					'url'          => $mcp_url,
					'requiresAuth' => true, // Forward user JWT for user context.
					'staticAuth'   => array(
						'type'  => 'bearer',
						'value' => $api_key, // API key for server-to-server auth.
					),
				),
			),
		);

		return $this->register_custom_domain( $domain_config );
	}

	/**
	 * Unregister the orders domain from the cloud.
	 *
	 * Called when WooCommerce is deactivated.
	 *
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function unregister_orders_domain() {
		return $this->delete_custom_domain( 'order-support' );
	}

	/**
	 * Get document statistics from the cloud.
	 *
	 * Returns document counts and breakdown by type from the cloud.
	 * This reflects the actual state of indexed documents.
	 *
	 * @return array|WP_Error Document stats or WP_Error.
	 */
	public function get_document_stats() {
		$workspace_id = $this->get_workspace_id();
		$endpoint     = 'tenant/documents';

		// Include workspaceId for workspace-scoped counts (consistent with sync operations).
		if ( ! empty( $workspace_id ) ) {
			$endpoint .= '?workspaceId=' . rawurlencode( $workspace_id );
		}

		return $this->request( 'GET', $endpoint );
	}

	/**
	 * Get cached document statistics from the cloud.
	 *
	 * Fetches stats with caching to avoid repeated API calls on page loads.
	 * Cache is cleared when sync operations complete.
	 *
	 * @param bool $force_refresh Force a fresh fetch from the cloud.
	 * @return array|WP_Error Document stats or WP_Error.
	 */
	public function get_cached_document_stats( $force_refresh = false ) {
		$cache_key = 'intufind_cloud_stats';

		// Check cache first (unless forced).
		if ( ! $force_refresh ) {
			$cached = get_transient( $cache_key );
			if ( false !== $cached ) {
				return $cached;
			}
		}

		$result = $this->get_document_stats();

		if ( is_wp_error( $result ) ) {
			// Cache errors for 2 minutes to avoid hammering API.
			set_transient( $cache_key, $result, 2 * MINUTE_IN_SECONDS );
			return $result;
		}

		// Cache success for 5 minutes.
		set_transient( $cache_key, $result['data'], 5 * MINUTE_IN_SECONDS );

		return $result['data'];
	}

	/**
	 * Clear the cached cloud stats.
	 *
	 * Should be called after sync operations to refresh counts.
	 *
	 * @return void
	 */
	public static function clear_cloud_stats_cache() {
		delete_transient( 'intufind_cloud_stats' );
	}

	/**
	 * Search products using semantic search.
	 *
	 * @param string $query   Search query text.
	 * @param int    $limit   Maximum results to return.
	 * @param array  $filters Optional filters (categories, price range, etc.).
	 * @return array|WP_Error Search results or WP_Error.
	 */
	public function search_products( $query, $limit = 10, $filters = array() ) {
		$data = array(
			'text'          => $query,
			'limit'         => $limit,
			'searchContext' => 'native', // Mark as native WordPress integration for analytics.
		);

		if ( ! empty( $filters ) ) {
			$data['filters'] = $filters;
		}

		return $this->request( 'POST', 'products/search', $data );
	}

	/**
	 * Search posts using semantic search.
	 *
	 * @param string $query     Search query text.
	 * @param int    $limit     Maximum results to return.
	 * @param array  $filters   Optional filters (post_type, status, etc.).
	 * @param array  $post_types Optional post types to search.
	 * @return array|WP_Error Search results or WP_Error.
	 */
	public function search_posts( $query, $limit = 10, $filters = array(), $post_types = array() ) {
		$data = array(
			'text'          => $query,
			'limit'         => $limit,
			'searchContext' => 'native', // Mark as native WordPress integration for analytics.
		);

		// Add post type filter if specified.
		if ( ! empty( $post_types ) ) {
			$filters['postType'] = $post_types;
		}

		if ( ! empty( $filters ) ) {
			$data['filters'] = $filters;
		}

		return $this->request( 'POST', 'posts/search', $data );
	}

	/**
	 * Get product recommendations.
	 *
	 * @param int   $product_id      The product ID to get recommendations for.
	 * @param int   $limit           Maximum recommendations to return.
	 * @param array $exclude_ids     Product IDs to exclude from results.
	 * @param int   $user_id         Optional user ID for personalization.
	 * @return array|WP_Error Recommendations or WP_Error.
	 */
	public function get_recommendations( $product_id, $limit = 4, $exclude_ids = array(), $user_id = 0 ) {
		$data = array(
			'productId'          => (string) $product_id,
			'maxRecommendations' => $limit,
		);

		if ( ! empty( $exclude_ids ) ) {
			$data['excludeProductIds'] = array_map( 'strval', $exclude_ids );
		}

		if ( $user_id > 0 ) {
			$data['userId'] = (string) $user_id;
		}

		return $this->request( 'POST', 'recommendations/products', $data );
	}

	// =========================================================================
	// Taxonomy Operations
	// =========================================================================

	/**
	 * Upsert taxonomies to the cloud.
	 *
	 * @param array $taxonomies Array of taxonomy term data.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function upsert_taxonomies( $taxonomies ) {
		// Taxonomy bulk upsert expects a direct array, not wrapped in an object.
		return $this->request( 'POST', 'taxonomies/bulk', $taxonomies );
	}

	/**
	 * Delete taxonomies from the cloud.
	 *
	 * @param array $taxonomy_ids Array of taxonomy term IDs to delete.
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function delete_taxonomies( $taxonomy_ids ) {
		return $this->request( 'DELETE', 'taxonomies/bulk', array( 'ids' => $taxonomy_ids ) );
	}

	/**
	 * Get taxonomy term IDs from the cloud.
	 *
	 * @param string $taxonomy_name Optional taxonomy name filter.
	 * @param int    $limit         Maximum results.
	 * @param int    $offset        Offset for pagination.
	 * @return array|WP_Error Response with IDs or WP_Error.
	 */
	public function get_taxonomy_ids( $taxonomy_name = '', $limit = 10000, $offset = 0 ) {
		$params = array(
			'limit'  => $limit,
			'offset' => $offset,
		);

		if ( ! empty( $taxonomy_name ) ) {
			$params['taxonomyName'] = $taxonomy_name;
		}

		$query = http_build_query( $params );

		return $this->request( 'GET', 'taxonomies/ids?' . $query );
	}

	/**
	 * Deprovision the workspace and purge all data from the cloud.
	 *
	 * This is called when the user disconnects from WordPress.
	 * It notifies the cloud to delete all indexed documents and workspace data.
	 *
	 * @return array|WP_Error Response data or WP_Error.
	 */
	public function deprovision() {
		$workspace_id = $this->get_workspace_id();

		if ( empty( $workspace_id ) ) {
			return new WP_Error( 'no_workspace', __( 'No workspace configured.', 'intufind' ) );
		}

		return $this->request(
			'POST',
			'provision/deprovision',
			array(
				'siteUrl'   => $workspace_id,
				'purgeData' => true,
			)
		);
	}
}
