<?php
/**
 * Chat Widget handler.
 *
 * Injects the Intufind chat widget on the frontend.
 * Widget configuration is managed via the Intufind Dashboard.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Chat Widget class.
 *
 * Handles loading the chat widget loader script on the frontend.
 */
class Intufind_Chat_Widget {

	/**
	 * Option names.
	 */
	const OPTION_ENABLE_CHAT = 'intufind_enable_chat_widget';

	/**
	 * CDN base URL for widget assets.
	 */
	const CDN_URL = 'https://cdn.intufind.com';

	/**
	 * API instance.
	 *
	 * @var Intufind_API
	 */
	private $api;

	/**
	 * Constructor.
	 *
	 * @param Intufind_API $api API client instance.
	 */
	public function __construct( Intufind_API $api ) {
		$this->api = $api;
	}

	/**
	 * Initialize chat widget functionality.
	 *
	 * @return void
	 */
	public function init() {
		// Only load on frontend, not admin.
		if ( is_admin() ) {
			return;
		}

		// Check if chat is enabled and we're connected.
		if ( ! $this->is_enabled() || ! $this->is_connected() ) {
			return;
		}

		// Enqueue the chat loader script.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_chat_loader' ) );
	}

	/**
	 * Check if chat widget is enabled.
	 *
	 * @return bool
	 */
	public function is_enabled() {
		return (bool) get_option( self::OPTION_ENABLE_CHAT, true );
	}

	/**
	 * Check if plugin is connected.
	 *
	 * @return bool
	 */
	private function is_connected() {
		return (bool) get_option( INTUFIND_OPTION_CONNECTED, false );
	}

	/**
	 * Get the CDN base URL.
	 *
	 * Can be overridden via constant for local development.
	 *
	 * @return string
	 */
	private function get_cdn_url() {
		if ( defined( 'INTUFIND_CHAT_WIDGET_URL' ) ) {
			return rtrim( INTUFIND_CHAT_WIDGET_URL, '/' );
		}
		return self::CDN_URL;
	}

	/**
	 * Enqueue the chat loader script.
	 *
	 * @return void
	 */
	public function enqueue_chat_loader() {
		$cdn_url    = $this->get_cdn_url();
		$loader_url = $cdn_url . '/chat-loader.js';

		// Enqueue the loader script.
		wp_enqueue_script(
			'intufind-chat-loader',
			$loader_url,
			array(),
			null, // No version - CDN manages caching.
			true  // Load in footer.
		);

		// Build the widget configuration.
		$config = $this->build_config();

		// Inline script to initialize the widget.
		$inline_script = sprintf(
			'window.intufindConfig = %s;',
			wp_json_encode( $config )
		);

		wp_add_inline_script( 'intufind-chat-loader', $inline_script, 'before' );
	}

	/**
	 * Build the widget configuration.
	 *
	 * @return array
	 */
	private function build_config() {
		$config = array(
			'publishableKey'    => get_option( INTUFIND_OPTION_PUBLISHABLE_KEY, '' ),
			'workspaceId'       => get_option( INTUFIND_OPTION_WORKSPACE_ID, '' ),
			'apiUrl'            => INTUFIND_API_URL,
			'widgetUrl'         => $this->get_cdn_url() . '/chat-widget.html',
		);

		// Add page context.
		$config['pageContext'] = $this->get_page_context();

		// Add user JWT if available (for authenticated MCP operations).
		$user_jwt = $this->get_user_jwt();
		if ( $user_jwt ) {
			$config['userJwt'] = $user_jwt;
		}

		// Debug mode.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			$config['debugMode'] = true;
		}

		/**
		 * Filter the chat widget configuration.
		 *
		 * @param array $config Widget configuration.
		 */
		return apply_filters( 'intufind_chat_widget_config', $config );
	}

	/**
	 * Get page context for the widget.
	 *
	 * @return array
	 */
	private function get_page_context() {
		$context = array(
			'currentPageUrl'   => is_singular() ? get_permalink() : home_url( add_query_arg( null, null ) ),
			'currentPageTitle' => is_singular() ? get_the_title() : wp_get_document_title(),
			'currentProductId' => null,
		);

		// Add product context if on a WooCommerce product page.
		if ( function_exists( 'is_product' ) && is_product() ) {
			global $product;
			if ( $product instanceof WC_Product ) {
				$context['currentProductId'] = $product->get_id();
			} elseif ( function_exists( 'wc_get_product' ) ) {
				$wc_product = wc_get_product( get_the_ID() );
				if ( $wc_product ) {
					$context['currentProductId'] = $wc_product->get_id();
				}
			}
		}

		return $context;
	}

	/**
	 * Get user JWT for authenticated operations.
	 *
	 * @return string|null
	 */
	private function get_user_jwt() {
		if ( ! is_user_logged_in() ) {
			return null;
		}

		// Check if MCP is registered (needed for authenticated operations).
		if ( ! get_option( INTUFIND_OPTION_MCP_REGISTERED, false ) ) {
			return null;
		}

		// Use MCP service to generate token.
		$mcp = new Intufind_MCP( $this->api );
		$token_data = $mcp->get_user_token_for_widget();

		return $token_data ? $token_data['token'] : null;
	}

	/**
	 * Check if chat widget is available.
	 *
	 * @return bool
	 */
	public static function is_available() {
		$publishable_key = get_option( INTUFIND_OPTION_PUBLISHABLE_KEY, '' );
		$workspace_id    = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		$connected       = get_option( INTUFIND_OPTION_CONNECTED, false );

		return $connected && ! empty( $publishable_key ) && ! empty( $workspace_id );
	}
}
