<?php
/**
 * Document exclusion logic.
 *
 * Centralized rules for determining which content should be
 * excluded from cloud synchronization.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Centralized exclusion logic for document sync.
 *
 * Handles all rules for excluding content from sync:
 * - System post types
 * - Page builder templates
 * - WooCommerce functional pages
 * - Password protected content
 * - Non-viewable content
 * - System taxonomies
 */
class Intufind_Exclusions {

	/**
	 * Option name for enabled taxonomies.
	 *
	 * @var string
	 */
	const OPTION_SYNC_TAXONOMIES = 'intufind_sync_taxonomies';

	/**
	 * WooCommerce product types to exclude (variations handled by parent).
	 *
	 * @var array
	 */
	const EXCLUDED_PRODUCT_TYPES = array(
		'variation',
	);

	/**
	 * System taxonomies that should never be synced.
	 *
	 * @var array
	 */
	const EXCLUDED_SYSTEM_TAXONOMIES = array(
		// WordPress system taxonomies.
		'nav_menu',
		'link_category',
		'post_format',
		'wp_theme',
		'wp_template_part_area',
		'wp_pattern_category',
		'wp_block',
		'wp_navigation',
		'wp_global_styles',

		// Internal/system taxonomies.
		'oembed_cache',
		'customize_changeset',
		'user_request',

		// Background jobs.
		'scheduled-action',
		'action-group',

		// Page builder taxonomies.
		'elementor_library_type',
		'elementor_library_category',
		'elementor_font_type',
		'fl-builder-template-category',
		'et_tb_item_type',
	);

	/**
	 * System post types that should never be synced.
	 *
	 * @var array
	 */
	const EXCLUDED_SYSTEM_POST_TYPES = array(
		// WordPress system types.
		'attachment',
		'revision',
		'nav_menu_item',
		'custom_css',
		'customize_changeset',
		'oembed_cache',
		'user_request',
		'wp_block',
		'wp_template',
		'wp_template_part',
		'wp_navigation',
		'wp_global_styles',

		// WooCommerce sensitive data.
		'shop_order',
		'shop_coupon',
		'shop_order_refund',

		// Background jobs.
		'scheduled-action',

		// ACF system types.
		'acf-field',
		'acf-field-group',
	);

	/**
	 * Page builder template post types to exclude.
	 *
	 * @var array
	 */
	const PAGE_BUILDER_POST_TYPES = array(
		// Elementor.
		'elementor_library',
		'e-floating-buttons',

		// Beaver Builder.
		'fl-builder-template',
		'fl-theme-layout',

		// Divi.
		'et_pb_layout',
		'et_header_layout',
		'et_footer_layout',
		'et_body_layout',

		// Oxygen.
		'ct_template',
		'oxy_user_library',

		// WPBakery.
		'vc_grid_item',
		'templatera',

		// Other builders.
		'brizy_template',
		'thrive_template',
		'fusion_template',
		'kadence_template',
	);

	/**
	 * WooCommerce page types to exclude.
	 *
	 * @var array
	 */
	const WOO_PAGE_TYPES = array(
		'cart',
		'checkout',
		'myaccount',
	);

	/**
	 * Slugs that indicate transactional/system pages.
	 *
	 * @var array
	 */
	const EXCLUDED_SLUGS = array(
		'cart',
		'checkout',
		'my-account',
		'account',
		'order-received',
		'sitemap',
		'site-map',
		'404',
		'search',
		'sample-page',
	);

	/**
	 * Titles that indicate transactional/system pages (lowercase).
	 *
	 * @var array
	 */
	const EXCLUDED_TITLES = array(
		'cart',
		'checkout',
		'my account',
		'sitemap',
		'sample page',
	);

	/**
	 * Regex pattern for sitemap-related content.
	 *
	 * @var string
	 */
	const SITEMAP_PATTERN = '/^(.*[-_\s]?sitemap[-_\s]?.*|.*site[-_\s]?map.*)$/i';

	/**
	 * Cache of excluded post type names.
	 *
	 * @var array|null
	 */
	private $excluded_post_types_cache = null;

	/**
	 * Get all available post types for sync configuration.
	 *
	 * Returns post types that could potentially be synced,
	 * excluding system and page builder types.
	 *
	 * @return array Array of post type name => label.
	 */
	public function get_available_post_types() {
		$all_types = get_post_types(
			array(
				'public' => true,
			),
			'objects'
		);

		$excluded = $this->get_all_excluded_post_types();
		$available = array();

		foreach ( $all_types as $post_type ) {
			// Skip if in excluded list.
			if ( in_array( $post_type->name, $excluded, true ) ) {
				continue;
			}

			// Must have title and either editor support or ACF fields.
			if ( ! $this->is_content_capable( $post_type->name ) ) {
				continue;
			}

			$available[ $post_type->name ] = $post_type->label;
		}

		// Always include core types if they exist.
		$core_types = array( 'post', 'page' );
		foreach ( $core_types as $type ) {
			if ( post_type_exists( $type ) && ! isset( $available[ $type ] ) ) {
				$type_obj             = get_post_type_object( $type );
				$available[ $type ] = $type_obj ? $type_obj->label : ucfirst( $type );
			}
		}

		// Include product if WooCommerce active.
		if ( class_exists( 'WooCommerce' ) && ! isset( $available['product'] ) ) {
			$available['product'] = __( 'Products', 'intufind' );
		}

		return $available;
	}

	/**
	 * Get the enabled post types for sync.
	 *
	 * @return array Array of enabled post type names.
	 */
	public function get_enabled_post_types() {
		$saved = get_option( INTUFIND_OPTION_SYNC_POST_TYPES, array() );

		// Default to post, page, and product.
		if ( empty( $saved ) ) {
			$defaults = array( 'post', 'page' );
			if ( class_exists( 'WooCommerce' ) ) {
				$defaults[] = 'product';
			}
			return $defaults;
		}

		// If saved as associative array (old format), convert.
		if ( isset( $saved[ array_key_first( $saved ) ] ) && is_bool( $saved[ array_key_first( $saved ) ] ) ) {
			return array_keys( array_filter( $saved ) );
		}

		return (array) $saved;
	}

	/**
	 * Check if a post type is enabled for sync.
	 *
	 * @param string $post_type Post type name.
	 * @return bool
	 */
	public function is_post_type_enabled( $post_type ) {
		$enabled = $this->get_enabled_post_types();
		return in_array( $post_type, $enabled, true );
	}

	/**
	 * Get all excluded post types (system + page builder).
	 *
	 * @return array Array of post type names to exclude.
	 */
	public function get_all_excluded_post_types() {
		if ( null !== $this->excluded_post_types_cache ) {
			return $this->excluded_post_types_cache;
		}

		$excluded = array_merge(
			self::EXCLUDED_SYSTEM_POST_TYPES,
			self::PAGE_BUILDER_POST_TYPES,
			$this->detect_active_page_builder_types()
		);

		$this->excluded_post_types_cache = array_unique( $excluded );

		return $this->excluded_post_types_cache;
	}

	/**
	 * Detect page builder post types that actually exist.
	 *
	 * @return array Active page builder post types.
	 */
	private function detect_active_page_builder_types() {
		$detected = array();

		// Only include types that actually exist on this install.
		foreach ( self::PAGE_BUILDER_POST_TYPES as $type ) {
			if ( post_type_exists( $type ) ) {
				$detected[] = $type;
			}
		}

		return $detected;
	}

	/**
	 * Check if a post type supports meaningful content.
	 *
	 * @param string $post_type Post type name.
	 * @return bool
	 */
	private function is_content_capable( $post_type ) {
		// Products always pass (handled by WooCommerce provider).
		if ( 'product' === $post_type ) {
			return true;
		}

		$type_obj = get_post_type_object( $post_type );
		if ( ! $type_obj || ! is_post_type_viewable( $type_obj ) ) {
			return false;
		}

		$has_title  = post_type_supports( $post_type, 'title' );
		$has_editor = post_type_supports( $post_type, 'editor' );
		$has_acf    = $this->has_acf_fields( $post_type );

		return $has_title && ( $has_editor || $has_acf );
	}

	/**
	 * Check if a post type has ACF field groups.
	 *
	 * @param string $post_type Post type name.
	 * @return bool
	 */
	private function has_acf_fields( $post_type ) {
		if ( ! function_exists( 'acf_get_field_groups' ) ) {
			return false;
		}

		$groups = acf_get_field_groups( array( 'post_type' => $post_type ) );
		return ! empty( $groups );
	}

	/**
	 * Check if a specific post should be excluded from sync.
	 *
	 * This is the main entry point for checking individual posts.
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $post_type Post type.
	 * @return bool True if excluded, false if should sync.
	 */
	public function should_exclude( $post_id, $post_type ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return true;
		}

		// Check post type enabled.
		if ( ! $this->is_post_type_enabled( $post_type ) ) {
			return true;
		}

		// Check manual exclusion meta (via list table column toggle).
		if ( Intufind_List_Columns::is_excluded( $post_id ) ) {
			return true;
		}

		// Check post status.
		if ( ! $this->is_syncable_status( $post->post_status, $post_id ) ) {
			return true;
		}

		// Product-specific checks.
		if ( 'product' === $post_type && class_exists( 'WooCommerce' ) ) {
			if ( $this->should_exclude_product( $post_id ) ) {
				return true;
			}
		}

		// Check various exclusion rules.
		if ( $this->is_system_page( $post_id, $post ) ) {
			return true;
		}

		if ( $this->is_password_protected( $post ) ) {
			return true;
		}

		if ( ! $this->has_accessible_url( $post ) ) {
			return true;
		}

		// Allow filtering.
		return apply_filters( 'intufind_should_exclude_post', false, $post_id, $post_type, $post );
	}

	/**
	 * Check if post status allows sync.
	 *
	 * @param string $status  Post status.
	 * @param int    $post_id Post ID for private post check.
	 * @return bool
	 */
	private function is_syncable_status( $status, $post_id ) {
		// Published is always syncable.
		if ( 'publish' === $status ) {
			return true;
		}

		// Private posts can be synced (they'll be marked as non-searchable/chatbot only).
		if ( 'private' === $status ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if a WooCommerce product should be excluded.
	 *
	 * @param int $product_id Product ID.
	 * @return bool
	 */
	private function should_exclude_product( $product_id ) {
		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return true;
		}

		// Exclude variations.
		if ( $product->is_type( 'variation' ) ) {
			return true;
		}

		// Exclude hidden products.
		if ( 'hidden' === $product->get_catalog_visibility() ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if post is a system/transactional page.
	 *
	 * @param int      $post_id Post ID.
	 * @param \WP_Post $post    Post object.
	 * @return bool
	 */
	private function is_system_page( $post_id, $post ) {
		// WooCommerce pages.
		if ( class_exists( 'WooCommerce' ) && function_exists( 'wc_get_page_id' ) ) {
			foreach ( self::WOO_PAGE_TYPES as $page_type ) {
				$woo_page_id = wc_get_page_id( $page_type );
				if ( $woo_page_id > 0 && (int) $woo_page_id === (int) $post_id ) {
					return true;
				}
			}
		}

		// Check slug.
		if ( in_array( $post->post_name, self::EXCLUDED_SLUGS, true ) ) {
			return true;
		}

		// Check title.
		$lower_title = strtolower( trim( $post->post_title ) );
		if ( in_array( $lower_title, self::EXCLUDED_TITLES, true ) ) {
			return true;
		}

		// Check sitemap pattern.
		if ( preg_match( self::SITEMAP_PATTERN, $post->post_name ) ) {
			return true;
		}
		if ( preg_match( self::SITEMAP_PATTERN, $post->post_title ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if post is password protected.
	 *
	 * @param \WP_Post $post Post object.
	 * @return bool
	 */
	private function is_password_protected( $post ) {
		return ! empty( $post->post_password );
	}

	/**
	 * Check if post has an accessible URL.
	 *
	 * @param \WP_Post $post Post object.
	 * @return bool
	 */
	private function has_accessible_url( $post ) {
		$type_obj = get_post_type_object( $post->post_type );
		if ( ! $type_obj || ! is_post_type_viewable( $type_obj ) ) {
			return false;
		}

		$permalink = get_permalink( $post );
		if ( ! is_string( $permalink ) || empty( $permalink ) ) {
			return false;
		}

		// Private posts have valid URLs for chatbot context.
		if ( 'private' === $post->post_status ) {
			return true;
		}

		return true;
	}

	/**
	 * Get exclusion reason for a post (for debugging/UI).
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $post_type Post type.
	 * @return string|null Reason string or null if not excluded.
	 */
	public function get_exclusion_reason( $post_id, $post_type ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return __( 'Post not found', 'intufind' );
		}

		if ( ! $this->is_post_type_enabled( $post_type ) ) {
			return __( 'Post type not enabled', 'intufind' );
		}

		if ( Intufind_List_Columns::is_excluded( $post_id ) ) {
			return __( 'Manually excluded', 'intufind' );
		}

		if ( ! $this->is_syncable_status( $post->post_status, $post_id ) ) {
			/* translators: %s: post status */
			return sprintf( __( 'Status: %s', 'intufind' ), $post->post_status );
		}

		if ( 'product' === $post_type && class_exists( 'WooCommerce' ) ) {
			$product = wc_get_product( $post_id );
			if ( $product ) {
				if ( $product->is_type( 'variation' ) ) {
					return __( 'Product variation', 'intufind' );
				}
				if ( 'hidden' === $product->get_catalog_visibility() ) {
					return __( 'Hidden from catalog', 'intufind' );
				}
			}
		}

		if ( $this->is_system_page( $post_id, $post ) ) {
			return __( 'System page', 'intufind' );
		}

		if ( $this->is_password_protected( $post ) ) {
			return __( 'Password protected', 'intufind' );
		}

		if ( ! $this->has_accessible_url( $post ) ) {
			return __( 'No accessible URL', 'intufind' );
		}

		return null;
	}

	// =========================================================================
	// Taxonomy Exclusions
	// =========================================================================

	/**
	 * Get all available taxonomies for sync configuration.
	 *
	 * Returns taxonomies that could potentially be synced,
	 * excluding system taxonomies.
	 *
	 * @return array Array of taxonomy name => label.
	 */
	public function get_available_taxonomies() {
		$all_taxonomies = get_taxonomies(
			array(
				'public' => true,
			),
			'objects'
		);

		$available = array();

		foreach ( $all_taxonomies as $taxonomy ) {
			// Skip if in excluded list.
			if ( in_array( $taxonomy->name, self::EXCLUDED_SYSTEM_TAXONOMIES, true ) ) {
				continue;
			}

			// Must be public, publicly_queryable, or show_ui.
			if ( ! $taxonomy->public && ! $taxonomy->publicly_queryable && ! $taxonomy->show_ui ) {
				continue;
			}

			$available[ $taxonomy->name ] = $taxonomy->label;
		}

		// Also include WooCommerce product attributes (pa_*).
		$this->add_woocommerce_attributes( $available );

		// Sort alphabetically by label.
		asort( $available );

		return $available;
	}

	/**
	 * Add WooCommerce product attributes to available taxonomies.
	 *
	 * @param array $available Reference to available taxonomies array.
	 * @return void
	 */
	private function add_woocommerce_attributes( &$available ) {
		if ( ! class_exists( 'WooCommerce' ) || ! function_exists( 'wc_get_attribute_taxonomies' ) ) {
			return;
		}

		$attribute_taxonomies = wc_get_attribute_taxonomies();
		foreach ( $attribute_taxonomies as $attribute ) {
			$taxonomy_name = wc_attribute_taxonomy_name( $attribute->attribute_name );
			if ( ! isset( $available[ $taxonomy_name ] ) && taxonomy_exists( $taxonomy_name ) ) {
				$available[ $taxonomy_name ] = $attribute->attribute_label;
			}
		}
	}

	/**
	 * Get the enabled taxonomies for sync.
	 *
	 * @return array Array of enabled taxonomy names.
	 */
	public function get_enabled_taxonomies() {
		$saved = get_option( self::OPTION_SYNC_TAXONOMIES, array() );

		// Default to common taxonomies.
		if ( empty( $saved ) ) {
			return $this->get_default_taxonomies();
		}

		// If saved as associative array (enabled/disabled), convert.
		if ( isset( $saved[ array_key_first( $saved ) ] ) && is_bool( $saved[ array_key_first( $saved ) ] ) ) {
			return array_keys( array_filter( $saved ) );
		}

		return (array) $saved;
	}

	/**
	 * Get default taxonomies to enable.
	 *
	 * @return array Array of taxonomy names.
	 */
	private function get_default_taxonomies() {
		$defaults = array( 'category', 'post_tag' );

		// Add WooCommerce taxonomies if active.
		if ( class_exists( 'WooCommerce' ) ) {
			$defaults[] = 'product_cat';
			$defaults[] = 'product_tag';

			// Add common product attributes.
			$common_attributes = array( 'pa_color', 'pa_colour', 'pa_size', 'pa_brand' );
			foreach ( $common_attributes as $attr ) {
				if ( taxonomy_exists( $attr ) ) {
					$defaults[] = $attr;
				}
			}
		}

		return $defaults;
	}

	/**
	 * Check if a taxonomy is enabled for sync.
	 *
	 * @param string $taxonomy Taxonomy name.
	 * @return bool
	 */
	public function is_taxonomy_enabled( $taxonomy ) {
		// Always exclude system taxonomies.
		if ( in_array( $taxonomy, self::EXCLUDED_SYSTEM_TAXONOMIES, true ) ) {
			return false;
		}

		$enabled = $this->get_enabled_taxonomies();
		return in_array( $taxonomy, $enabled, true );
	}

	/**
	 * Save enabled taxonomies.
	 *
	 * @param array $taxonomies Array of taxonomy names to enable.
	 * @return bool
	 */
	public function save_enabled_taxonomies( $taxonomies ) {
		// Filter out system taxonomies.
		$taxonomies = array_filter(
			$taxonomies,
			function ( $tax ) {
				return ! in_array( $tax, self::EXCLUDED_SYSTEM_TAXONOMIES, true );
			}
		);

		return update_option( self::OPTION_SYNC_TAXONOMIES, array_values( $taxonomies ) );
	}

	/**
	 * Get taxonomy term count for a taxonomy.
	 *
	 * @param string $taxonomy Taxonomy name.
	 * @return int Term count.
	 */
	public function get_taxonomy_term_count( $taxonomy ) {
		$count = wp_count_terms( array( 'taxonomy' => $taxonomy, 'hide_empty' => false ) );
		return is_wp_error( $count ) ? 0 : (int) $count;
	}

	/**
	 * Get taxonomy info with term counts.
	 *
	 * @return array Array of taxonomy info with counts.
	 */
	public function get_taxonomy_info() {
		$available = $this->get_available_taxonomies();
		$enabled   = $this->get_enabled_taxonomies();
		$info      = array();

		foreach ( $available as $taxonomy => $label ) {
			$tax_object  = get_taxonomy( $taxonomy );
			$post_types  = $tax_object ? $tax_object->object_type : array();
			$post_labels = array();

			foreach ( $post_types as $post_type ) {
				$pt_object = get_post_type_object( $post_type );
				if ( $pt_object ) {
					$post_labels[] = $pt_object->labels->name;
				}
			}

			$info[ $taxonomy ] = array(
				'name'        => $taxonomy,
				'label'       => $label,
				'enabled'     => in_array( $taxonomy, $enabled, true ),
				'count'       => $this->get_taxonomy_term_count( $taxonomy ),
				'post_types'  => $post_types,
				'post_labels' => $post_labels,
			);
		}

		return $info;
	}
}
