<?php
/**
 * List Columns functionality.
 *
 * Adds sync and searchable columns to WordPress post list tables
 * for granular control over AI indexing.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles post list table columns for sync control.
 */
class Intufind_List_Columns {

	/**
	 * Post meta keys.
	 */
	const META_EXCLUDE_SYNC = '_intufind_exclude_sync';
	const META_SEARCHABLE   = '_intufind_searchable';

	/**
	 * Exclusions instance.
	 *
	 * @var Intufind_Exclusions
	 */
	private $exclusions;

	/**
	 * Sync status instance.
	 *
	 * @var Intufind_Sync_Status
	 */
	private $status;

	/**
	 * API instance.
	 *
	 * @var Intufind_API
	 */
	private $api;

	/**
	 * Content extractor instance.
	 *
	 * @var Intufind_Content_Extractor
	 */
	private $extractor;

	/**
	 * Constructor.
	 *
	 * @param Intufind_Exclusions        $exclusions Exclusions instance.
	 * @param Intufind_Sync_Status       $status     Sync status instance.
	 * @param Intufind_API               $api        API instance.
	 * @param Intufind_Content_Extractor $extractor  Content extractor instance.
	 */
	public function __construct( Intufind_Exclusions $exclusions, Intufind_Sync_Status $status, Intufind_API $api, Intufind_Content_Extractor $extractor ) {
		$this->exclusions = $exclusions;
		$this->status     = $status;
		$this->api        = $api;
		$this->extractor  = $extractor;
	}

	/**
	 * Initialize hooks.
	 */
	public function init() {
		// Only proceed if connected.
		if ( ! get_option( INTUFIND_OPTION_CONNECTED, false ) ) {
			return;
		}

		add_action( 'admin_init', array( $this, 'register_columns' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'wp_ajax_intufind_toggle_sync', array( $this, 'ajax_toggle_sync' ) );
		add_action( 'wp_ajax_intufind_toggle_searchable', array( $this, 'ajax_toggle_searchable' ) );
	}

	/**
	 * Enqueue assets for list table pages.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_assets( $hook ) {
		// Only load on list table pages.
		if ( 'edit.php' !== $hook ) {
			return;
		}

		// Check if this post type has our columns.
		$post_type = isset( $_GET['post_type'] ) ? sanitize_key( $_GET['post_type'] ) : 'post';
		$enabled_types = $this->exclusions->get_enabled_post_types();

		if ( ! in_array( $post_type, $enabled_types, true ) ) {
			return;
		}

		// Enqueue styles.
		wp_enqueue_style(
			'intufind-list-columns',
			INTUFIND_PLUGIN_URL . 'admin/css/intufind-admin.css',
			array(),
			INTUFIND_VERSION
		);

		// Enqueue scripts.
		wp_enqueue_script(
			'intufind-list-columns',
			INTUFIND_PLUGIN_URL . 'admin/js/intufind-admin.js',
			array( 'jquery' ),
			INTUFIND_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'intufind-list-columns',
			'intufindAdmin',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'intufind_admin' ),
				'strings' => array(
					'error' => __( 'Error', 'intufind' ),
				),
			)
		);
	}

	/**
	 * Register columns for enabled post types.
	 */
	public function register_columns() {
		$enabled_types = $this->exclusions->get_enabled_post_types();

		foreach ( $enabled_types as $post_type ) {
			// Add columns.
			add_filter( "manage_{$post_type}_posts_columns", array( $this, 'add_columns' ) );
			add_action( "manage_{$post_type}_posts_custom_column", array( $this, 'render_column' ), 10, 2 );

			// Bulk actions.
			add_filter( "bulk_actions-edit-{$post_type}", array( $this, 'add_bulk_actions' ) );
			add_filter( "handle_bulk_actions-edit-{$post_type}", array( $this, 'handle_bulk_actions' ), 10, 3 );
		}

		// Admin notices for bulk actions.
		add_action( 'admin_notices', array( $this, 'bulk_action_notices' ) );
	}

	/**
	 * Add columns to post list table.
	 *
	 * @param array $columns Existing columns.
	 * @return array Modified columns.
	 */
	public function add_columns( $columns ) {
		$new_columns = array();

		// Column headers with WordPress-style icon markup and tooltips.
		$sync_title       = __( 'Intufind: AI Sync', 'intufind' );
		$searchable_title = __( 'Intufind: Searchable', 'intufind' );

		$sync_header = sprintf(
			'<span class="intufind-col-header dashicons dashicons-cloud-saved" aria-hidden="true" title="%s"></span><span class="screen-reader-text">%s</span>',
			esc_attr( $sync_title ),
			esc_html( $sync_title )
		);
		$searchable_header = sprintf(
			'<span class="intufind-col-header dashicons dashicons-visibility" aria-hidden="true" title="%s"></span><span class="screen-reader-text">%s</span>',
			esc_attr( $searchable_title ),
			esc_html( $searchable_title )
		);

		foreach ( $columns as $key => $label ) {
			// Insert before date column.
			if ( 'date' === $key ) {
				$new_columns['intufind_sync']       = $sync_header;
				$new_columns['intufind_searchable'] = $searchable_header;
			}
			$new_columns[ $key ] = $label;
		}

		// If date doesn't exist, add at end.
		if ( ! isset( $new_columns['intufind_sync'] ) ) {
			$new_columns['intufind_sync']       = $sync_header;
			$new_columns['intufind_searchable'] = $searchable_header;
		}

		return $new_columns;
	}

	/**
	 * Render column content.
	 *
	 * @param string $column  Column name.
	 * @param int    $post_id Post ID.
	 */
	public function render_column( $column, $post_id ) {
		if ( 'intufind_sync' === $column ) {
			$this->render_sync_column( $post_id );
		} elseif ( 'intufind_searchable' === $column ) {
			$this->render_searchable_column( $post_id );
		}
	}

	/**
	 * Render sync column.
	 *
	 * @param int $post_id Post ID.
	 */
	private function render_sync_column( $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return;
		}

		// Check if post status allows syncing (published or private).
		$syncable_statuses = array( 'publish', 'private' );
		if ( ! in_array( $post->post_status, $syncable_statuses, true ) ) {
			$this->render_disabled_toggle( 'sync', __( 'Publish to enable sync', 'intufind' ) );
			return;
		}

		// Check for non-overridable exclusions (system pages, password protected, etc.)
		// but NOT the manual exclusion check (that's what we're toggling).
		if ( $this->is_system_excluded( $post_id, $post ) ) {
			$reason = $this->exclusions->get_exclusion_reason( $post_id, $post->post_type );
			$this->render_disabled_toggle( 'sync', $reason ?: __( 'Cannot be synced', 'intufind' ) );
			return;
		}

		// Get manual sync setting.
		// Default: published = synced, private = not synced (but user can override).
		$exclude_meta = get_post_meta( $post_id, self::META_EXCLUDE_SYNC, true );
		if ( '' === $exclude_meta ) {
			// No explicit setting - use defaults.
			$is_synced = 'publish' === $post->post_status;
		} else {
			$is_synced = 'yes' !== $exclude_meta;
		}

		// Get sync status for indicator.
		$sync_status  = $this->status->get_status( $post_id );
		$status_class = $this->get_status_class( $sync_status );
		$nonce        = wp_create_nonce( 'intufind_toggle_sync_' . $post_id );

		?>
		<div class="intufind-col-toggle" data-post-id="<?php echo esc_attr( $post_id ); ?>">
			<label class="intufind-mini-toggle">
				<input 
					type="checkbox" 
					class="intufind-sync-toggle"
					data-post-id="<?php echo esc_attr( $post_id ); ?>"
					data-nonce="<?php echo esc_attr( $nonce ); ?>"
					<?php checked( $is_synced ); ?>
				/>
				<span class="intufind-mini-toggle__slider"></span>
			</label>
			<?php if ( $is_synced ) : ?>
				<span class="intufind-col-status intufind-col-status--<?php echo esc_attr( $status_class ); ?>" title="<?php echo esc_attr( $this->get_status_tooltip( $sync_status ) ); ?>"></span>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Check if post has non-overridable exclusions.
	 *
	 * These are things users shouldn't be able to override (system pages, etc.)
	 *
	 * @param int      $post_id Post ID.
	 * @param \WP_Post $post    Post object.
	 * @return bool
	 */
	private function is_system_excluded( $post_id, $post ) {
		// Password protected.
		if ( ! empty( $post->post_password ) ) {
			return true;
		}

		// WooCommerce system pages.
		if ( class_exists( 'WooCommerce' ) ) {
			$woo_pages = array( 'cart', 'checkout', 'myaccount' );
			foreach ( $woo_pages as $page_type ) {
				if ( $post_id === wc_get_page_id( $page_type ) ) {
					return true;
				}
			}
		}

		// Product variations (handled at parent level).
		if ( 'product_variation' === $post->post_type ) {
			return true;
		}

		return false;
	}

	/**
	 * Render searchable column.
	 *
	 * @param int $post_id Post ID.
	 */
	private function render_searchable_column( $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return;
		}

		// Not a syncable status (drafts, etc.).
		if ( ! in_array( $post->post_status, array( 'publish', 'private' ), true ) ) {
			$this->render_disabled_toggle( 'searchable', __( 'Publish first', 'intufind' ) );
			return;
		}

		// System-excluded posts can't be synced, so searchable doesn't apply.
		if ( $this->is_system_excluded( $post_id, $post ) ) {
			$this->render_disabled_toggle( 'searchable', __( 'Cannot be synced', 'intufind' ) );
			return;
		}

		// Check if synced - searchable only applies to synced content.
		if ( ! self::is_synced( $post_id ) ) {
			$this->render_disabled_toggle( 'searchable', __( 'Enable sync first', 'intufind' ) );
			return;
		}

		// Get searchable setting.
		// Default: published = searchable, private = not searchable (chatbot only).
		$searchable_meta = get_post_meta( $post_id, self::META_SEARCHABLE, true );
		if ( '' === $searchable_meta ) {
			// No explicit setting - use defaults.
			$is_searchable = 'publish' === $post->post_status;
		} else {
			$is_searchable = 'no' !== $searchable_meta;
		}

		$nonce = wp_create_nonce( 'intufind_toggle_searchable_' . $post_id );

		?>
		<div class="intufind-col-toggle" data-post-id="<?php echo esc_attr( $post_id ); ?>">
			<label class="intufind-mini-toggle">
				<input 
					type="checkbox" 
					class="intufind-searchable-toggle"
					data-post-id="<?php echo esc_attr( $post_id ); ?>"
					data-nonce="<?php echo esc_attr( $nonce ); ?>"
					<?php checked( $is_searchable ); ?>
				/>
				<span class="intufind-mini-toggle__slider"></span>
			</label>
		</div>
		<?php
	}

	/**
	 * Render a disabled toggle.
	 *
	 * @param string $type    Toggle type.
	 * @param string $tooltip Tooltip text.
	 * @param bool   $show_dash Whether to show dash indicator.
	 */
	private function render_disabled_toggle( $type, $tooltip, $show_dash = true ) {
		?>
		<div class="intufind-col-toggle intufind-col-toggle--disabled" title="<?php echo esc_attr( $tooltip ); ?>">
			<?php if ( $show_dash ) : ?>
				<span class="intufind-col-dash">&mdash;</span>
			<?php else : ?>
				<span class="dashicons dashicons-lock" style="color: #999; font-size: 14px;"></span>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Get status class for indicator.
	 *
	 * @param array $sync_status Sync status array.
	 * @return string Status class.
	 */
	private function get_status_class( $sync_status ) {
		$status = $sync_status['status'] ?? '';

		switch ( $status ) {
			case 'synced':
				return 'success';
			case 'error':
				return 'error';
			case 'pending':
				return 'pending';
			default:
				return 'pending';
		}
	}

	/**
	 * Get status tooltip.
	 *
	 * @param array $sync_status Sync status array.
	 * @return string Tooltip text.
	 */
	private function get_status_tooltip( $sync_status ) {
		$status = $sync_status['status'] ?? '';

		switch ( $status ) {
			case 'synced':
				$time = $sync_status['timestamp'] ?? 0;
				if ( $time ) {
					return sprintf(
						/* translators: %s: time ago */
						__( 'Synced %s', 'intufind' ),
						$this->status->format_time_ago( $time )
					);
				}
				return __( 'Synced', 'intufind' );

			case 'error':
				$error = $sync_status['error'] ?? '';
				return $error ? $error : __( 'Sync error', 'intufind' );

			case 'pending':
				return __( 'Pending sync', 'intufind' );

			default:
				return __( 'Not synced yet', 'intufind' );
		}
	}

	/**
	 * Add bulk actions.
	 *
	 * @param array $actions Existing actions.
	 * @return array Modified actions.
	 */
	public function add_bulk_actions( $actions ) {
		$actions['intufind_enable_sync']    = __( 'Enable AI Sync', 'intufind' );
		$actions['intufind_disable_sync']   = __( 'Disable AI Sync', 'intufind' );
		$actions['intufind_make_searchable'] = __( 'Make Searchable', 'intufind' );
		$actions['intufind_hide_search']    = __( 'Hide from Search', 'intufind' );

		return $actions;
	}

	/**
	 * Handle bulk actions.
	 *
	 * @param string $redirect_url Redirect URL.
	 * @param string $action       Action name.
	 * @param array  $post_ids     Post IDs.
	 * @return string Modified redirect URL.
	 */
	public function handle_bulk_actions( $redirect_url, $action, $post_ids ) {
		$valid_actions = array(
			'intufind_enable_sync',
			'intufind_disable_sync',
			'intufind_make_searchable',
			'intufind_hide_search',
		);

		if ( ! in_array( $action, $valid_actions, true ) ) {
			return $redirect_url;
		}

		$count = 0;

		foreach ( $post_ids as $post_id ) {
			switch ( $action ) {
				case 'intufind_enable_sync':
					delete_post_meta( $post_id, self::META_EXCLUDE_SYNC );
					$count++;
					break;

				case 'intufind_disable_sync':
					update_post_meta( $post_id, self::META_EXCLUDE_SYNC, 'yes' );
					$count++;
					break;

				case 'intufind_make_searchable':
					update_post_meta( $post_id, self::META_SEARCHABLE, 'yes' );
					$count++;
					break;

				case 'intufind_hide_search':
					update_post_meta( $post_id, self::META_SEARCHABLE, 'no' );
					$count++;
					break;
			}
		}

		return add_query_arg(
			array(
				'intufind_bulk_action' => $action,
				'intufind_bulk_count'  => $count,
			),
			$redirect_url
		);
	}

	/**
	 * Display bulk action notices.
	 */
	public function bulk_action_notices() {
		if ( empty( $_GET['intufind_bulk_action'] ) || empty( $_GET['intufind_bulk_count'] ) ) {
			return;
		}

		$action = sanitize_text_field( wp_unslash( $_GET['intufind_bulk_action'] ) );
		$count  = intval( $_GET['intufind_bulk_count'] );

		$messages = array(
			'intufind_enable_sync'    => sprintf(
				/* translators: %d: number of posts */
				_n( 'AI sync enabled for %d item.', 'AI sync enabled for %d items.', $count, 'intufind' ),
				$count
			),
			'intufind_disable_sync'   => sprintf(
				/* translators: %d: number of posts */
				_n( 'AI sync disabled for %d item.', 'AI sync disabled for %d items.', $count, 'intufind' ),
				$count
			),
			'intufind_make_searchable' => sprintf(
				/* translators: %d: number of posts */
				_n( '%d item made searchable.', '%d items made searchable.', $count, 'intufind' ),
				$count
			),
			'intufind_hide_search'    => sprintf(
				/* translators: %d: number of posts */
				_n( '%d item hidden from search.', '%d items hidden from search.', $count, 'intufind' ),
				$count
			),
		);

		if ( isset( $messages[ $action ] ) ) {
			printf(
				'<div class="notice notice-success is-dismissible"><p>%s</p></div>',
				esc_html( $messages[ $action ] )
			);
		}
	}

	/**
	 * AJAX handler for toggling sync.
	 */
	public function ajax_toggle_sync() {
		$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		$nonce   = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

		if ( ! $post_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid post ID.', 'intufind' ) ) );
		}

		if ( ! wp_verify_nonce( $nonce, 'intufind_toggle_sync_' . $post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Session expired. Please refresh the page and try again.', 'intufind' ) ) );
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		// Get current effective state and toggle to opposite.
		$is_currently_synced = self::is_synced( $post_id );
		$post                = get_post( $post_id );

		if ( $is_currently_synced ) {
			// Was synced, now exclude.
			update_post_meta( $post_id, self::META_EXCLUDE_SYNC, 'yes' );
			$is_synced = false;

			// If it was actually synced to the cloud, delete it immediately.
			$sync_status = $this->status->get_status( $post_id );
			if ( Intufind_Sync_Status::STATUS_SYNCED === $sync_status['status'] ) {
				$post_type = $post ? $post->post_type : 'post';

				// Delete from cloud.
				if ( 'product' === $post_type ) {
					$result = $this->api->delete_products( array( (string) $post_id ) );
				} else {
					$result = $this->api->delete_posts( array( (string) $post_id ) );
				}

				// Clear the sync status.
				if ( ! is_wp_error( $result ) ) {
					$this->status->clear_status( $post_id );
				}
			}
		} else {
			// Was excluded, now include.
			update_post_meta( $post_id, self::META_EXCLUDE_SYNC, 'no' );
			$is_synced    = true;
			$sync_success = false;

			// Immediately sync to the cloud.
			$post_type = $post ? $post->post_type : 'post';
			$document  = $this->extractor->extract( $post_id, $post_type );

			if ( $document ) {
				// Add searchable flag based on current setting.
				$document['searchable'] = self::is_searchable( $post_id );

				// Upsert to cloud.
				if ( 'product' === $post_type ) {
					$result = $this->api->upsert_products( array( $document ) );
				} else {
					$result = $this->api->upsert_posts( array( $document ) );
				}

				// Update sync status.
				if ( ! is_wp_error( $result ) ) {
					$this->status->mark_synced( $post_id, '', 'manual' );
					$sync_success = true;
				} else {
					$this->status->mark_error( $post_id, $result->get_error_message() );
				}
			}
		}

		$response = array(
			'synced'     => $is_synced,
			'syncStatus' => $is_synced ? ( $sync_success ? 'success' : 'error' ) : null,
			'message'    => $is_synced ? __( 'Added to AI', 'intufind' ) : __( 'Removed from AI', 'intufind' ),
		);

		// Include searchable nonce when enabling sync so JS can restore the toggle.
		if ( $is_synced ) {
			$response['searchableNonce'] = wp_create_nonce( 'intufind_toggle_searchable_' . $post_id );
		}

		wp_send_json_success( $response );
	}

	/**
	 * AJAX handler for toggling searchable.
	 */
	public function ajax_toggle_searchable() {
		$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		$nonce   = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

		if ( ! $post_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid post ID.', 'intufind' ) ) );
		}

		if ( ! wp_verify_nonce( $nonce, 'intufind_toggle_searchable_' . $post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Session expired. Please refresh the page and try again.', 'intufind' ) ) );
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'intufind' ) ) );
		}

		// Get current effective state and toggle to opposite.
		$is_currently_searchable = self::is_searchable( $post_id );

		if ( $is_currently_searchable ) {
			// Was searchable, now hide.
			update_post_meta( $post_id, self::META_SEARCHABLE, 'no' );
			$is_searchable = false;
		} else {
			// Was hidden, now searchable.
			update_post_meta( $post_id, self::META_SEARCHABLE, 'yes' );
			$is_searchable = true;
		}

		wp_send_json_success(
			array(
				'searchable' => $is_searchable,
				'message'    => $is_searchable ? __( 'Now searchable', 'intufind' ) : __( 'Hidden from search', 'intufind' ),
			)
		);
	}

	/**
	 * Check if a post is synced (not excluded).
	 *
	 * Defaults: published = synced, private = not synced (unless explicitly enabled).
	 *
	 * @param int $post_id Post ID.
	 * @return bool Whether synced.
	 */
	public static function is_synced( $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return false;
		}

		$exclude_meta = get_post_meta( $post_id, self::META_EXCLUDE_SYNC, true );

		// Explicit setting overrides defaults.
		if ( 'yes' === $exclude_meta ) {
			return false;
		}
		if ( 'no' === $exclude_meta ) {
			return true;
		}

		// Default based on post status.
		return 'publish' === $post->post_status;
	}

	/**
	 * Check if a post is excluded from sync.
	 *
	 * @param int $post_id Post ID.
	 * @return bool Whether excluded.
	 */
	public static function is_excluded( $post_id ) {
		return ! self::is_synced( $post_id );
	}

	/**
	 * Check if a post is searchable.
	 *
	 * Defaults: published = searchable, private = not searchable (chatbot only).
	 * User can override either default.
	 *
	 * @param int $post_id Post ID.
	 * @return bool Whether searchable.
	 */
	public static function is_searchable( $post_id ) {
		// Must be synced to be searchable.
		if ( ! self::is_synced( $post_id ) ) {
			return false;
		}

		$post = get_post( $post_id );
		if ( ! $post ) {
			return false;
		}

		$searchable_meta = get_post_meta( $post_id, self::META_SEARCHABLE, true );

		// Explicit setting overrides defaults.
		if ( 'no' === $searchable_meta ) {
			return false;
		}
		if ( 'yes' === $searchable_meta ) {
			return true;
		}

		// Default based on post status.
		return 'publish' === $post->post_status;
	}
}
