<?php
/**
 * MCP (Model Context Protocol) server.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MCP Server class.
 *
 * Provides a REST API endpoint for MCP (Model Context Protocol) communication,
 * enabling AI agents to interact with WordPress/WooCommerce data.
 *
 * Security: User context is enforced via JWT for logged-in users, or
 * email+order verification for guests. This prevents cross-user data leakage.
 */
class Intufind_MCP {

	/**
	 * API client instance.
	 *
	 * @var Intufind_API
	 */
	private $api;

	/**
	 * Current user context (set during authentication).
	 *
	 * @var array|null
	 */
	private $user_context = null;

	/**
	 * REST namespace.
	 *
	 * @var string
	 */
	const REST_NAMESPACE = 'intufind/v1';

	/**
	 * JWT algorithm.
	 *
	 * @var string
	 */
	const JWT_ALGORITHM = 'HS256';

	/**
	 * JWT expiration time in seconds (24 hours).
	 *
	 * @var int
	 */
	const JWT_EXPIRATION = 86400;

	/**
	 * Rate limit for order lookup (requests per window).
	 *
	 * @var int
	 */
	const RATE_LIMIT_MAX_REQUESTS = 5;

	/**
	 * Rate limit window in seconds (1 minute).
	 *
	 * @var int
	 */
	const RATE_LIMIT_WINDOW = 60;

	/**
	 * Option key for JWT signing secret.
	 *
	 * @var string
	 */
	const OPTION_JWT_SECRET = 'intufind_jwt_secret';

	/**
	 * Constructor.
	 *
	 * @param Intufind_API $api API client instance.
	 */
	public function __construct( Intufind_API $api ) {
		$this->api = $api;
	}

	/**
	 * Initialize MCP server.
	 *
	 * @return void
	 */
	public function init() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes.
	 *
	 * @return void
	 */
	public function register_routes() {
		// Main MCP endpoint.
		register_rest_route(
			self::REST_NAMESPACE,
			'/mcp',
			array(
				'methods'             => \WP_REST_Server::ALLMETHODS,
				'callback'            => array( $this, 'handle_mcp_request' ),
				'permission_callback' => array( $this, 'authenticate' ),
			)
		);

		// Domain-specific endpoint.
		register_rest_route(
			self::REST_NAMESPACE,
			'/mcp/domain/(?P<domain>[a-zA-Z0-9_-]+)',
			array(
				'methods'             => \WP_REST_Server::ALLMETHODS,
				'callback'            => array( $this, 'handle_domain_request' ),
				'permission_callback' => array( $this, 'authenticate' ),
			)
		);

	}

	/**
	 * Get user JWT for widget initialization.
	 *
	 * Called by the plugin when rendering widget config to include
	 * a JWT for authenticated users. This avoids exposing a REST endpoint.
	 *
	 * @return array|null User token data or null if not logged in.
	 */
	public function get_user_token_for_widget() {
		if ( ! is_user_logged_in() ) {
			return null;
		}

		$user_id = get_current_user_id();
		$token   = $this->generate_user_jwt( $user_id );

		if ( ! $token ) {
			return null;
		}

		$user = wp_get_current_user();

		return array(
			'token' => $token,
			'user'  => array(
				'id'    => $user_id,
				'email' => $user->user_email,
				'name'  => $user->display_name,
			),
		);
	}

	/**
	 * Authenticate MCP request.
	 *
	 * Validates requests using:
	 * 1. Tenant API key (server-to-server auth from cloud via Bearer token)
	 * 2. User JWT (optional, for user-scoped operations via X-User-JWT header)
	 *
	 * The cloud must send:
	 * - Authorization: Bearer <api_key>
	 * - X-Intufind-Origin: cloud
	 * - X-User-JWT: <user_jwt> (optional, for authenticated users)
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return bool
	 */
	public function authenticate( \WP_REST_Request $request ) {
		// Allow preflight requests.
		if ( 'OPTIONS' === $request->get_method() ) {
			return true;
		}

		// Plugin must be connected to a workspace.
		$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		if ( empty( $workspace_id ) ) {
			return false;
		}

		// Get stored API key.
		$api_key = get_option( INTUFIND_OPTION_API_KEY, '' );
		if ( empty( $api_key ) ) {
			return false;
		}

		// Verify cloud origin header (defense in depth).
		$cloud_origin = $request->get_header( 'x-intufind-origin' );
		if ( 'cloud' !== $cloud_origin ) {
			return false;
		}

		// Validate Bearer token (API key auth from cloud).
		$auth_header = $request->get_header( 'authorization' );
		if ( empty( $auth_header ) || ! preg_match( '/Bearer\s+(.+)/i', $auth_header, $matches ) ) {
			return false;
		}

		$token = $matches[1];
		if ( ! hash_equals( $api_key, $token ) ) {
			return false;
		}

		// API key valid - now check for optional user JWT.
		$user_jwt = $request->get_header( 'x-user-jwt' );
		if ( ! empty( $user_jwt ) ) {
			$this->user_context = $this->validate_user_jwt( $user_jwt );
			// Note: Invalid JWT doesn't fail auth - just means no user context.
			// Tools that require user context will handle this themselves.
		}

		return true;
	}

	/**
	 * Generate a JWT for a WordPress user.
	 *
	 * Called by the widget to get a user token for authenticated MCP calls.
	 * Uses a dedicated JWT secret (separate from API key) for signing.
	 *
	 * @param int $user_id WordPress user ID.
	 * @return string|null JWT token or null on failure.
	 */
	public function generate_user_jwt( $user_id ) {
		$user = get_user_by( 'id', $user_id );
		if ( ! $user ) {
			return null;
		}

		$secret = $this->get_jwt_secret();
		if ( empty( $secret ) ) {
			return null;
		}

		$now = time();

		// Determine user type based on WooCommerce customer status.
		$user_type = 'guest';
		if ( function_exists( 'wc_get_customer' ) ) {
			$customer = wc_get_customer( $user_id );
			if ( $customer && $customer->get_id() ) {
				$user_type = 'customer';
			}
		}

		// Build payload matching cloud's SecureUserToken format.
		$payload = array(
			'iss'          => home_url(),
			'user_id'      => $user_id,
			'user_type'    => $user_type,
			'iat'          => $now,
			'nbf'          => $now,                          // Not valid before now.
			'exp'          => $now + self::JWT_EXPIRATION,
			'jti'          => wp_generate_uuid4(),           // Unique token ID for potential revocation.
			'user_email'   => $user->user_email,
			'user_roles'   => $user->roles,
			'display_name' => $user->display_name,
			'first_name'   => $user->first_name,
			'last_name'    => $user->last_name,
			'permissions'  => array(
				'can_view_customer_data' => true, // User can view their own data.
			),
		);

		return $this->encode_jwt( $payload, $secret );
	}

	/**
	 * Validate a user JWT and extract user context.
	 *
	 * Uses the dedicated JWT secret (separate from API key) for verification.
	 *
	 * @param string $token JWT token.
	 * @return array|null User context or null if invalid.
	 */
	private function validate_user_jwt( $token ) {
		$secret = $this->get_jwt_secret();
		if ( empty( $secret ) ) {
			return null;
		}

		$payload = $this->decode_jwt( $token, $secret );
		if ( ! $payload ) {
			return null;
		}

		$now = time();

		// Verify not-before (nbf) claim.
		if ( isset( $payload['nbf'] ) && $payload['nbf'] > $now ) {
			return null;
		}

		// Verify expiration.
		if ( ! isset( $payload['exp'] ) || $payload['exp'] < $now ) {
			return null;
		}

		// Verify issuer matches this site.
		if ( ! isset( $payload['iss'] ) || $payload['iss'] !== home_url() ) {
			return null;
		}

		return array(
			'user_id' => $payload['sub'] ?? null,
			'email'   => $payload['email'] ?? null,
			'name'    => $payload['name'] ?? null,
			'jti'     => $payload['jti'] ?? null, // Include for potential revocation checks.
		);
	}

	/**
	 * Encode a JWT.
	 *
	 * @param array  $payload JWT payload.
	 * @param string $secret  Secret key.
	 * @return string JWT token.
	 */
	private function encode_jwt( $payload, $secret ) {
		$header = array(
			'typ' => 'JWT',
			'alg' => self::JWT_ALGORITHM,
		);

		$segments = array(
			$this->base64url_encode( wp_json_encode( $header ) ),
			$this->base64url_encode( wp_json_encode( $payload ) ),
		);

		$signing_input = implode( '.', $segments );
		$signature     = hash_hmac( 'sha256', $signing_input, $secret, true );
		$segments[]    = $this->base64url_encode( $signature );

		return implode( '.', $segments );
	}

	/**
	 * Decode and verify a JWT.
	 *
	 * @param string $token  JWT token.
	 * @param string $secret Secret key.
	 * @return array|null Payload or null if invalid.
	 */
	private function decode_jwt( $token, $secret ) {
		$segments = explode( '.', $token );
		if ( count( $segments ) !== 3 ) {
			return null;
		}

		list( $header_b64, $payload_b64, $signature_b64 ) = $segments;

		// Verify signature.
		$signing_input    = $header_b64 . '.' . $payload_b64;
		$expected_sig     = hash_hmac( 'sha256', $signing_input, $secret, true );
		$actual_sig       = $this->base64url_decode( $signature_b64 );

		if ( ! hash_equals( $expected_sig, $actual_sig ) ) {
			return null;
		}

		// Decode payload.
		$payload = json_decode( $this->base64url_decode( $payload_b64 ), true );
		if ( ! is_array( $payload ) ) {
			return null;
		}

		return $payload;
	}

	/**
	 * Base64 URL encode.
	 *
	 * @param string $data Data to encode.
	 * @return string Encoded data.
	 */
	private function base64url_encode( $data ) {
		return rtrim( strtr( base64_encode( $data ), '+/', '-_' ), '=' );
	}

	/**
	 * Base64 URL decode.
	 *
	 * @param string $data Data to decode.
	 * @return string Decoded data.
	 */
	private function base64url_decode( $data ) {
		return base64_decode( strtr( $data, '-_', '+/' ) );
	}

	/**
	 * Get current user context.
	 *
	 * @return array|null User context or null if not authenticated.
	 */
	public function get_user_context() {
		return $this->user_context;
	}

	/**
	 * Get JWT signing secret.
	 *
	 * Uses the JWT secret from the cloud, which enables the cloud
	 * to verify user tokens for authenticated MCP operations.
	 *
	 * Falls back to a local secret for backwards compatibility.
	 *
	 * @return string JWT signing secret.
	 */
	private function get_jwt_secret() {
		// Prefer cloud-provided secret (enables authenticated chat).
		$jwt_secret = get_option( INTUFIND_OPTION_JWT_SECRET, '' );
		if ( ! empty( $jwt_secret ) ) {
			return $jwt_secret;
		}

		// Fallback to local secret (legacy, local-only verification).
		$secret = get_option( self::OPTION_JWT_SECRET, '' );

		if ( empty( $secret ) ) {
			// Generate a new secret on first use.
			$secret = wp_generate_password( 64, true, true );
			update_option( self::OPTION_JWT_SECRET, $secret, false );
		}

		return $secret;
	}

	/**
	 * Check rate limit for order lookup.
	 *
	 * Uses transients to track request counts per IP.
	 *
	 * @return bool True if within rate limit, false if exceeded.
	 */
	private function check_rate_limit() {
		$ip = $this->get_client_ip();
		if ( empty( $ip ) ) {
			return true; // Can't rate limit without IP.
		}

		$transient_key = 'intufind_rl_' . md5( $ip );
		$data          = get_transient( $transient_key );

		if ( false === $data ) {
			// First request in this window.
			set_transient(
				$transient_key,
				array(
					'count'  => 1,
					'window' => time(),
				),
				self::RATE_LIMIT_WINDOW
			);
			return true;
		}

		// Check if within limit.
		if ( $data['count'] >= self::RATE_LIMIT_MAX_REQUESTS ) {
			return false;
		}

		// Increment counter.
		$data['count']++;
		set_transient( $transient_key, $data, self::RATE_LIMIT_WINDOW );

		return true;
	}

	/**
	 * Get client IP address.
	 *
	 * Security: Only trusts proxy headers (X-Forwarded-For, etc.) if REMOTE_ADDR
	 * is a private/reserved IP (indicating we're behind a proxy). This prevents
	 * IP spoofing attacks where clients send fake X-Forwarded-For headers.
	 *
	 * @return string Client IP address.
	 */
	private function get_client_ip() {
		// Get the direct connection IP first.
		$remote_addr = isset( $_SERVER['REMOTE_ADDR'] )
			? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) )
			: '';

		// Check if we're behind a proxy (REMOTE_ADDR is private/reserved).
		$behind_proxy = $remote_addr && ! filter_var(
			$remote_addr,
			FILTER_VALIDATE_IP,
			FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE
		);

		// Only trust forwarded headers if behind a proxy.
		if ( $behind_proxy ) {
			$forwarded_headers = array(
				'HTTP_CF_CONNECTING_IP', // Cloudflare (most specific).
				'HTTP_X_REAL_IP',        // Nginx.
				'HTTP_X_FORWARDED_FOR',  // Generic proxy (may contain chain).
			);

			foreach ( $forwarded_headers as $header ) {
				if ( ! empty( $_SERVER[ $header ] ) ) {
					// X-Forwarded-For can contain multiple IPs - use the first (client).
					$ip = explode( ',', sanitize_text_field( wp_unslash( $_SERVER[ $header ] ) ) )[0];
					$ip = trim( $ip );
					if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) ) {
						return $ip;
					}
				}
			}
		}

		// Fall back to REMOTE_ADDR (direct connection or proxy not detected).
		if ( $remote_addr && filter_var( $remote_addr, FILTER_VALIDATE_IP ) ) {
			return $remote_addr;
		}

		return '';
	}

	/**
	 * Handle main MCP request.
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response
	 */
	public function handle_mcp_request( \WP_REST_Request $request ) {
		// Handle OPTIONS preflight.
		if ( 'OPTIONS' === $request->get_method() ) {
			return new \WP_REST_Response( null, 204 );
		}

		$body = $request->get_json_params();
		if ( empty( $body ) ) {
			return $this->error_response( 0, -32700, 'Parse error - Invalid JSON' );
		}

		return $this->process_jsonrpc( $body, null );
	}

	/**
	 * Handle domain-specific MCP request.
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response
	 */
	public function handle_domain_request( \WP_REST_Request $request ) {
		$domain = $request->get_param( 'domain' );

		// Validate domain.
		$available_domains = $this->get_available_domains();
		if ( ! in_array( $domain, $available_domains, true ) ) {
			return $this->error_response( 0, -32602, "Domain '$domain' not found" );
		}

		// Handle OPTIONS preflight.
		if ( 'OPTIONS' === $request->get_method() ) {
			return new \WP_REST_Response( null, 204 );
		}

		$body = $request->get_json_params();
		if ( empty( $body ) ) {
			return $this->error_response( 0, -32700, 'Parse error - Invalid JSON' );
		}

		return $this->process_jsonrpc( $body, $domain );
	}

	/**
	 * Process JSON-RPC request.
	 *
	 * @param array       $body   Request body.
	 * @param string|null $domain Domain name or null.
	 * @return \WP_REST_Response
	 */
	private function process_jsonrpc( $body, $domain ) {
		// Validate JSON-RPC format.
		if ( ! isset( $body['jsonrpc'] ) || '2.0' !== $body['jsonrpc'] ) {
			return $this->error_response( 0, -32600, 'Invalid JSON-RPC version' );
		}

		if ( ! isset( $body['method'] ) || ! isset( $body['id'] ) ) {
			return $this->error_response( 0, -32600, 'Missing required fields' );
		}

		$method = $body['method'];
		$params = $body['params'] ?? array();
		$id     = $body['id'];

		try {
			$result = $this->handle_method( $method, $params, $domain );
			return $this->success_response( $id, $result );
		} catch ( \Exception $e ) {
			return $this->error_response( $id, -32603, $e->getMessage() );
		}
	}

	/**
	 * Handle JSON-RPC method.
	 *
	 * @param string      $method Method name.
	 * @param array       $params Method parameters.
	 * @param string|null $domain Domain name.
	 * @return array
	 * @throws \Exception On unknown method.
	 */
	private function handle_method( $method, $params, $domain ) {
		switch ( $method ) {
			case 'initialize':
				return $this->handle_initialize( $domain );

			case 'tools/list':
				return $this->handle_tools_list( $domain );

			case 'tools/call':
				return $this->handle_tools_call( $params, $domain );

			case 'ping':
				return array(
					'status'    => 'ok',
					'timestamp' => time(),
				);

			default:
				throw new \Exception( "Method not found: $method" );
		}
	}

	/**
	 * Handle initialize method.
	 *
	 * @param string|null $domain Domain name.
	 * @return array
	 */
	private function handle_initialize( $domain ) {
		return array(
			'protocolVersion' => '2024-11-05',
			'capabilities'    => array(
				'tools' => array(
					'listChanged' => false,
				),
			),
			'serverInfo'      => array(
				'name'    => 'Intufind MCP Server',
				'version' => INTUFIND_VERSION,
			),
			'availableDomains' => $this->get_available_domains(),
		);
	}

	/**
	 * Handle tools/list method.
	 *
	 * @param string|null $domain Domain name.
	 * @return array
	 */
	private function handle_tools_list( $domain ) {
		$tools = array();

		// Orders domain (only if WooCommerce active).
		if ( ( null === $domain || 'orders' === $domain ) && class_exists( 'WooCommerce' ) ) {
			$tools = array_merge( $tools, $this->get_orders_tools() );
		}

		return array( 'tools' => $tools );
	}

	/**
	 * Handle tools/call method.
	 *
	 * @param array       $params Method parameters.
	 * @param string|null $domain Domain name.
	 * @return array
	 * @throws \Exception On tool execution error.
	 */
	private function handle_tools_call( $params, $domain ) {
		$tool_name = $params['name'] ?? '';
		$arguments = $params['arguments'] ?? array();

		// Execute tool based on name.
		$result = $this->execute_tool( $tool_name, $arguments );

		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => wp_json_encode( $result ),
				),
			),
		);
	}

	/**
	 * Execute a tool.
	 *
	 * @param string $tool_name Tool name.
	 * @param array  $arguments Tool arguments.
	 * @return array
	 * @throws \Exception On unknown tool.
	 */
	private function execute_tool( $tool_name, $arguments ) {
		switch ( $tool_name ) {
			case 'get_my_orders':
				return $this->tool_get_my_orders( $arguments );

			case 'lookup_order':
				return $this->tool_lookup_order( $arguments );

			case 'get_order_details':
				return $this->tool_get_order_details( $arguments );

			default:
				throw new \Exception( "Unknown tool: $tool_name" );
		}
	}

	/**
	 * Get available domains.
	 *
	 * @return array
	 */
	public function get_available_domains() {
		$domains = array();

		// Orders domain only available with WooCommerce.
		if ( class_exists( 'WooCommerce' ) ) {
			$domains[] = 'orders';
		}

		return $domains;
	}

	/**
	 * Get orders tools definition.
	 *
	 * Tools are designed to prevent cross-user data leakage:
	 * - get_my_orders: Requires authenticated user (JWT)
	 * - lookup_order: For guests, requires order number + email verification
	 * - get_order_details: Requires ownership verification
	 *
	 * @return array
	 */
	private function get_orders_tools() {
		return array(
			array(
				'name'        => 'get_my_orders',
				'description' => 'Get the current user\'s orders. Only works for logged-in users.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'limit'  => array(
							'type'        => 'integer',
							'description' => 'Number of orders to retrieve (max 20)',
							'default'     => 10,
						),
						'status' => array(
							'type'        => 'array',
							'items'       => array( 'type' => 'string' ),
							'description' => 'Filter by order status',
						),
					),
				),
			),
			array(
				'name'        => 'lookup_order',
				'description' => 'Look up an order by order number and email. Use this for guest users who need to check their order status.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'order_number' => array(
							'type'        => 'string',
							'description' => 'The order number (e.g., "12345" or "#12345")',
						),
						'email'        => array(
							'type'        => 'string',
							'description' => 'The email address used for the order',
						),
					),
					'required'   => array( 'order_number', 'email' ),
				),
			),
			array(
				'name'        => 'get_order_details',
				'description' => 'Get detailed information about a specific order. Requires prior ownership verification via lookup_order or get_my_orders.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'order_id' => array(
							'type'        => 'integer',
							'description' => 'Order ID (from lookup_order or get_my_orders)',
						),
						'email'    => array(
							'type'        => 'string',
							'description' => 'Email for ownership verification (required for guests)',
						),
					),
					'required'   => array( 'order_id' ),
				),
			),
		);
	}

	/**
	 * Tool: Get current user's orders.
	 *
	 * Requires authenticated user context (JWT). Returns only orders
	 * belonging to the authenticated user.
	 *
	 * @param array $args Tool arguments.
	 * @return array
	 */
	private function tool_get_my_orders( $args ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return array( 'error' => 'WooCommerce not active' );
		}

		// Require user authentication.
		$user_context = $this->get_user_context();
		if ( empty( $user_context ) || empty( $user_context['email'] ) ) {
			return array(
				'error'   => 'Authentication required',
				'message' => 'This tool requires user authentication. Please ask the customer to log in to view their orders, or use lookup_order with their order number and email.',
			);
		}

		$query_args = array(
			'limit'    => min( $args['limit'] ?? 10, 20 ),
			'orderby'  => 'date',
			'order'    => 'DESC',
			'customer' => $user_context['email'], // Filter by customer email.
		);

		if ( ! empty( $args['status'] ) ) {
			$query_args['status'] = $args['status'];
		}

		$orders    = wc_get_orders( $query_args );
		$formatted = array();

		foreach ( $orders as $order ) {
			$formatted[] = $this->format_order_summary( $order );
		}

		return array(
			'orders'  => $formatted,
			'count'   => count( $formatted ),
			'message' => count( $formatted ) > 0
				? sprintf( 'Found %d order(s) for %s', count( $formatted ), $user_context['name'] ?? $user_context['email'] )
				: 'No orders found for this account.',
		);
	}

	/**
	 * Tool: Look up order by order number and email.
	 *
	 * For guest users who need to check their order status.
	 * Requires both order number and email to match for verification.
	 * Rate limited to prevent brute-force enumeration attacks.
	 *
	 * @param array $args Tool arguments.
	 * @return array
	 */
	private function tool_lookup_order( $args ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return array( 'error' => 'WooCommerce not active' );
		}

		// Rate limit to prevent brute-force attacks.
		if ( ! $this->check_rate_limit() ) {
			// Log rate limit violation for security monitoring.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( '[Intufind] Rate limit exceeded for order lookup from IP: ' . $this->get_client_ip() );
			}
			return array(
				'error'   => 'Rate limit exceeded',
				'message' => 'Too many requests. Please try again later.',
			);
		}

		$order_number = $args['order_number'] ?? '';
		$email        = $args['email'] ?? '';

		if ( empty( $order_number ) || empty( $email ) ) {
			return array(
				'error'   => 'Missing required information',
				'message' => 'Please provide both the order number and the email address used for the order.',
			);
		}

		// Basic email format validation.
		if ( ! is_email( $email ) ) {
			return array(
				'error'   => 'Invalid email format',
				'message' => 'Please provide a valid email address.',
			);
		}

		// Clean up and validate order number (remove # prefix, limit length).
		$order_number = ltrim( $order_number, '#' );
		$order_number = substr( $order_number, 0, 20 ); // Limit length.

		// Try to get order by ID/number.
		$order = wc_get_order( $order_number );

		if ( ! $order ) {
			// Try searching by order number meta.
			$orders = wc_get_orders(
				array(
					'meta_key'   => '_order_number',
					'meta_value' => $order_number,
					'limit'      => 1,
				)
			);
			$order = ! empty( $orders ) ? $orders[0] : null;
		}

		// Use same error message for both "not found" and "email mismatch" to prevent enumeration.
		$generic_not_found = array(
			'error'   => 'Order not found',
			'message' => 'We could not find an order matching that order number and email combination. Please verify both are correct.',
		);

		if ( ! $order ) {
			return $generic_not_found;
		}

		// Verify email matches (case-insensitive).
		$order_email = strtolower( $order->get_billing_email() );
		$input_email = strtolower( trim( $email ) );

		if ( $order_email !== $input_email ) {
			// Return same error as "not found" to prevent order ID enumeration.
			return $generic_not_found;
		}

		// Return order summary (not full details for initial lookup).
		return array(
			'order'   => $this->format_order_summary( $order ),
			'message' => sprintf(
				'Found order #%s placed on %s. Status: %s.',
				$order->get_id(),
				$order->get_date_created()->format( 'F j, Y' ),
				wc_get_order_status_name( $order->get_status() )
			),
		);
	}

	/**
	 * Tool: Get order details.
	 *
	 * Returns detailed information about an order after ownership verification.
	 *
	 * @param array $args Tool arguments.
	 * @return array
	 */
	private function tool_get_order_details( $args ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return array( 'error' => 'WooCommerce not active' );
		}

		$order_id = $args['order_id'] ?? 0;
		$email    = $args['email'] ?? '';

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return array( 'error' => 'Order not found' );
		}

		// Verify ownership - check user context or email.
		$user_context = $this->get_user_context();
		$order_email  = strtolower( $order->get_billing_email() );

		$is_owner = false;

		// Check if authenticated user owns this order.
		if ( ! empty( $user_context['email'] ) ) {
			$is_owner = strtolower( $user_context['email'] ) === $order_email;
		}

		// Check if provided email matches (for guests).
		if ( ! $is_owner && ! empty( $email ) ) {
			$is_owner = strtolower( trim( $email ) ) === $order_email;
		}

		if ( ! $is_owner ) {
			return array(
				'error'   => 'Access denied',
				'message' => 'You do not have permission to view this order. Please verify the email address used for this order.',
			);
		}

		// Return full order details.
		$items = array();
		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();
			$items[] = array(
				'name'       => $item->get_name(),
				'quantity'   => $item->get_quantity(),
				'total'      => $order->get_currency() . ' ' . number_format( (float) $item->get_total(), 2 ),
				'product_id' => $product ? $product->get_id() : null,
				'sku'        => $product ? $product->get_sku() : null,
			);
		}

		return array(
			'order' => array(
				'id'             => $order->get_id(),
				'status'         => $order->get_status(),
				'status_label'   => wc_get_order_status_name( $order->get_status() ),
				'total'          => $order->get_currency() . ' ' . $order->get_total(),
				'date_created'   => $order->get_date_created()->format( 'F j, Y g:i a' ),
				'billing'        => array(
					'name'  => trim( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() ),
					'email' => $order->get_billing_email(),
					'phone' => $order->get_billing_phone(),
				),
				'shipping'       => array(
					'address' => $order->get_formatted_shipping_address() ?: 'Same as billing',
					'method'  => $order->get_shipping_method() ?: 'N/A',
				),
				'payment_method' => $order->get_payment_method_title(),
				'items'          => $items,
				'notes'          => $this->get_customer_order_notes( $order ),
			),
		);
	}

	/**
	 * Format order summary (minimal info for listings).
	 *
	 * @param \WC_Order $order Order object.
	 * @return array
	 */
	private function format_order_summary( $order ) {
		return array(
			'id'           => $order->get_id(),
			'status'       => $order->get_status(),
			'status_label' => wc_get_order_status_name( $order->get_status() ),
			'total'        => $order->get_currency() . ' ' . $order->get_total(),
			'date_created' => $order->get_date_created()->format( 'F j, Y' ),
			'item_count'   => $order->get_item_count(),
		);
	}

	/**
	 * Get customer-visible order notes.
	 *
	 * @param \WC_Order $order Order object.
	 * @return array
	 */
	private function get_customer_order_notes( $order ) {
		$notes = wc_get_order_notes(
			array(
				'order_id' => $order->get_id(),
				'type'     => 'customer',
			)
		);

		$formatted = array();
		foreach ( $notes as $note ) {
			$formatted[] = array(
				'date'    => $note->date_created->format( 'F j, Y g:i a' ),
				'content' => $note->content,
			);
		}

		return $formatted;
	}

	/**
	 * Create success response.
	 *
	 * @param int   $id     Request ID.
	 * @param array $result Result data.
	 * @return \WP_REST_Response
	 */
	private function success_response( $id, $result ) {
		return new \WP_REST_Response(
			array(
				'jsonrpc' => '2.0',
				'id'      => $id,
				'result'  => $result,
			),
			200
		);
	}

	/**
	 * Create error response.
	 *
	 * @param int    $id      Request ID.
	 * @param int    $code    Error code.
	 * @param string $message Error message.
	 * @return \WP_REST_Response
	 */
	private function error_response( $id, $code, $message ) {
		return new \WP_REST_Response(
			array(
				'jsonrpc' => '2.0',
				'id'      => $id,
				'error'   => array(
					'code'    => $code,
					'message' => $message,
				),
			),
			400
		);
	}

	/**
	 * Get MCP server status.
	 *
	 * @return array
	 */
	public function get_status() {
		$domains = $this->get_available_domains();

		return array(
			'status'    => 'active',
			'domains'   => $domains,
			'endpoints' => array(
				'base'          => rest_url( self::REST_NAMESPACE . '/mcp' ),
				'domain_orders' => class_exists( 'WooCommerce' ) ? rest_url( self::REST_NAMESPACE . '/mcp/domain/orders' ) : null,
			),
		);
	}
}
