<?php
/**
 * Main plugin class.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main Intufind plugin class.
 *
 * Orchestrates all plugin functionality including admin pages,
 * API communication, content sync, and MCP server.
 */
class Intufind_Plugin {

	/**
	 * API client instance.
	 *
	 * @var Intufind_API|null
	 */
	private $api = null;

	/**
	 * Admin instance.
	 *
	 * @var Intufind_Admin|null
	 */
	private $admin = null;

	/**
	 * Sync manager instance.
	 *
	 * @var Intufind_Sync|null
	 */
	private $sync = null;

	/**
	 * MCP server instance.
	 *
	 * @var Intufind_MCP|null
	 */
	private $mcp = null;

	/**
	 * Content extractor instance.
	 *
	 * @var Intufind_Content_Extractor|null
	 */
	private $extractor = null;

	/**
	 * Exclusions manager instance.
	 *
	 * @var Intufind_Exclusions|null
	 */
	private $exclusions = null;

	/**
	 * Sync status tracker instance.
	 *
	 * @var Intufind_Sync_Status|null
	 */
	private $sync_status = null;

	/**
	 * List columns handler instance.
	 *
	 * @var Intufind_List_Columns|null
	 */
	private $list_columns = null;

	/**
	 * Search override instance.
	 *
	 * @var Intufind_Search_Override|null
	 */
	private $search_override = null;

	/**
	 * Recommendations override instance.
	 *
	 * @var Intufind_Recommendations_Override|null
	 */
	private $recommendations_override = null;

	/**
	 * Chat widget instance.
	 *
	 * @var Intufind_Chat_Widget|null
	 */
	private $chat_widget = null;

	/**
	 * Initialize the plugin.
	 *
	 * @return void
	 */
	public function init() {
		// Initialize API client.
		$this->api = new Intufind_API();

		// Initialize document sync components.
		$this->extractor   = new Intufind_Content_Extractor();
		$this->exclusions  = new Intufind_Exclusions();
		$this->sync_status = new Intufind_Sync_Status();

		// Initialize sync manager with all dependencies.
		$this->sync = new Intufind_Sync(
			$this->api,
			$this->extractor,
			$this->exclusions,
			$this->sync_status
		);
		$this->sync->init();

		// Initialize admin if in admin context.
		if ( is_admin() ) {
			$this->admin = new Intufind_Admin( $this->api, $this->sync );
			$this->admin->init();

			// Initialize list columns for post tables.
			$this->list_columns = new Intufind_List_Columns( $this->exclusions, $this->sync_status, $this->api, $this->extractor );
			$this->list_columns->init();
		}

		// Initialize MCP server (REST API).
		$this->mcp = new Intufind_MCP( $this->api );
		$this->mcp->init();

		// Initialize search override on frontend.
		$this->search_override = new Intufind_Search_Override( $this->api );
		$this->search_override->init();

		// Initialize recommendations override (WooCommerce only).
		if ( class_exists( 'WooCommerce' ) ) {
			$this->recommendations_override = new Intufind_Recommendations_Override( $this->api );
			$this->recommendations_override->init();
		}

		// Initialize chat widget on frontend.
		$this->chat_widget = new Intufind_Chat_Widget( $this->api );
		$this->chat_widget->init();
	}

	/**
	 * Output user JWT for the chat widget.
	 *
	 * When a user is logged in, this outputs a JWT that the widget
	 * can use for authenticated MCP operations (e.g., viewing their orders).
	 *
	 * Merges into window.intufindConfig which is the standard widget config object.
	 *
	 * @return void
	 */
	public function output_widget_user_context() {
		// Only output if plugin is connected.
		if ( ! $this->is_connected() ) {
			return;
		}

		// Only output for logged-in users.
		if ( ! is_user_logged_in() ) {
			return;
		}

		// Get user token from MCP service.
		$user_token = $this->mcp->get_user_token_for_widget();
		if ( ! $user_token ) {
			return;
		}

		// Merge userJwt into the standard intufindConfig object.
		// This runs early in <head> before the widget loader.
		printf(
			'<script id="intufind-user-jwt">window.intufindConfig=window.intufindConfig||{};window.intufindConfig.userJwt=%s;</script>',
			wp_json_encode( $user_token['token'] )
		);
	}

	/**
	 * Get the API client instance.
	 *
	 * @return Intufind_API
	 */
	public function get_api() {
		return $this->api;
	}

	/**
	 * Check if the plugin is connected to the cloud.
	 *
	 * @return bool
	 */
	public function is_connected() {
		return (bool) get_option( INTUFIND_OPTION_CONNECTED, false );
	}

	/**
	 * Get the current workspace ID.
	 *
	 * @return string|null
	 */
	public function get_workspace_id() {
		$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		return ! empty( $workspace_id ) ? $workspace_id : null;
	}

	/**
	 * Generate a workspace ID from the site domain.
	 *
	 * Creates a URL-safe identifier from the site's domain name.
	 * Example: "www.my-store.com" becomes "my-store-com"
	 *
	 * @return string Generated workspace ID.
	 */
	public static function generate_workspace_id() {
		$host = wp_parse_url( home_url(), PHP_URL_HOST );

		// Remove www. prefix.
		$host = preg_replace( '/^www\./', '', $host );

		// Replace dots with hyphens and sanitize.
		$workspace_id = sanitize_title( str_replace( '.', '-', $host ) );

		return $workspace_id;
	}
}
