<?php
/**
 * WooCommerce Recommendations Override.
 *
 * Replaces WooCommerce native recommendations (related, upsells, cross-sells)
 * with AI-powered recommendations from the Intufind cloud.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Intufind Recommendations Override class.
 *
 * Hooks into WooCommerce recommendation filters to provide
 * AI-powered product recommendations.
 */
class Intufind_Recommendations_Override {

	/**
	 * Option names for recommendations settings.
	 */
	const OPTION_ENABLE_RELATED     = 'intufind_enable_related_products';
	const OPTION_ENABLE_UPSELLS     = 'intufind_enable_upsells';
	const OPTION_ENABLE_CROSS_SELLS = 'intufind_enable_cross_sells';
	const OPTION_MAX_RECOMMENDATIONS = 'intufind_max_recommendations';
	const OPTION_CACHE_DURATION     = 'intufind_recommendations_cache_duration';

	/**
	 * Cache group for recommendations.
	 */
	const CACHE_GROUP = 'intufind_recommendations';

	/**
	 * API client instance.
	 *
	 * @var Intufind_API
	 */
	private $api;

	/**
	 * Constructor.
	 *
	 * @param Intufind_API $api API client instance.
	 */
	public function __construct( Intufind_API $api ) {
		$this->api = $api;
	}

	/**
	 * Initialize hooks.
	 *
	 * @return void
	 */
	public function init() {
		// Only run on frontend and when WooCommerce is active.
		if ( is_admin() || ! class_exists( 'WooCommerce' ) ) {
			return;
		}

		// Only if connected.
		if ( ! get_option( INTUFIND_OPTION_CONNECTED, false ) ) {
			return;
		}

		// Register hooks based on settings.
		if ( $this->is_related_enabled() ) {
			add_filter( 'woocommerce_related_products', array( $this, 'get_related_products' ), 10, 3 );
		}

		if ( $this->is_upsells_enabled() ) {
			add_filter( 'woocommerce_product_get_upsell_ids', array( $this, 'get_upsell_products' ), 10, 2 );
		}

		if ( $this->is_cross_sells_enabled() ) {
			add_filter( 'woocommerce_product_get_cross_sell_ids', array( $this, 'get_cross_sell_products' ), 10, 2 );
		}
	}

	/**
	 * Check if related products override is enabled.
	 *
	 * @return bool
	 */
	public function is_related_enabled() {
		return (bool) get_option( self::OPTION_ENABLE_RELATED, false );
	}

	/**
	 * Check if upsells override is enabled.
	 *
	 * @return bool
	 */
	public function is_upsells_enabled() {
		return (bool) get_option( self::OPTION_ENABLE_UPSELLS, false );
	}

	/**
	 * Check if cross-sells override is enabled.
	 *
	 * @return bool
	 */
	public function is_cross_sells_enabled() {
		return (bool) get_option( self::OPTION_ENABLE_CROSS_SELLS, false );
	}

	/**
	 * Get maximum recommendations count.
	 *
	 * @return int
	 */
	public function get_max_recommendations() {
		return (int) get_option( self::OPTION_MAX_RECOMMENDATIONS, 4 );
	}

	/**
	 * Get cache duration in seconds.
	 *
	 * @return int
	 */
	public function get_cache_duration() {
		return (int) get_option( self::OPTION_CACHE_DURATION, 3600 );
	}

	/**
	 * Get related products using AI recommendations.
	 *
	 * @param array $related_ids Default related product IDs.
	 * @param int   $product_id  Current product ID.
	 * @param array $args        Query arguments.
	 * @return array Product IDs.
	 */
	public function get_related_products( $related_ids, $product_id, $args ) {
		$max_items = isset( $args['posts_per_page'] ) ? (int) $args['posts_per_page'] : $this->get_max_recommendations();

		$recommendations = $this->fetch_recommendations( $product_id, 'related', $max_items, $related_ids );

		if ( false === $recommendations ) {
			return $related_ids; // Fallback to native.
		}

		return $recommendations;
	}

	/**
	 * Get upsell products using AI recommendations.
	 *
	 * @param array      $upsell_ids Default upsell IDs.
	 * @param WC_Product $product    Current product.
	 * @return array Product IDs.
	 */
	public function get_upsell_products( $upsell_ids, $product ) {
		$product_id = is_object( $product ) ? $product->get_id() : (int) $product;

		if ( ! $product_id ) {
			return $upsell_ids;
		}

		$max_items = apply_filters( 'woocommerce_upsells_total', $this->get_max_recommendations() );

		// Preserve manually set upsells, fill remaining slots with AI.
		$manual_upsells  = array_slice( $upsell_ids, 0, $max_items );
		$remaining_slots = $max_items - count( $manual_upsells );

		if ( $remaining_slots <= 0 ) {
			return $manual_upsells;
		}

		$recommendations = $this->fetch_recommendations(
			$product_id,
			'upsell',
			$remaining_slots,
			array_merge( $manual_upsells, array( $product_id ) )
		);

		if ( false === $recommendations ) {
			return $upsell_ids; // Fallback to native.
		}

		// Merge manual upsells with AI recommendations.
		return array_merge( $manual_upsells, $recommendations );
	}

	/**
	 * Get cross-sell products using AI recommendations.
	 *
	 * @param array      $cross_sell_ids Default cross-sell IDs.
	 * @param WC_Product $product        Current product.
	 * @return array Product IDs.
	 */
	public function get_cross_sell_products( $cross_sell_ids, $product ) {
		$product_id = is_object( $product ) ? $product->get_id() : (int) $product;

		if ( ! $product_id ) {
			return $cross_sell_ids;
		}

		$max_items = apply_filters( 'woocommerce_cross_sells_total', $this->get_max_recommendations() );

		// Preserve manually set cross-sells, fill remaining slots with AI.
		$manual_cross_sells = array_slice( $cross_sell_ids, 0, $max_items );
		$remaining_slots    = $max_items - count( $manual_cross_sells );

		if ( $remaining_slots <= 0 ) {
			return $manual_cross_sells;
		}

		$recommendations = $this->fetch_recommendations(
			$product_id,
			'cross_sell',
			$remaining_slots,
			array_merge( $manual_cross_sells, array( $product_id ) )
		);

		if ( false === $recommendations ) {
			return $cross_sell_ids; // Fallback to native.
		}

		// Merge manual cross-sells with AI recommendations.
		return array_merge( $manual_cross_sells, $recommendations );
	}

	/**
	 * Fetch recommendations from API with caching.
	 *
	 * @param int    $product_id  Product ID.
	 * @param string $context     Context (related, upsell, cross_sell).
	 * @param int    $limit       Maximum recommendations.
	 * @param array  $exclude_ids IDs to exclude.
	 * @return array|false Product IDs or false on failure.
	 */
	private function fetch_recommendations( $product_id, $context, $limit, $exclude_ids = array() ) {
		// Try cache first.
		$cache_key = $this->get_cache_key( $product_id, $context, $limit );
		$cached    = wp_cache_get( $cache_key, self::CACHE_GROUP );

		if ( false !== $cached ) {
			// Filter out excluded IDs from cached results.
			return array_values( array_diff( $cached, $exclude_ids ) );
		}

		// Get current user for personalization.
		$user_id = get_current_user_id();

		// Call API.
		$result = $this->api->get_recommendations( $product_id, $limit + count( $exclude_ids ), $exclude_ids, $user_id );

		if ( is_wp_error( $result ) ) {
			$this->log_error( "Recommendations API error ($context)", $result );
			return false;
		}

		// Extract product IDs from results.
		$ids = $this->extract_ids_from_results( $result );

		if ( empty( $ids ) ) {
			return false;
		}

		// Filter out excluded IDs.
		$ids = array_values( array_diff( $ids, $exclude_ids ) );

		// Limit to requested count.
		$ids = array_slice( $ids, 0, $limit );

		// Cache the full results (before exclusion filtering).
		$cache_duration = $this->get_cache_duration();
		if ( $cache_duration > 0 ) {
			wp_cache_set( $cache_key, $ids, self::CACHE_GROUP, $cache_duration );
		}

		return $ids;
	}

	/**
	 * Extract product IDs from API results.
	 *
	 * @param array $result API response.
	 * @return array Product IDs.
	 */
	private function extract_ids_from_results( $result ) {
		$ids = array();

		// Handle different response structures.
		$recommendations = array();
		if ( isset( $result['data']['recommendations'] ) ) {
			$recommendations = $result['data']['recommendations'];
		} elseif ( isset( $result['recommendations'] ) ) {
			$recommendations = $result['recommendations'];
		}

		foreach ( $recommendations as $rec ) {
			$id = null;

			// Try different ID locations.
			if ( isset( $rec['product']['id'] ) ) {
				$id = $rec['product']['id'];
			} elseif ( isset( $rec['productId'] ) ) {
				$id = $rec['productId'];
			} elseif ( isset( $rec['id'] ) ) {
				$id = $rec['id'];
			}

			if ( $id ) {
				// Handle "product_123" format.
				if ( preg_match( '/^product_(\d+)$/', $id, $matches ) ) {
					$ids[] = (int) $matches[1];
				} elseif ( is_numeric( $id ) ) {
					$ids[] = (int) $id;
				}
			}
		}

		return $ids;
	}

	/**
	 * Generate cache key.
	 *
	 * @param int    $product_id Product ID.
	 * @param string $context    Context.
	 * @param int    $limit      Limit.
	 * @return string Cache key.
	 */
	private function get_cache_key( $product_id, $context, $limit ) {
		$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		return sprintf( '%s_%s_%d_%s_%d', self::CACHE_GROUP, $workspace_id, $product_id, $context, $limit );
	}

	/**
	 * Log an error.
	 *
	 * @param string   $message Error message.
	 * @param WP_Error $error   WP_Error object.
	 * @return void
	 */
	private function log_error( $message, $error ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( sprintf( 'Intufind Recommendations: %s - %s', $message, $error->get_error_message() ) );
		}
	}

	/**
	 * Clear recommendations cache for a product.
	 *
	 * @param int $product_id Product ID.
	 * @return void
	 */
	public function clear_product_cache( $product_id ) {
		$contexts = array( 'related', 'upsell', 'cross_sell' );
		$limits   = array( 4, 6, 8, 10 ); // Common limits.

		foreach ( $contexts as $context ) {
			foreach ( $limits as $limit ) {
				$cache_key = $this->get_cache_key( $product_id, $context, $limit );
				wp_cache_delete( $cache_key, self::CACHE_GROUP );
			}
		}
	}

	/**
	 * Clear all recommendations cache.
	 *
	 * @return void
	 */
	public function clear_all_cache() {
		wp_cache_flush_group( self::CACHE_GROUP );
	}
}
