<?php
/**
 * Search override functionality.
 *
 * Replaces default WordPress and WooCommerce search with
 * Intufind semantic search for better relevance.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Intufind Search Override class.
 *
 * Hooks into WordPress and WooCommerce search queries to provide
 * semantic search results from the Intufind cloud API.
 */
class Intufind_Search_Override {

	/**
	 * Option names for search settings.
	 */
	const OPTION_ENABLE_WP_SEARCH      = 'intufind_enable_wp_search';
	const OPTION_ENABLE_WOO_SEARCH     = 'intufind_enable_woo_search';
	const OPTION_ENABLE_MIXED_SEARCH   = 'intufind_enable_mixed_search';
	const OPTION_SEARCH_LIMIT          = 'intufind_search_limit';
	const OPTION_SEARCH_CACHE_DURATION = 'intufind_search_cache_duration';

	/**
	 * Cache group for search results.
	 */
	const CACHE_GROUP = 'intufind_search';

	/**
	 * API client instance.
	 *
	 * @var Intufind_API
	 */
	private $api;

	/**
	 * Original search term storage.
	 *
	 * @var string
	 */
	private $original_search_term = '';

	/**
	 * Constructor.
	 *
	 * @param Intufind_API $api API client instance.
	 */
	public function __construct( Intufind_API $api ) {
		$this->api = $api;
	}

	/**
	 * Initialize hooks.
	 *
	 * @return void
	 */
	public function init() {
		// Only run on frontend.
		if ( is_admin() ) {
			return;
		}

		// Only if connected.
		if ( ! get_option( INTUFIND_OPTION_CONNECTED, false ) ) {
			return;
		}

		// Hook into WordPress search if enabled.
		if ( $this->is_wp_search_enabled() ) {
			add_action( 'pre_get_posts', array( $this, 'modify_search_query' ), 999 );
		}

		// Hook into WooCommerce search if enabled and WooCommerce is active.
		if ( $this->is_woo_search_enabled() && class_exists( 'WooCommerce' ) ) {
			add_filter( 'woocommerce_product_query', array( $this, 'modify_product_query' ), 999 );
		}
	}

	/**
	 * Check if WordPress search override is enabled.
	 *
	 * @return bool
	 */
	public function is_wp_search_enabled() {
		return (bool) get_option( self::OPTION_ENABLE_WP_SEARCH, false );
	}

	/**
	 * Check if WooCommerce search override is enabled.
	 *
	 * @return bool
	 */
	public function is_woo_search_enabled() {
		return (bool) get_option( self::OPTION_ENABLE_WOO_SEARCH, false );
	}

	/**
	 * Check if mixed content search is enabled.
	 *
	 * @return bool
	 */
	public function is_mixed_search_enabled() {
		return (bool) get_option( self::OPTION_ENABLE_MIXED_SEARCH, true );
	}

	/**
	 * Get the search result limit.
	 *
	 * @return int
	 */
	public function get_search_limit() {
		return (int) get_option( self::OPTION_SEARCH_LIMIT, 20 );
	}

	/**
	 * Get cache duration in seconds.
	 *
	 * @return int
	 */
	public function get_cache_duration() {
		return (int) get_option( self::OPTION_SEARCH_CACHE_DURATION, 3600 );
	}

	/**
	 * Modify WordPress search query.
	 *
	 * @param WP_Query $query The WordPress query object.
	 * @return void
	 */
	public function modify_search_query( $query ) {
		// Only modify main frontend search queries.
		if ( is_admin() || ! $query->is_main_query() || ! $query->is_search() ) {
			return;
		}

		$search_term = $query->get( 's' );
		if ( empty( $search_term ) ) {
			return;
		}

		// Check post type - if it's explicitly 'product', let WooCommerce handler deal with it.
		$post_type = $query->get( 'post_type' );
		if ( 'product' === $post_type ) {
			return;
		}

		// Determine search type based on post types and settings.
		$is_woocommerce_active = class_exists( 'WooCommerce' );
		$is_mixed_enabled      = $this->is_mixed_search_enabled();

		// If no specific post type and mixed search is enabled, do mixed search.
		if ( ( empty( $post_type ) || 'any' === $post_type ) && $is_woocommerce_active && $is_mixed_enabled ) {
			$this->handle_mixed_search( $query, $search_term );
		} else {
			// Post-only search.
			$this->handle_post_search( $query, $search_term );
		}
	}

	/**
	 * Modify WooCommerce product query.
	 *
	 * @param WC_Query $query The WooCommerce query object.
	 * @return WC_Query
	 */
	public function modify_product_query( $query ) {
		// Only process search queries.
		if ( ! isset( $query->query_vars['s'] ) || empty( $query->query_vars['s'] ) ) {
			return $query;
		}

		$search_term = $query->query_vars['s'];

		// Try to get cached results.
		$cache_key = $this->get_cache_key( 'product', $search_term );
		$cached    = wp_cache_get( $cache_key, self::CACHE_GROUP );

		if ( false !== $cached ) {
			if ( ! empty( $cached['ids'] ) ) {
				$this->apply_product_results( $query, $cached['ids'], $search_term );
			}
			return $query;
		}

		// Call API for semantic search.
		$limit   = $this->get_search_limit();
		$results = $this->api->search_products( $search_term, $limit );

		if ( is_wp_error( $results ) ) {
			// On error, fall back to native search.
			$this->log_error( 'Product search API error', $results );
			return $query;
		}

		// Extract IDs from results.
		$ids = $this->extract_ids_from_results( $results );

		// Cache the results.
		$cache_data = array( 'ids' => $ids );
		wp_cache_set( $cache_key, $cache_data, self::CACHE_GROUP, $this->get_cache_duration() );

		if ( ! empty( $ids ) ) {
			$this->apply_product_results( $query, $ids, $search_term );
		}

		return $query;
	}

	/**
	 * Handle mixed content search (products + posts).
	 *
	 * @param WP_Query $query       The WordPress query object.
	 * @param string   $search_term The search term.
	 * @return void
	 */
	private function handle_mixed_search( $query, $search_term ) {
		// Try to get cached results.
		$cache_key = $this->get_cache_key( 'mixed', $search_term );
		$cached    = wp_cache_get( $cache_key, self::CACHE_GROUP );

		if ( false !== $cached ) {
			if ( ! empty( $cached['ids'] ) ) {
				$this->apply_mixed_results( $query, $cached['ids'], $cached['post_types'], $search_term );
			}
			return;
		}

		$limit = $this->get_search_limit();
		$half  = (int) ceil( $limit / 2 );

		// Search both products and posts.
		$product_results = $this->api->search_products( $search_term, $half );
		$post_results    = $this->api->search_posts( $search_term, $half );

		// Handle errors - fall back to native search.
		if ( is_wp_error( $product_results ) && is_wp_error( $post_results ) ) {
			$this->log_error( 'Mixed search API error (both failed)', $product_results );
			return;
		}

		// Combine results.
		$all_results = array();

		if ( ! is_wp_error( $product_results ) ) {
			$product_ids = $this->extract_ids_from_results( $product_results );
			foreach ( $product_ids as $id ) {
				$all_results[] = array(
					'id'   => $id,
					'type' => 'product',
				);
			}
		}

		if ( ! is_wp_error( $post_results ) ) {
			$post_ids = $this->extract_ids_from_results( $post_results );
			foreach ( $post_ids as $id ) {
				$all_results[] = array(
					'id'   => $id,
					'type' => 'post',
				);
			}
		}

		if ( empty( $all_results ) ) {
			return;
		}

		// Extract IDs and determine post types.
		$ids        = array_column( $all_results, 'id' );
		$post_types = array( 'post', 'page' );
		if ( ! is_wp_error( $product_results ) && ! empty( $product_ids ) ) {
			$post_types[] = 'product';
		}

		// Cache the results.
		$cache_data = array(
			'ids'        => $ids,
			'post_types' => $post_types,
		);
		wp_cache_set( $cache_key, $cache_data, self::CACHE_GROUP, $this->get_cache_duration() );

		$this->apply_mixed_results( $query, $ids, $post_types, $search_term );
	}

	/**
	 * Handle post-only search.
	 *
	 * @param WP_Query $query       The WordPress query object.
	 * @param string   $search_term The search term.
	 * @return void
	 */
	private function handle_post_search( $query, $search_term ) {
		// Try to get cached results.
		$cache_key = $this->get_cache_key( 'post', $search_term );
		$cached    = wp_cache_get( $cache_key, self::CACHE_GROUP );

		if ( false !== $cached ) {
			if ( ! empty( $cached['ids'] ) ) {
				$this->apply_post_results( $query, $cached['ids'], $search_term );
			}
			return;
		}

		$limit   = $this->get_search_limit();
		$results = $this->api->search_posts( $search_term, $limit );

		if ( is_wp_error( $results ) ) {
			$this->log_error( 'Post search API error', $results );
			return;
		}

		$ids = $this->extract_ids_from_results( $results );

		// Cache the results.
		$cache_data = array( 'ids' => $ids );
		wp_cache_set( $cache_key, $cache_data, self::CACHE_GROUP, $this->get_cache_duration() );

		if ( ! empty( $ids ) ) {
			$this->apply_post_results( $query, $ids, $search_term );
		}
	}

	/**
	 * Extract IDs from API results.
	 *
	 * @param array $results API response.
	 * @return array Array of IDs.
	 */
	private function extract_ids_from_results( $results ) {
		$ids = array();

		if ( ! isset( $results['data']['results'] ) || ! is_array( $results['data']['results'] ) ) {
			return $ids;
		}

		foreach ( $results['data']['results'] as $result ) {
			// ID can be at result.id (current API format), result._id, or result._source.id (legacy formats).
			$id = null;
			if ( isset( $result['id'] ) ) {
				$id = $result['id'];
			} elseif ( isset( $result['_id'] ) ) {
				$id = $result['_id'];
			} elseif ( isset( $result['_source']['id'] ) ) {
				$id = $result['_source']['id'];
			}

			if ( $id ) {
				// Extract numeric ID if it's in format like "post_123" or "product_456".
				if ( preg_match( '/^(?:post|product|page)_(\d+)$/', $id, $matches ) ) {
					$ids[] = (int) $matches[1];
				} elseif ( is_numeric( $id ) ) {
					$ids[] = (int) $id;
				}
			}
		}

		return $ids;
	}

	/**
	 * Apply product search results to WooCommerce query.
	 *
	 * @param WC_Query $query       The WooCommerce query object.
	 * @param array    $ids         Array of product IDs.
	 * @param string   $search_term Original search term.
	 * @return void
	 */
	private function apply_product_results( $query, $ids, $search_term ) {
		// Store original search term.
		$query->query_vars['original_search_term'] = $search_term;

		// Set post__in to limit results.
		$query->query_vars['post__in'] = $ids;

		// Clear the search term to prevent default search.
		$query->query_vars['s'] = '';

		// Preserve semantic search order.
		$query->query_vars['orderby'] = 'post__in';

		// Add filter to restore search term for display.
		$this->original_search_term = $search_term;
		add_filter( 'the_search_query', array( $this, 'restore_search_term' ) );
		add_filter( 'get_search_query', array( $this, 'restore_search_term' ) );
	}

	/**
	 * Apply mixed search results to WordPress query.
	 *
	 * @param WP_Query $query       The WordPress query object.
	 * @param array    $ids         Array of post/product IDs.
	 * @param array    $post_types  Post types to include.
	 * @param string   $search_term Original search term.
	 * @return void
	 */
	private function apply_mixed_results( $query, $ids, $post_types, $search_term ) {
		$query->set( 'post__in', $ids );
		$query->set( 'post_type', $post_types );
		$query->set( 'orderby', 'post__in' );
		$query->set( 's', '' );
		$query->set( 'original_search_term', $search_term );

		// Add filter to restore search term for display.
		$this->original_search_term = $search_term;
		add_filter( 'the_search_query', array( $this, 'restore_search_term' ) );
		add_filter( 'get_search_query', array( $this, 'restore_search_term' ) );
	}

	/**
	 * Apply post search results to WordPress query.
	 *
	 * @param WP_Query $query       The WordPress query object.
	 * @param array    $ids         Array of post IDs.
	 * @param string   $search_term Original search term.
	 * @return void
	 */
	private function apply_post_results( $query, $ids, $search_term ) {
		$query->set( 'post__in', $ids );
		$query->set( 'orderby', 'post__in' );
		$query->set( 's', '' );
		$query->set( 'original_search_term', $search_term );

		// Add filter to restore search term for display.
		$this->original_search_term = $search_term;
		add_filter( 'the_search_query', array( $this, 'restore_search_term' ) );
		add_filter( 'get_search_query', array( $this, 'restore_search_term' ) );
	}

	/**
	 * Restore the original search term for display purposes.
	 *
	 * @param string $search_term Current search term.
	 * @return string Original search term.
	 */
	public function restore_search_term( $search_term ) {
		if ( ! empty( $this->original_search_term ) ) {
			return $this->original_search_term;
		}

		global $wp_query;
		if ( isset( $wp_query->query_vars['original_search_term'] ) ) {
			return $wp_query->query_vars['original_search_term'];
		}

		return $search_term;
	}

	/**
	 * Generate a cache key for search results.
	 *
	 * @param string $type        Search type (product, post, mixed).
	 * @param string $search_term The search term.
	 * @return string Cache key.
	 */
	private function get_cache_key( $type, $search_term ) {
		$workspace_id = get_option( INTUFIND_OPTION_WORKSPACE_ID, '' );
		return sprintf(
			'%s_%s_%s',
			$type,
			$workspace_id,
			md5( strtolower( trim( $search_term ) ) )
		);
	}

	/**
	 * Log an error.
	 *
	 * @param string   $message Error message.
	 * @param WP_Error $error   WP_Error object.
	 * @return void
	 */
	private function log_error( $message, $error ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( sprintf( 'Intufind Search: %s - %s', $message, $error->get_error_message() ) );
		}
	}

	/**
	 * Clear all search caches.
	 *
	 * @return void
	 */
	public function clear_cache() {
		wp_cache_flush_group( self::CACHE_GROUP );
	}
}
