<?php
/**
 * Sync status tracking for documents.
 *
 * Tracks per-document sync status including success/error state,
 * timestamps, content hashes for change detection, and error messages.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Per-document sync status tracking.
 *
 * Provides methods to track and query sync status for individual documents.
 */
class Intufind_Sync_Status {

	/**
	 * Meta key for sync status.
	 *
	 * @var string
	 */
	const META_STATUS = '_intufind_sync_status';

	/**
	 * Meta key for last sync timestamp.
	 *
	 * @var string
	 */
	const META_TIMESTAMP = '_intufind_sync_timestamp';

	/**
	 * Meta key for content hash.
	 *
	 * @var string
	 */
	const META_HASH = '_intufind_sync_hash';

	/**
	 * Meta key for error message.
	 *
	 * @var string
	 */
	const META_ERROR = '_intufind_sync_error';

	/**
	 * Meta key for sync method.
	 *
	 * @var string
	 */
	const META_METHOD = '_intufind_sync_method';

	/**
	 * Status: Successfully synced.
	 *
	 * @var string
	 */
	const STATUS_SYNCED = 'synced';

	/**
	 * Status: Pending sync.
	 *
	 * @var string
	 */
	const STATUS_PENDING = 'pending';

	/**
	 * Status: Sync failed.
	 *
	 * @var string
	 */
	const STATUS_ERROR = 'error';

	/**
	 * Status: Not synced (never synced or deleted from cloud).
	 *
	 * @var string
	 */
	const STATUS_NOT_SYNCED = 'not_synced';

	/**
	 * Mark a document as successfully synced.
	 *
	 * @param int    $post_id      Post ID.
	 * @param string $content_hash Content hash at time of sync.
	 * @param string $method       Sync method (auto, manual, bulk).
	 * @return void
	 */
	public function mark_synced( $post_id, $content_hash = '', $method = 'auto' ) {
		update_post_meta( $post_id, self::META_STATUS, self::STATUS_SYNCED );
		update_post_meta( $post_id, self::META_TIMESTAMP, time() );
		update_post_meta( $post_id, self::META_METHOD, $method );

		if ( ! empty( $content_hash ) ) {
			update_post_meta( $post_id, self::META_HASH, $content_hash );
		}

		// Clear any previous error.
		delete_post_meta( $post_id, self::META_ERROR );
	}

	/**
	 * Mark a document as pending sync.
	 *
	 * @param int $post_id Post ID.
	 * @return void
	 */
	public function mark_pending( $post_id ) {
		update_post_meta( $post_id, self::META_STATUS, self::STATUS_PENDING );
		update_post_meta( $post_id, self::META_TIMESTAMP, time() );
	}

	/**
	 * Mark a document sync as failed.
	 *
	 * @param int    $post_id       Post ID.
	 * @param string $error_message Error message.
	 * @return void
	 */
	public function mark_error( $post_id, $error_message = '' ) {
		update_post_meta( $post_id, self::META_STATUS, self::STATUS_ERROR );
		update_post_meta( $post_id, self::META_TIMESTAMP, time() );

		if ( ! empty( $error_message ) ) {
			update_post_meta( $post_id, self::META_ERROR, $error_message );
		}
	}

	/**
	 * Clear all sync status for a document.
	 *
	 * Used when a document is deleted from the cloud.
	 *
	 * @param int $post_id Post ID.
	 * @return void
	 */
	public function clear_status( $post_id ) {
		delete_post_meta( $post_id, self::META_STATUS );
		delete_post_meta( $post_id, self::META_TIMESTAMP );
		delete_post_meta( $post_id, self::META_HASH );
		delete_post_meta( $post_id, self::META_ERROR );
		delete_post_meta( $post_id, self::META_METHOD );
	}

	/**
	 * Get the sync status for a document.
	 *
	 * @param int $post_id Post ID.
	 * @return array Status data with keys: status, timestamp, hash, error, method.
	 */
	public function get_status( $post_id ) {
		$status = get_post_meta( $post_id, self::META_STATUS, true );

		return array(
			'status'    => $status ?: self::STATUS_NOT_SYNCED,
			'timestamp' => (int) get_post_meta( $post_id, self::META_TIMESTAMP, true ),
			'hash'      => get_post_meta( $post_id, self::META_HASH, true ),
			'error'     => get_post_meta( $post_id, self::META_ERROR, true ),
			'method'    => get_post_meta( $post_id, self::META_METHOD, true ),
		);
	}

	/**
	 * Check if content has changed since last sync.
	 *
	 * @param int    $post_id      Post ID.
	 * @param string $current_hash Current content hash.
	 * @return bool True if content changed.
	 */
	public function has_changed( $post_id, $current_hash ) {
		$stored_hash = get_post_meta( $post_id, self::META_HASH, true );
		return $stored_hash !== $current_hash;
	}

	/**
	 * Check if document needs sync (not synced, pending, or changed).
	 *
	 * @param int    $post_id      Post ID.
	 * @param string $current_hash Current content hash.
	 * @return bool True if sync needed.
	 */
	public function needs_sync( $post_id, $current_hash = '' ) {
		$status = get_post_meta( $post_id, self::META_STATUS, true );

		// Not synced or pending always needs sync.
		if ( empty( $status ) || self::STATUS_PENDING === $status || self::STATUS_NOT_SYNCED === $status ) {
			return true;
		}

		// Error status needs retry.
		if ( self::STATUS_ERROR === $status ) {
			return true;
		}

		// Check for content changes.
		if ( ! empty( $current_hash ) ) {
			return $this->has_changed( $post_id, $current_hash );
		}

		return false;
	}

	/**
	 * Get counts of documents by sync status.
	 *
	 * @param string              $post_type  Post type to query.
	 * @param Intufind_Exclusions $exclusions Optional exclusions instance to filter out excluded posts.
	 * @return array Counts by status.
	 */
	public function get_status_counts( $post_type = '', $exclusions = null ) {
		$counts = array(
			'synced'     => 0,
			'pending'    => 0,
			'error'      => 0,
			'not_synced' => 0,
			'total'      => 0,
		);

		// If we have exclusions, use filtered counting for accuracy.
		if ( $exclusions instanceof Intufind_Exclusions ) {
			return $this->get_filtered_status_counts( $post_type, $exclusions );
		}

		// Fallback to simple query without exclusion filtering.
		global $wpdb;

		// Build query for eligible posts.
		$where = "p.post_status IN ('publish', 'private')";
		if ( ! empty( $post_type ) ) {
			$where .= $wpdb->prepare( ' AND p.post_type = %s', $post_type );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$results = $wpdb->get_results(
			"SELECT
				COALESCE(pm.meta_value, 'not_synced') as status,
				COUNT(*) as count
			FROM {$wpdb->posts} p
			LEFT JOIN {$wpdb->postmeta} pm
				ON p.ID = pm.post_id
				AND pm.meta_key = '" . self::META_STATUS . "'
			WHERE {$where}
			GROUP BY COALESCE(pm.meta_value, 'not_synced')"
		);

		foreach ( $results as $row ) {
			$status_key = $row->status;
			if ( isset( $counts[ $status_key ] ) ) {
				$counts[ $status_key ] = (int) $row->count;
			} elseif ( 'not_synced' === $status_key || empty( $status_key ) ) {
				$counts['not_synced'] = (int) $row->count;
			}
			$counts['total'] += (int) $row->count;
		}

		return $counts;
	}

	/**
	 * Get status counts filtered by exclusions.
	 *
	 * This method accurately counts only posts that would actually be synced,
	 * excluding system pages, password-protected content, etc.
	 *
	 * @param string              $post_type  Post type to query.
	 * @param Intufind_Exclusions $exclusions Exclusions instance.
	 * @return array Counts by status.
	 */
	private function get_filtered_status_counts( $post_type, $exclusions ) {
		$counts = array(
			'synced'     => 0,
			'pending'    => 0,
			'error'      => 0,
			'not_synced' => 0,
			'total'      => 0,
		);

		// Query all eligible posts.
		$args = array(
			'post_type'      => $post_type,
			'post_status'    => array( 'publish', 'private' ),
			'posts_per_page' => -1,
			'fields'         => 'ids',
		);

		$query    = new WP_Query( $args );
		$post_ids = $query->posts;

		foreach ( $post_ids as $post_id ) {
			// Skip excluded posts.
			if ( $exclusions->should_exclude( $post_id, $post_type ) ) {
				continue;
			}

			$status = get_post_meta( $post_id, self::META_STATUS, true );

			if ( empty( $status ) ) {
				$status = self::STATUS_NOT_SYNCED;
			}

			if ( isset( $counts[ $status ] ) ) {
				++$counts[ $status ];
			} else {
				++$counts['not_synced'];
			}

			++$counts['total'];
		}

		return $counts;
	}

	/**
	 * Get documents with a specific status.
	 *
	 * @param string $status    Status to query.
	 * @param string $post_type Post type to filter.
	 * @param int    $limit     Maximum results.
	 * @param int    $offset    Result offset.
	 * @return array Array of post IDs.
	 */
	public function get_posts_by_status( $status, $post_type = '', $limit = 100, $offset = 0 ) {
		$args = array(
			'post_type'      => ! empty( $post_type ) ? $post_type : 'any',
			'post_status'    => array( 'publish', 'private' ),
			'posts_per_page' => $limit,
			'offset'         => $offset,
			'fields'         => 'ids',
		);

		if ( self::STATUS_NOT_SYNCED === $status ) {
			// Posts without the meta key or with empty value.
			$args['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => self::META_STATUS,
					'compare' => 'NOT EXISTS',
				),
				array(
					'key'   => self::META_STATUS,
					'value' => '',
				),
			);
		} else {
			$args['meta_query'] = array(
				array(
					'key'   => self::META_STATUS,
					'value' => $status,
				),
			);
		}

		$query = new WP_Query( $args );
		return $query->posts;
	}

	/**
	 * Get documents with errors for retry.
	 *
	 * @param string $post_type Post type to filter.
	 * @param int    $limit     Maximum results.
	 * @return array Array of post data with ID and error.
	 */
	public function get_error_posts( $post_type = '', $limit = 50 ) {
		$post_ids = $this->get_posts_by_status( self::STATUS_ERROR, $post_type, $limit );
		$results  = array();

		foreach ( $post_ids as $post_id ) {
			$results[] = array(
				'id'        => $post_id,
				'error'     => get_post_meta( $post_id, self::META_ERROR, true ),
				'timestamp' => (int) get_post_meta( $post_id, self::META_TIMESTAMP, true ),
			);
		}

		return $results;
	}

	/**
	 * Get last sync time for a post type.
	 *
	 * @param string $post_type Post type.
	 * @return int|null Unix timestamp or null.
	 */
	public function get_last_sync_time( $post_type ) {
		return get_option( "intufind_last_sync_{$post_type}", null );
	}

	/**
	 * Update last sync time for a post type.
	 *
	 * @param string $post_type Post type.
	 * @return void
	 */
	public function update_last_sync_time( $post_type ) {
		update_option( "intufind_last_sync_{$post_type}", time() );
	}

	/**
	 * Format timestamp for display.
	 *
	 * @param int  $timestamp Unix timestamp.
	 * @param bool $future    Whether to format as future time.
	 * @return string Formatted time string.
	 */
	public function format_time_ago( $timestamp, $future = false ) {
		if ( empty( $timestamp ) ) {
			return __( 'Never', 'intufind' );
		}

		$diff = human_time_diff( $timestamp, time() );

		if ( $future && $timestamp > time() ) {
			return sprintf(
				/* translators: %s: human-readable time difference */
				__( 'in %s', 'intufind' ),
				$diff
			);
		}

		return sprintf(
			/* translators: %s: human-readable time difference */
			__( '%s ago', 'intufind' ),
			$diff
		);
	}

	/**
	 * Get human-readable status label.
	 *
	 * @param string $status Status code.
	 * @return string Translated label.
	 */
	public function get_status_label( $status ) {
		$labels = array(
			self::STATUS_SYNCED     => __( 'Synced', 'intufind' ),
			self::STATUS_PENDING    => __( 'Pending', 'intufind' ),
			self::STATUS_ERROR      => __( 'Error', 'intufind' ),
			self::STATUS_NOT_SYNCED => __( 'Not synced', 'intufind' ),
		);

		return $labels[ $status ] ?? $status;
	}

	/**
	 * Get status badge CSS class.
	 *
	 * @param string $status Status code.
	 * @return string CSS class.
	 */
	public function get_status_class( $status ) {
		$classes = array(
			self::STATUS_SYNCED     => 'intufind-badge--success',
			self::STATUS_PENDING    => 'intufind-badge--warning',
			self::STATUS_ERROR      => 'intufind-badge--error',
			self::STATUS_NOT_SYNCED => 'intufind-badge--muted',
		);

		return $classes[ $status ] ?? 'intufind-badge--muted';
	}

	/**
	 * Get total count of documents marked as synced.
	 *
	 * @return int Total synced count across all post types.
	 */
	public function get_total_synced_count() {
		global $wpdb;

		$count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT pm.post_id)
				FROM {$wpdb->postmeta} pm
				INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
				WHERE pm.meta_key = %s
				AND pm.meta_value = %s
				AND p.post_status IN ('publish', 'private')",
				self::META_STATUS,
				self::STATUS_SYNCED
			)
		);

		return (int) $count;
	}

	/**
	 * Reset all sync status metadata.
	 *
	 * Clears sync status for all posts. Used when cloud state doesn't match
	 * local metadata (e.g., after cloud environment reset).
	 *
	 * @return int Number of posts affected.
	 */
	public function reset_all_sync_status() {
		global $wpdb;

		// Get all meta keys we use.
		$meta_keys = array(
			self::META_STATUS,
			self::META_TIMESTAMP,
			self::META_HASH,
			self::META_ERROR,
			self::META_METHOD,
		);

		$deleted = 0;
		foreach ( $meta_keys as $meta_key ) {
			$deleted += $wpdb->delete(
				$wpdb->postmeta,
				array( 'meta_key' => $meta_key ),
				array( '%s' )
			);
		}

		// Also clear last sync time options.
		$wpdb->query(
			"DELETE FROM {$wpdb->options}
			WHERE option_name LIKE 'intufind_last_sync_%'"
		);

		return $deleted;
	}
}
