<?php
/**
 * Document sync manager.
 *
 * Orchestrates document synchronization between WordPress and the cloud
 * including real-time hooks, batch processing, change detection, and cleanup.
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Document sync manager.
 *
 * Core sync functionality including:
 * - Real-time sync on content changes
 * - Batch processing for bulk operations
 * - Change detection to skip unchanged content
 * - Cloud cleanup of orphaned documents
 */
class Intufind_Sync {

	/**
	 * Batch queue transient key.
	 *
	 * @var string
	 */
	const BATCH_QUEUE_KEY = 'intufind_sync_batch_queue';

	/**
	 * Batch process hook name.
	 *
	 * @var string
	 */
	const BATCH_PROCESS_HOOK = 'intufind_process_sync_batch';

	/**
	 * Scheduled sync hook name.
	 *
	 * @var string
	 */
	const SCHEDULED_SYNC_HOOK = 'intufind_scheduled_sync';

	/**
	 * Maximum items per bulk API request.
	 *
	 * @var int
	 */
	const BULK_BATCH_SIZE = 100;

	/**
	 * Batch processing delay in seconds.
	 *
	 * @var int
	 */
	const BATCH_DELAY = 30;

	/**
	 * Auto-sync settings option names.
	 */
	const OPTION_AUTO_SYNC_ENABLED    = 'intufind_auto_sync_enabled';
	const OPTION_SYNC_INTERVAL        = 'intufind_sync_interval';
	const OPTION_SYNC_HOUR            = 'intufind_sync_hour';
	const OPTION_SYNC_SCHEDULE_VER    = 'intufind_sync_schedule_version';
	const OPTION_SYNC_ON_CONNECT      = 'intufind_sync_on_connect';
	const TRANSIENT_INITIAL_SYNC      = 'intufind_initial_sync_triggered';

	/**
	 * Current schedule version - increment to force reschedule.
	 */
	const SCHEDULE_VERSION = 2;

	/**
	 * Default sync interval (24 hours in seconds).
	 */
	const DEFAULT_SYNC_INTERVAL = 86400;

	/**
	 * Default sync hour (3 AM).
	 */
	const DEFAULT_SYNC_HOUR = 3;

	/**
	 * API client instance.
	 *
	 * @var Intufind_API
	 */
	private $api;

	/**
	 * Content extractor instance.
	 *
	 * @var Intufind_Content_Extractor
	 */
	private $extractor;

	/**
	 * Exclusions manager instance.
	 *
	 * @var Intufind_Exclusions
	 */
	private $exclusions;

	/**
	 * Sync status tracker instance.
	 *
	 * @var Intufind_Sync_Status
	 */
	private $status;

	/**
	 * Constructor.
	 *
	 * @param Intufind_API                    $api        API client instance.
	 * @param Intufind_Content_Extractor|null $extractor  Content extractor.
	 * @param Intufind_Exclusions|null        $exclusions Exclusions manager.
	 * @param Intufind_Sync_Status|null       $status     Status tracker.
	 */
	public function __construct(
		Intufind_API $api,
		?Intufind_Content_Extractor $extractor = null,
		?Intufind_Exclusions $exclusions = null,
		?Intufind_Sync_Status $status = null
	) {
		$this->api        = $api;
		$this->extractor  = $extractor ?? new Intufind_Content_Extractor();
		$this->exclusions = $exclusions ?? new Intufind_Exclusions();
		$this->status     = $status ?? new Intufind_Sync_Status();
	}


	/**
	 * Initialize sync functionality.
	 *
	 * @return void
	 */
	public function init() {
		// Register custom cron interval.
		add_filter( 'cron_schedules', array( $this, 'add_cron_intervals' ) );

		// Only initialize hooks if connected.
		if ( ! $this->is_connected() ) {
			return;
		}

		// WordPress post hooks.
		add_action( 'save_post', array( $this, 'handle_post_save' ), 10, 3 );
		add_action( 'before_delete_post', array( $this, 'handle_post_delete' ) );
		add_action( 'transition_post_status', array( $this, 'handle_status_change' ), 10, 3 );

		// WooCommerce product hooks (more reliable than generic post hooks).
		if ( class_exists( 'WooCommerce' ) ) {
			add_action( 'woocommerce_update_product', array( $this, 'handle_product_update' ) );
			add_action( 'woocommerce_new_product', array( $this, 'handle_product_update' ) );
			add_action( 'woocommerce_delete_product', array( $this, 'handle_product_delete' ) );
		}

		// Taxonomy term hooks.
		add_action( 'created_term', array( $this, 'handle_term_created' ), 10, 3 );
		add_action( 'edited_term', array( $this, 'handle_term_edited' ), 10, 3 );
		add_action( 'delete_term', array( $this, 'handle_term_deleted' ), 10, 4 );

		// Batch processing hook.
		add_action( self::BATCH_PROCESS_HOOK, array( $this, 'process_batch' ) );

		// Scheduled sync hook.
		add_action( self::SCHEDULED_SYNC_HOOK, array( $this, 'run_scheduled_sync' ) );

		// Schedule sync if auto-sync is enabled and not already scheduled.
		$this->maybe_schedule_sync();
	}

	/**
	 * Add custom cron intervals.
	 *
	 * @param array $schedules Existing schedules.
	 * @return array Modified schedules.
	 */
	public function add_cron_intervals( $schedules ) {
		$interval = $this->get_sync_interval();

		$schedules['intufind_sync_interval'] = array(
			'interval' => $interval,
			'display'  => sprintf(
				/* translators: %d: number of hours */
				__( 'Every %d hours (Intufind Sync)', 'intufind' ),
				$interval / 3600
			),
		);

		return $schedules;
	}

	/**
	 * Check if auto-sync is enabled.
	 *
	 * @return bool
	 */
	public function is_auto_sync_enabled() {
		return (bool) get_option( self::OPTION_AUTO_SYNC_ENABLED, true );
	}

	/**
	 * Get the sync interval in seconds.
	 *
	 * @return int
	 */
	public function get_sync_interval() {
		return (int) get_option( self::OPTION_SYNC_INTERVAL, self::DEFAULT_SYNC_INTERVAL );
	}

	/**
	 * Get the preferred sync hour (0-23).
	 *
	 * @return int
	 */
	public function get_sync_hour() {
		return (int) get_option( self::OPTION_SYNC_HOUR, self::DEFAULT_SYNC_HOUR );
	}

	/**
	 * Get the timestamp for the next sync time based on preferred hour.
	 *
	 * @return int Timestamp.
	 */
	private function get_next_sync_time() {
		$hour = $this->get_sync_hour();
		$now  = time();

		$date = new DateTime( 'now', wp_timezone() );
		$date->setTime( $hour, 0, 0 );

		// If we've already passed this hour today, schedule for tomorrow.
		if ( $date->getTimestamp() <= $now ) {
			$date->modify( '+1 day' );
		}

		return $date->getTimestamp();
	}

	/**
	 * Schedule sync if auto-sync is enabled and not already scheduled.
	 *
	 * Also handles migration from old schedule format by checking version.
	 *
	 * @return void
	 */
	public function maybe_schedule_sync() {
		if ( ! $this->is_auto_sync_enabled() ) {
			$this->unschedule_sync();
			update_option( self::OPTION_SYNC_SCHEDULE_VER, self::SCHEDULE_VERSION );
			return;
		}

		// Check if we need to migrate from old schedule.
		$current_version = (int) get_option( self::OPTION_SYNC_SCHEDULE_VER, 0 );
		if ( $current_version < self::SCHEDULE_VERSION ) {
			$this->reschedule_sync();
			update_option( self::OPTION_SYNC_SCHEDULE_VER, self::SCHEDULE_VERSION );
			return;
		}

		// Schedule if not already scheduled.
		if ( ! wp_next_scheduled( self::SCHEDULED_SYNC_HOOK ) ) {
			wp_schedule_event(
				$this->get_next_sync_time(),
				'intufind_sync_interval',
				self::SCHEDULED_SYNC_HOOK
			);
		}
	}

	/**
	 * Reschedule sync with updated settings.
	 *
	 * Called when sync settings are changed.
	 *
	 * @return void
	 */
	public function reschedule_sync() {
		$this->unschedule_sync();

		if ( $this->is_auto_sync_enabled() ) {
			wp_schedule_event(
				$this->get_next_sync_time(),
				'intufind_sync_interval',
				self::SCHEDULED_SYNC_HOOK
			);
		}
	}

	/**
	 * Unschedule all sync events.
	 *
	 * @return void
	 */
	public function unschedule_sync() {
		$timestamp = wp_next_scheduled( self::SCHEDULED_SYNC_HOOK );
		while ( $timestamp ) {
			wp_unschedule_event( $timestamp, self::SCHEDULED_SYNC_HOOK );
			$timestamp = wp_next_scheduled( self::SCHEDULED_SYNC_HOOK );
		}
	}

	/**
	 * Get next scheduled sync time.
	 *
	 * @return int|false Timestamp or false if not scheduled.
	 */
	public function get_next_scheduled_sync() {
		return wp_next_scheduled( self::SCHEDULED_SYNC_HOOK );
	}

	/**
	 * Check if sync on connect is enabled.
	 *
	 * @return bool
	 */
	public static function is_sync_on_connect_enabled() {
		return (bool) get_option( self::OPTION_SYNC_ON_CONNECT, true );
	}

	/**
	 * Trigger initial sync for all enabled content types.
	 *
	 * Called when first connecting the plugin.
	 *
	 * @return array Results of sync operation.
	 */
	public function trigger_initial_sync() {
		$enabled_types = $this->exclusions->get_enabled_post_types();
		$results       = array(
			'synced'  => 0,
			'errors'  => 0,
			'skipped' => 0,
			'deleted' => 0,
			'types'   => array(),
		);

		foreach ( $enabled_types as $post_type ) {
			$type_results = $this->manual_sync( $post_type );

			$results['synced']  += $type_results['synced'];
			$results['errors']  += $type_results['errors'];
			$results['skipped'] += $type_results['skipped'];
			$results['deleted'] += $type_results['deleted'];
			$results['types'][ $post_type ] = $type_results;
		}

		// Set transient to show admin notice.
		set_transient( self::TRANSIENT_INITIAL_SYNC, $results, 600 );

		return $results;
	}

	/**
	 * Check if initial sync was recently triggered.
	 *
	 * @return array|false Sync results or false if not triggered.
	 */
	public static function get_initial_sync_status() {
		return get_transient( self::TRANSIENT_INITIAL_SYNC );
	}

	/**
	 * Clear initial sync status.
	 *
	 * @return void
	 */
	public static function clear_initial_sync_status() {
		delete_transient( self::TRANSIENT_INITIAL_SYNC );
	}

	/**
	 * Check if plugin is connected.
	 *
	 * @return bool
	 */
	private function is_connected() {
		return (bool) get_option( INTUFIND_OPTION_CONNECTED, false );
	}

	// =========================================================================
	// Real-time Hooks
	// =========================================================================

	/**
	 * Handle post save event.
	 *
	 * @param int      $post_id Post ID.
	 * @param \WP_Post $post    Post object.
	 * @param bool     $update  Whether this is an update.
	 * @return void
	 */
	public function handle_post_save( $post_id, $post, $update ) {
		// Skip revisions, autosaves, and auto-drafts.
		if ( wp_is_post_revision( $post_id ) || wp_is_post_autosave( $post_id ) ) {
			return;
		}
		if ( in_array( $post->post_status, array( 'auto-draft', 'inherit' ), true ) ) {
			return;
		}

		// Products handled by WooCommerce hooks.
		if ( 'product' === $post->post_type && class_exists( 'WooCommerce' ) ) {
			return;
		}

		// Check if post type is enabled.
		if ( ! $this->exclusions->is_post_type_enabled( $post->post_type ) ) {
			return;
		}

		// Queue for sync if published.
		if ( 'publish' === $post->post_status ) {
			if ( ! $this->exclusions->should_exclude( $post_id, $post->post_type ) ) {
				$this->queue_sync( $post_id, $post->post_type, 'upsert' );
			}
		}
	}

	/**
	 * Handle post delete event.
	 *
	 * @param int $post_id Post ID.
	 * @return void
	 */
	public function handle_post_delete( $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return;
		}

		// Products handled by WooCommerce hooks.
		if ( 'product' === $post->post_type && class_exists( 'WooCommerce' ) ) {
			return;
		}

		// Only delete from cloud if it was synced.
		$status = $this->status->get_status( $post_id );
		if ( Intufind_Sync_Status::STATUS_SYNCED === $status['status'] ) {
			$this->queue_sync( $post_id, $post->post_type, 'delete' );
		}
	}

	/**
	 * Handle post status change.
	 *
	 * @param string   $new_status New status.
	 * @param string   $old_status Old status.
	 * @param \WP_Post $post       Post object.
	 * @return void
	 */
	public function handle_status_change( $new_status, $old_status, $post ) {
		// Products handled by WooCommerce hooks.
		if ( 'product' === $post->post_type && class_exists( 'WooCommerce' ) ) {
			return;
		}

		if ( ! $this->exclusions->is_post_type_enabled( $post->post_type ) ) {
			return;
		}

		// Unpublished -> Published: Add to cloud.
		if ( 'publish' !== $old_status && 'publish' === $new_status ) {
			if ( ! $this->exclusions->should_exclude( $post->ID, $post->post_type ) ) {
				$this->queue_sync( $post->ID, $post->post_type, 'upsert' );
			}
		}
		// Published -> Unpublished: Remove from cloud.
		elseif ( 'publish' === $old_status && 'publish' !== $new_status ) {
			$status = $this->status->get_status( $post->ID );
			if ( Intufind_Sync_Status::STATUS_SYNCED === $status['status'] ) {
				$this->queue_sync( $post->ID, $post->post_type, 'delete' );
			}
		}
	}

	/**
	 * Handle WooCommerce product update.
	 *
	 * @param int $product_id Product ID.
	 * @return void
	 */
	public function handle_product_update( $product_id ) {
		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return;
		}

		// Skip variations.
		if ( $product->is_type( 'variation' ) ) {
			return;
		}

		// Check exclusions.
		if ( $this->exclusions->should_exclude( $product_id, 'product' ) ) {
			return;
		}

		// Only sync published products.
		if ( 'publish' === $product->get_status() ) {
			$this->queue_sync( $product_id, 'product', 'upsert' );
		}
	}

	/**
	 * Handle WooCommerce product delete.
	 *
	 * @param int $product_id Product ID.
	 * @return void
	 */
	public function handle_product_delete( $product_id ) {
		$status = $this->status->get_status( $product_id );
		if ( Intufind_Sync_Status::STATUS_SYNCED === $status['status'] ) {
			$this->queue_sync( $product_id, 'product', 'delete' );
		}
	}

	// =========================================================================
	// Batch Queue Processing
	// =========================================================================

	/**
	 * Queue a sync operation.
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $post_type Post type.
	 * @param string $action    Action: upsert or delete.
	 * @return void
	 */
	public function queue_sync( $post_id, $post_type, $action ) {
		$queue = get_transient( self::BATCH_QUEUE_KEY );
		if ( ! is_array( $queue ) ) {
			$queue = array();
		}

		$key           = $post_id . '|' . $post_type . '|' . $action;
		$queue[ $key ] = array(
			'post_id'   => $post_id,
			'post_type' => $post_type,
			'action'    => $action,
			'timestamp' => time(),
		);

		set_transient( self::BATCH_QUEUE_KEY, $queue, 300 );

		// Mark as pending.
		if ( 'upsert' === $action ) {
			$this->status->mark_pending( $post_id );
		}

		// Schedule batch processing.
		if ( ! wp_next_scheduled( self::BATCH_PROCESS_HOOK ) ) {
			wp_schedule_single_event( time() + self::BATCH_DELAY, self::BATCH_PROCESS_HOOK );
		}
	}

	/**
	 * Process the sync batch queue.
	 *
	 * @return void
	 */
	public function process_batch() {
		$queue = get_transient( self::BATCH_QUEUE_KEY );
		if ( empty( $queue ) ) {
			return;
		}

		// Clear queue before processing.
		delete_transient( self::BATCH_QUEUE_KEY );

		// Group by post type and action.
		$grouped = array();
		foreach ( $queue as $item ) {
			$key = $item['post_type'] . '|' . $item['action'];
			if ( ! isset( $grouped[ $key ] ) ) {
				$grouped[ $key ] = array();
			}
			$grouped[ $key ][] = $item;
		}

		// Process each group.
		foreach ( $grouped as $key => $items ) {
			list( $post_type, $action ) = explode( '|', $key );

			if ( 'delete' === $action ) {
				$this->process_bulk_delete( $post_type, $items );
			} else {
				$this->process_bulk_upsert( $post_type, $items );
			}
		}
	}

	/**
	 * Process bulk upsert for a post type.
	 *
	 * @param string $post_type Post type.
	 * @param array  $items     Items to upsert.
	 * @return void
	 */
	private function process_bulk_upsert( $post_type, $items ) {
		$documents      = array();
		$processed_ids  = array();
		$skipped_count  = 0;

		foreach ( $items as $item ) {
			$post_id = $item['post_id'];

			// Generate content hash for change detection.
			$hash = $this->extractor->generate_content_hash( $post_id, $post_type );

			// Skip if unchanged (unless it's never been synced).
			$current_status = $this->status->get_status( $post_id );
			if ( Intufind_Sync_Status::STATUS_SYNCED === $current_status['status'] ) {
				if ( ! $this->status->has_changed( $post_id, $hash ) ) {
					$skipped_count++;
					continue;
				}
			}

			// Extract document.
			$document = $this->extractor->extract( $post_id, $post_type );
			if ( ! $document ) {
				$this->status->mark_error( $post_id, __( 'Failed to extract content', 'intufind' ) );
				continue;
			}

			$document['_hash'] = $hash;
			$documents[]       = $document;
			$processed_ids[]   = $post_id;

			// Process in batches.
			if ( count( $documents ) >= self::BULK_BATCH_SIZE ) {
				$this->send_bulk_upsert( $post_type, $documents, $processed_ids );
				$documents     = array();
				$processed_ids = array();
			}
		}

		// Process remaining.
		if ( ! empty( $documents ) ) {
			$this->send_bulk_upsert( $post_type, $documents, $processed_ids );
		}

		// Update last sync time.
		$this->status->update_last_sync_time( $post_type );
	}

	/**
	 * Send bulk upsert to API.
	 *
	 * @param string $post_type    Post type.
	 * @param array  $documents    Documents to upsert.
	 * @param array  $processed_ids Post IDs being processed.
	 * @return void
	 */
	private function send_bulk_upsert( $post_type, $documents, $processed_ids ) {
		// Store hashes for status update.
		$hashes = array();
		foreach ( $documents as $doc ) {
			$hashes[ $doc['id'] ] = $doc['_hash'] ?? '';
			unset( $doc['_hash'] );
		}

		// Send to API.
		if ( 'product' === $post_type ) {
			$result = $this->api->upsert_products( $documents );
		} else {
			$result = $this->api->upsert_posts( $documents );
		}

		// Update status based on result.
		if ( is_wp_error( $result ) ) {
			$error_message = $result->get_error_message();
			foreach ( $processed_ids as $post_id ) {
				$this->status->mark_error( $post_id, $error_message );
			}
		} else {
			// Check for individual failures in response.
			$failed = $this->extract_failed_items( $result );

			foreach ( $processed_ids as $post_id ) {
				if ( in_array( (string) $post_id, $failed['ids'], true ) ) {
					$error_msg = $this->find_failed_item_error( $failed['items'], $post_id );
					$this->status->mark_error( $post_id, $error_msg );
				} else {
					$hash = $hashes[ (string) $post_id ] ?? '';
					$this->status->mark_synced( $post_id, $hash, 'auto' );
				}
			}
		}
	}

	/**
	 * Extract failed items from API response.
	 *
	 * Parses the API response to extract failed item IDs and their error messages.
	 * Handles both camelCase and snake_case response formats.
	 *
	 * @param array $response API response array.
	 * @return array{items: array, ids: array} Failed items and their IDs.
	 */
	private function extract_failed_items( $response ) {
		$failed_items = $response['data']['failedItems'] ?? array();
		$failed_ids   = array_column( $failed_items, 'id' );

		return array(
			'items' => $failed_items,
			'ids'   => $failed_ids,
		);
	}

	/**
	 * Find error message for a specific item in failed items array.
	 *
	 * @param array  $failed_items Array of failed items from API response.
	 * @param string $post_id      Post ID to find error for.
	 * @return string Error message.
	 */
	private function find_failed_item_error( $failed_items, $post_id ) {
		$error_msg = __( 'Cloud sync rejected', 'intufind' );

		foreach ( $failed_items as $failed ) {
			if ( (string) ( $failed['id'] ?? '' ) === (string) $post_id ) {
				$error_msg = $failed['error'] ?? $error_msg;
				break;
			}
		}

		return $error_msg;
	}

	/**
	 * Process bulk delete for a post type.
	 *
	 * @param string $post_type Post type.
	 * @param array  $items     Items to delete.
	 * @return void
	 */
	private function process_bulk_delete( $post_type, $items ) {
		$ids = array_map(
			function ( $item ) {
				return (string) $item['post_id'];
			},
			$items
		);

		// Chunk to batch size.
		$chunks = array_chunk( $ids, self::BULK_BATCH_SIZE );

		foreach ( $chunks as $chunk ) {
			if ( 'product' === $post_type ) {
				$result = $this->api->delete_products( $chunk );
			} else {
				$result = $this->api->delete_posts( $chunk );
			}

			// Clear status for deleted items.
			if ( ! is_wp_error( $result ) ) {
				foreach ( $chunk as $post_id ) {
					$this->status->clear_status( (int) $post_id );
				}
			}
		}
	}

	// =========================================================================
	// Manual & Scheduled Sync
	// =========================================================================

	/**
	 * Run manual sync for a post type.
	 *
	 * @param string $post_type Post type to sync.
	 * @return array Result with counts and error_messages array.
	 */
	public function manual_sync( $post_type ) {
		$results = array(
			'synced'         => 0,
			'skipped'        => 0,
			'errors'         => 0,
			'deleted'        => 0,
			'error_messages' => array(), // Unique error messages for user feedback.
		);

		// Get all eligible posts.
		$args = array(
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'fields'         => 'ids',
		);

		$query     = new WP_Query( $args );
		$post_ids  = $query->posts;
		$synced_ids = array();

		// Process in batches.
		$documents     = array();
		$batch_ids     = array();

		foreach ( $post_ids as $post_id ) {
			// Check exclusions.
			if ( $this->exclusions->should_exclude( $post_id, $post_type ) ) {
				$results['skipped']++;
				continue;
			}

			// Extract document.
			$document = $this->extractor->extract( $post_id, $post_type );
			if ( ! $document ) {
				$results['errors']++;
				continue;
			}

			$hash              = $this->extractor->generate_content_hash( $post_id, $post_type );
			$document['_hash'] = $hash;
			$documents[]       = $document;
			$batch_ids[]       = $post_id;
			$synced_ids[]      = $post_id;

			// Process batch.
			if ( count( $documents ) >= self::BULK_BATCH_SIZE ) {
				$batch_result = $this->send_manual_batch( $post_type, $documents, $batch_ids );
				$results['synced'] += $batch_result['synced'];
				$results['errors'] += $batch_result['errors'];
				$results['error_messages'] = array_merge( $results['error_messages'], $batch_result['error_messages'] );
				$documents = array();
				$batch_ids = array();
			}
		}

		// Process remaining.
		if ( ! empty( $documents ) ) {
			$batch_result = $this->send_manual_batch( $post_type, $documents, $batch_ids );
			$results['synced'] += $batch_result['synced'];
			$results['errors'] += $batch_result['errors'];
			$results['error_messages'] = array_merge( $results['error_messages'], $batch_result['error_messages'] );
		}

		// Deduplicate error messages.
		$results['error_messages'] = array_values( array_unique( $results['error_messages'] ) );

		// Clean up orphaned documents from cloud.
		$cleanup_count         = $this->cleanup_orphaned_documents( $post_type, $synced_ids );
		$results['deleted'] = $cleanup_count;

		// Update last sync time.
		$this->status->update_last_sync_time( $post_type );

		return $results;
	}

	/**
	 * Send manual sync batch to API.
	 *
	 * @param string $post_type Post type.
	 * @param array  $documents Documents to sync.
	 * @param array  $batch_ids Post IDs in batch.
	 * @return array Result counts and error_messages array.
	 */
	private function send_manual_batch( $post_type, $documents, $batch_ids ) {
		$result = array(
			'synced'         => 0,
			'errors'         => 0,
			'error_messages' => array(),
		);

		// Store hashes.
		$hashes = array();
		foreach ( $documents as &$doc ) {
			$hashes[ $doc['id'] ] = $doc['_hash'] ?? '';
			unset( $doc['_hash'] );
		}

		// Send to API.
		if ( 'product' === $post_type ) {
			$response = $this->api->upsert_products( $documents );
		} else {
			$response = $this->api->upsert_posts( $documents );
		}

		if ( is_wp_error( $response ) ) {
			$error_message = $response->get_error_message();
			foreach ( $batch_ids as $post_id ) {
				$this->status->mark_error( $post_id, $error_message );
			}
			$result['errors']           = count( $batch_ids );
			$result['error_messages'][] = $error_message;
		} else {
			// Check for individual failures in response.
			$failed = $this->extract_failed_items( $response );

			foreach ( $batch_ids as $post_id ) {
				if ( in_array( (string) $post_id, $failed['ids'], true ) ) {
					$error_msg = $this->find_failed_item_error( $failed['items'], $post_id );
					$this->status->mark_error( $post_id, $error_msg );
					$result['errors']++;
					$result['error_messages'][] = $error_msg;
				} else {
					$hash = $hashes[ (string) $post_id ] ?? '';
					$this->status->mark_synced( $post_id, $hash, 'manual' );
					$result['synced']++;
				}
			}
		}

		return $result;
	}

	/**
	 * Run scheduled daily sync.
	 *
	 * @return void
	 */
	public function run_scheduled_sync() {
		$enabled_types = $this->exclusions->get_enabled_post_types();

		foreach ( $enabled_types as $post_type ) {
			$this->manual_sync( $post_type );
		}
	}

	// =========================================================================
	// Cloud Cleanup
	// =========================================================================

	/**
	 * Clean up orphaned documents from cloud.
	 *
	 * Removes documents that exist in the cloud but not in WordPress.
	 *
	 * @param string $post_type  Post type.
	 * @param array  $valid_ids  IDs that should exist.
	 * @return int Number of deleted documents.
	 */
	public function cleanup_orphaned_documents( $post_type, $valid_ids ) {
		// Get IDs from cloud.
		$cloud_ids = $this->get_cloud_document_ids( $post_type );

		if ( empty( $cloud_ids ) ) {
			return 0;
		}

		// Find orphans.
		$valid_id_set = array_flip( array_map( 'strval', $valid_ids ) );
		$orphan_ids   = array();

		foreach ( $cloud_ids as $cloud_id ) {
			if ( ! isset( $valid_id_set[ $cloud_id ] ) ) {
				$orphan_ids[] = $cloud_id;
			}
		}

		if ( empty( $orphan_ids ) ) {
			return 0;
		}

		// Delete orphans in batches.
		$deleted = 0;
		$chunks  = array_chunk( $orphan_ids, self::BULK_BATCH_SIZE );

		foreach ( $chunks as $chunk ) {
			if ( 'product' === $post_type ) {
				$result = $this->api->delete_products( $chunk );
			} else {
				$result = $this->api->delete_posts( $chunk );
			}

			if ( ! is_wp_error( $result ) ) {
				$deleted += count( $chunk );
			}
		}

		return $deleted;
	}

	/**
	 * Get document IDs from cloud for a post type.
	 *
	 * @param string $post_type Post type.
	 * @return array Array of document IDs.
	 */
	private function get_cloud_document_ids( $post_type ) {
		$all_ids = array();
		$offset  = 0;
		$limit   = 10000;

		while ( true ) {
			$params = array(
				'limit'  => $limit,
				'offset' => $offset,
			);

			if ( 'product' !== $post_type ) {
				$params['post_type'] = $post_type;
			}

			// Build endpoint with query parameters.
			$endpoint = 'product' === $post_type ? 'products/ids' : 'posts/ids';
			$endpoint .= '?' . http_build_query( $params );

			$response = $this->api->request( 'GET', $endpoint );

			if ( is_wp_error( $response ) ) {
				break;
			}

			$ids = $response['data']['ids'] ?? array();
			if ( empty( $ids ) ) {
				break;
			}

			$all_ids = array_merge( $all_ids, $ids );

			// Check if there are more results.
			$has_more = $response['data']['hasMore'] ?? false;
			if ( ! $has_more ) {
				break;
			}

			$offset += $limit;
		}

		return $all_ids;
	}

	// =========================================================================
	// Utility Methods
	// =========================================================================

	/**
	 * Get sync statistics for a post type.
	 *
	 * @param string $post_type Post type.
	 * @return array Statistics including counts and last sync time.
	 */
	public function get_sync_stats( $post_type ) {
		// Pass exclusions to get accurate counts (excludes system pages, etc.).
		$counts    = $this->status->get_status_counts( $post_type, $this->exclusions );
		$last_sync = $this->status->get_last_sync_time( $post_type );

		return array(
			'counts'         => $counts,
			'last_sync'      => $last_sync,
			'last_sync_ago'  => $this->status->format_time_ago( $last_sync ),
			'post_type'      => $post_type,
			'post_type_label' => get_post_type_labels( get_post_type_object( $post_type ) )->name ?? $post_type,
		);
	}

	/**
	 * Retry failed syncs for a post type.
	 *
	 * @param string $post_type Post type.
	 * @param int    $limit     Maximum to retry.
	 * @return array Result counts.
	 */
	public function retry_failed_syncs( $post_type, $limit = 50 ) {
		$error_posts = $this->status->get_error_posts( $post_type, $limit );
		$results     = array(
			'synced' => 0,
			'errors' => 0,
		);

		foreach ( $error_posts as $post_data ) {
			$post_id = $post_data['id'];

			// Check if still exists and not excluded.
			if ( $this->exclusions->should_exclude( $post_id, $post_type ) ) {
				$this->status->clear_status( $post_id );
				continue;
			}

			// Extract and sync.
			$document = $this->extractor->extract( $post_id, $post_type );
			if ( ! $document ) {
				$results['errors']++;
				continue;
			}

			$hash = $this->extractor->generate_content_hash( $post_id, $post_type );

			if ( 'product' === $post_type ) {
				$response = $this->api->upsert_products( array( $document ) );
			} else {
				$response = $this->api->upsert_posts( array( $document ) );
			}

			if ( is_wp_error( $response ) ) {
				$this->status->mark_error( $post_id, $response->get_error_message() );
				$results['errors']++;
			} else {
				$this->status->mark_synced( $post_id, $hash, 'retry' );
				$results['synced']++;
			}
		}

		return $results;
	}

	/**
	 * Get pending and queued sync count.
	 *
	 * @return int Number of items waiting to sync.
	 */
	public function get_pending_count() {
		$queue = get_transient( self::BATCH_QUEUE_KEY );
		return is_array( $queue ) ? count( $queue ) : 0;
	}

	/**
	 * Get the exclusions manager.
	 *
	 * @return Intufind_Exclusions
	 */
	public function get_exclusions() {
		return $this->exclusions;
	}

	/**
	 * Get the status tracker.
	 *
	 * @return Intufind_Sync_Status
	 */
	public function get_status() {
		return $this->status;
	}

	/**
	 * Get the content extractor.
	 *
	 * @return Intufind_Content_Extractor
	 */
	public function get_extractor() {
		return $this->extractor;
	}

	// =========================================================================
	// Taxonomy Sync
	// =========================================================================

	/**
	 * Handle term created event.
	 *
	 * @param int    $term_id  Term ID.
	 * @param int    $tt_id    Term taxonomy ID.
	 * @param string $taxonomy Taxonomy slug.
	 * @return void
	 */
	public function handle_term_created( $term_id, $tt_id, $taxonomy ) {
		if ( $this->is_excluded_taxonomy( $taxonomy ) ) {
			return;
		}
		$this->queue_taxonomy_sync( $term_id, $taxonomy, 'upsert' );
	}

	/**
	 * Handle term edited event.
	 *
	 * @param int    $term_id  Term ID.
	 * @param int    $tt_id    Term taxonomy ID.
	 * @param string $taxonomy Taxonomy slug.
	 * @return void
	 */
	public function handle_term_edited( $term_id, $tt_id, $taxonomy ) {
		if ( $this->is_excluded_taxonomy( $taxonomy ) ) {
			return;
		}
		$this->queue_taxonomy_sync( $term_id, $taxonomy, 'upsert' );
	}

	/**
	 * Handle term deleted event.
	 *
	 * @param int    $term_id      Term ID.
	 * @param int    $tt_id        Term taxonomy ID.
	 * @param string $taxonomy     Taxonomy slug.
	 * @param mixed  $deleted_term Deleted term object.
	 * @return void
	 */
	public function handle_term_deleted( $term_id, $tt_id, $taxonomy, $deleted_term ) {
		if ( $this->is_excluded_taxonomy( $taxonomy ) ) {
			return;
		}
		$this->queue_taxonomy_sync( $term_id, $taxonomy, 'delete' );
	}

	/**
	 * Check if a taxonomy should be excluded from sync.
	 *
	 * @param string $taxonomy Taxonomy slug.
	 * @return bool
	 */
	private function is_excluded_taxonomy( $taxonomy ) {
		if ( ! $this->exclusions ) {
			// Fallback to system taxonomies if no exclusions instance.
			return in_array( $taxonomy, Intufind_Exclusions::EXCLUDED_SYSTEM_TAXONOMIES, true );
		}
		return ! $this->exclusions->is_taxonomy_enabled( $taxonomy );
	}

	/**
	 * Get syncable taxonomies.
	 *
	 * @return array Array of taxonomy names.
	 */
	public function get_syncable_taxonomies() {
		if ( ! $this->exclusions ) {
			// Fallback: get all public taxonomies minus system ones.
			$all_taxonomies = get_taxonomies( array( 'public' => true ), 'names' );
			return array_diff( $all_taxonomies, Intufind_Exclusions::EXCLUDED_SYSTEM_TAXONOMIES );
		}
		return $this->exclusions->get_enabled_taxonomies();
	}

	/**
	 * Queue a taxonomy term for sync.
	 *
	 * @param int    $term_id  Term ID.
	 * @param string $taxonomy Taxonomy slug.
	 * @param string $action   Action: upsert or delete.
	 * @return void
	 */
	public function queue_taxonomy_sync( $term_id, $taxonomy, $action ) {
		$queue = get_transient( self::BATCH_QUEUE_KEY . '_taxonomy' );
		if ( ! is_array( $queue ) ) {
			$queue = array();
		}

		$key           = 'tax_' . $term_id . '|' . $taxonomy . '|' . $action;
		$queue[ $key ] = array(
			'term_id'   => $term_id,
			'taxonomy'  => $taxonomy,
			'action'    => $action,
			'timestamp' => time(),
		);

		set_transient( self::BATCH_QUEUE_KEY . '_taxonomy', $queue, 300 );

		// Schedule batch processing.
		if ( ! wp_next_scheduled( self::BATCH_PROCESS_HOOK . '_taxonomy' ) ) {
			wp_schedule_single_event( time() + self::BATCH_DELAY, self::BATCH_PROCESS_HOOK . '_taxonomy' );
		}
	}

	/**
	 * Manual sync for all taxonomies.
	 *
	 * @return array Result with counts and error_messages array.
	 */
	public function manual_sync_taxonomies() {
		$results = array(
			'synced'         => 0,
			'skipped'        => 0,
			'errors'         => 0,
			'deleted'        => 0,
			'error_messages' => array(), // Unique error messages for user feedback.
		);

		$taxonomies           = $this->get_syncable_taxonomies();
		$documents            = array();
		$synced_ids           = array();
		$synced_taxonomies    = array(); // Track which taxonomies have terms synced.
		$current_batch_taxes  = array(); // Track taxonomies in current batch.
		$processed_taxonomies = array(); // Track all taxonomies we've processed (including empty ones).
		$had_errors           = false;

		foreach ( $taxonomies as $taxonomy ) {
			$terms = get_terms(
				array(
					'taxonomy'   => $taxonomy,
					'hide_empty' => false,
				)
			);

			if ( is_wp_error( $terms ) ) {
				continue;
			}

			// Track this taxonomy as processed (even if it has 0 terms).
			$processed_taxonomies[] = $taxonomy;

			foreach ( $terms as $term ) {
				$doc = $this->extract_taxonomy_term( $term, $taxonomy );
				if ( $doc ) {
					$documents[]           = $doc;
					$synced_ids[]          = $doc['id'];
					$current_batch_taxes[] = $taxonomy;

					// Process in batches.
					if ( count( $documents ) >= self::BULK_BATCH_SIZE ) {
						$batch_result = $this->send_taxonomy_batch( $documents );
						$results['synced'] += $batch_result['synced'];
						$results['errors'] += $batch_result['errors'];
						$results['error_messages'] = array_merge( $results['error_messages'], $batch_result['error_messages'] );

						// Track if we had any errors.
						if ( $batch_result['errors'] > 0 ) {
							$had_errors = true;
						} else {
							$synced_taxonomies = array_merge( $synced_taxonomies, $current_batch_taxes );
						}

						$documents           = array();
						$current_batch_taxes = array();
					}
				}
			}
		}

		// Process remaining.
		if ( ! empty( $documents ) ) {
			$batch_result = $this->send_taxonomy_batch( $documents );
			$results['synced'] += $batch_result['synced'];
			$results['errors'] += $batch_result['errors'];
			$results['error_messages'] = array_merge( $results['error_messages'], $batch_result['error_messages'] );

			// Track if we had any errors.
			if ( $batch_result['errors'] > 0 ) {
				$had_errors = true;
			} else {
				$synced_taxonomies = array_merge( $synced_taxonomies, $current_batch_taxes );
			}
		}

		// Deduplicate error messages.
		$results['error_messages'] = array_values( array_unique( $results['error_messages'] ) );

		// If no errors, mark ALL processed taxonomies as synced (including empty ones).
		// If there were errors, only mark the ones that successfully synced.
		if ( ! $had_errors ) {
			$synced_taxonomies = $processed_taxonomies;
		} else {
			$synced_taxonomies = array_unique( $synced_taxonomies );
		}

		// Update per-taxonomy sync times.
		foreach ( $synced_taxonomies as $taxonomy ) {
			$this->update_taxonomy_sync_time( $taxonomy );
		}

		// Clean up orphaned taxonomies from cloud.
		$cleanup_count      = $this->cleanup_orphaned_taxonomies( $synced_ids );
		$results['deleted'] = $cleanup_count;

		// Update global last sync time.
		$this->status->update_last_sync_time( 'taxonomy' );

		return $results;
	}

	/**
	 * Extract taxonomy term data for sync.
	 *
	 * @param \WP_Term $term     Term object.
	 * @param string   $taxonomy Taxonomy slug.
	 * @return array|null Term data or null.
	 */
	private function extract_taxonomy_term( $term, $taxonomy ) {
		if ( ! $term instanceof WP_Term ) {
			return null;
		}

		$taxonomy_obj = get_taxonomy( $taxonomy );

		$data = array(
			'id'            => 'term_' . $term->term_id,
			'termId'        => $term->term_id,
			'taxonomyName'  => $taxonomy,
			'taxonomyLabel' => $taxonomy_obj ? $taxonomy_obj->labels->singular_name : $taxonomy,
			'name'          => $term->name,
			'slug'          => $term->slug,
			'description'   => $term->description,
			'count'         => $term->count,
			'parent'        => $term->parent,
			'source'        => 'wordpress',
		);

		// Add hierarchy for nested terms.
		if ( $term->parent > 0 ) {
			$data['hierarchy'] = $this->get_term_hierarchy( $term->term_id, $taxonomy );
		}

		// Add WooCommerce-specific data.
		if ( class_exists( 'WooCommerce' ) ) {
			$data = $this->add_woocommerce_term_data( $data, $term, $taxonomy );
		}

		// Add term meta.
		$data['meta'] = $this->get_syncable_term_meta( $term->term_id, $taxonomy );

		return $data;
	}

	/**
	 * Get term hierarchy chain.
	 *
	 * @param int    $term_id  Term ID.
	 * @param string $taxonomy Taxonomy slug.
	 * @return array Parent term names from root to immediate parent.
	 */
	private function get_term_hierarchy( $term_id, $taxonomy ) {
		$hierarchy = array();
		$ancestors = get_ancestors( $term_id, $taxonomy, 'taxonomy' );

		foreach ( array_reverse( $ancestors ) as $ancestor_id ) {
			$ancestor = get_term( $ancestor_id, $taxonomy );
			if ( $ancestor && ! is_wp_error( $ancestor ) ) {
				$hierarchy[] = $ancestor->name;
			}
		}

		return $hierarchy;
	}

	/**
	 * Add WooCommerce-specific term data.
	 *
	 * @param array    $data     Term data array.
	 * @param \WP_Term $term     Term object.
	 * @param string   $taxonomy Taxonomy slug.
	 * @return array Enhanced term data.
	 */
	private function add_woocommerce_term_data( $data, $term, $taxonomy ) {
		// Product category specific data.
		if ( 'product_cat' === $taxonomy ) {
			$thumbnail_id = get_term_meta( $term->term_id, 'thumbnail_id', true );
			if ( $thumbnail_id ) {
				$data['thumbnailUrl'] = wp_get_attachment_url( $thumbnail_id );
			}
			$data['displayType'] = get_term_meta( $term->term_id, 'display_type', true );
		}

		// Product attribute (pa_*) specific data.
		if ( strpos( $taxonomy, 'pa_' ) === 0 ) {
			$data['isProductAttribute'] = true;
			$data['attributeType']       = $this->get_attribute_type( $taxonomy );

			// Check for swatches (common with variation swatches plugins).
			$swatch_value = $this->get_swatch_value( $term, $taxonomy );
			if ( $swatch_value ) {
				$data['swatch'] = $swatch_value;
			}
		}

		// Product tag display order.
		if ( 'product_tag' === $taxonomy ) {
			$data['type'] = 'product_tag';
		}

		return $data;
	}

	/**
	 * Get attribute type for a product attribute taxonomy.
	 *
	 * @param string $taxonomy Attribute taxonomy (pa_*).
	 * @return string Attribute type (text, select, color, image, etc.).
	 */
	private function get_attribute_type( $taxonomy ) {
		if ( ! function_exists( 'wc_get_attribute_taxonomies' ) ) {
			return 'select';
		}

		$attribute_name = str_replace( 'pa_', '', $taxonomy );
		$attributes     = wc_get_attribute_taxonomies();

		foreach ( $attributes as $attribute ) {
			if ( $attribute->attribute_name === $attribute_name ) {
				return $attribute->attribute_type ?: 'select';
			}
		}

		return 'select';
	}

	/**
	 * Get swatch value for term if available.
	 *
	 * Supports common variation swatches plugins.
	 *
	 * @param \WP_Term $term     Term object.
	 * @param string   $taxonomy Taxonomy slug.
	 * @return array|null Swatch data or null.
	 */
	private function get_swatch_value( $term, $taxonomy ) {
		$swatch = null;

		// Check for color swatch.
		$color = get_term_meta( $term->term_id, 'product_attribute_color', true );
		if ( ! $color ) {
			$color = get_term_meta( $term->term_id, 'pa_color', true );
		}
		if ( ! $color ) {
			// Support for TA WooCommerce Variation Swatches.
			$color = get_term_meta( $term->term_id, 'ta_color', true );
		}

		if ( $color ) {
			$swatch = array(
				'type'  => 'color',
				'value' => $color,
			);
		}

		// Check for image swatch.
		$image_id = get_term_meta( $term->term_id, 'product_attribute_image', true );
		if ( ! $image_id ) {
			$image_id = get_term_meta( $term->term_id, 'ta_image', true );
		}

		if ( $image_id && ! $swatch ) {
			$image_url = wp_get_attachment_url( $image_id );
			if ( $image_url ) {
				$swatch = array(
					'type'  => 'image',
					'value' => $image_url,
				);
			}
		}

		return $swatch;
	}

	/**
	 * Get syncable term meta.
	 *
	 * @param int    $term_id  Term ID.
	 * @param string $taxonomy Taxonomy slug.
	 * @return array Filtered term meta.
	 */
	private function get_syncable_term_meta( $term_id, $taxonomy ) {
		$meta     = array();
		$all_meta = get_term_meta( $term_id );

		if ( ! is_array( $all_meta ) ) {
			return $meta;
		}

		// Meta keys to exclude from sync.
		$excluded_keys = array(
			'thumbnail_id',
			'display_type',
			'product_attribute_color',
			'product_attribute_image',
			'pa_color',
			'ta_color',
			'ta_image',
			'order',
		);

		foreach ( $all_meta as $key => $values ) {
			// Skip excluded and internal keys.
			if ( in_array( $key, $excluded_keys, true ) || strpos( $key, '_' ) === 0 ) {
				continue;
			}

			// Only include scalar values.
			if ( ! empty( $values ) && is_array( $values ) ) {
				$value = maybe_unserialize( $values[0] );
				if ( is_scalar( $value ) && '' !== $value ) {
					$meta[ $key ] = $value;
				}
			}
		}

		return $meta;
	}

	/**
	 * Send taxonomy batch to API.
	 *
	 * @param array $documents Taxonomy term documents.
	 * @return array Result counts and error_messages array.
	 */
	private function send_taxonomy_batch( $documents ) {
		$result = array(
			'synced'         => 0,
			'errors'         => 0,
			'error_messages' => array(),
		);

		$response = $this->api->upsert_taxonomies( $documents );

		if ( is_wp_error( $response ) ) {
			$result['errors']           = count( $documents );
			$result['error_messages'][] = $response->get_error_message();
		} else {
			// Check for individual failures in response (same pattern as post/product batches).
			$failed = $this->extract_failed_items( $response );

			if ( ! empty( $failed['items'] ) ) {
				$result['errors'] = count( $failed['items'] );
				$result['synced'] = count( $documents ) - $result['errors'];

				// Extract unique error messages for feedback.
				foreach ( $failed['items'] as $failed_item ) {
					if ( ! empty( $failed_item['error'] ) ) {
						$result['error_messages'][] = $failed_item['error'];
					}
				}
				$result['error_messages'] = array_values( array_unique( $result['error_messages'] ) );
			} else {
				$result['synced'] = count( $documents );
			}
		}

		return $result;
	}

	/**
	 * Clean up orphaned taxonomies from cloud.
	 *
	 * @param array $valid_ids IDs that should exist.
	 * @return int Number of deleted terms.
	 */
	private function cleanup_orphaned_taxonomies( $valid_ids ) {
		// Get IDs from cloud.
		$response = $this->api->get_taxonomy_ids();

		if ( is_wp_error( $response ) ) {
			return 0;
		}

		$cloud_ids = $response['data']['ids'] ?? array();

		if ( empty( $cloud_ids ) ) {
			return 0;
		}

		// Find orphans.
		$valid_id_set = array_flip( array_map( 'strval', $valid_ids ) );
		$orphan_ids   = array();

		foreach ( $cloud_ids as $cloud_id ) {
			if ( ! isset( $valid_id_set[ $cloud_id ] ) ) {
				$orphan_ids[] = $cloud_id;
			}
		}

		if ( empty( $orphan_ids ) ) {
			return 0;
		}

		// Delete orphans in batches.
		$deleted = 0;
		$chunks  = array_chunk( $orphan_ids, self::BULK_BATCH_SIZE );

		foreach ( $chunks as $chunk ) {
			$result = $this->api->delete_taxonomies( $chunk );
			if ( ! is_wp_error( $result ) ) {
				$deleted += count( $chunk );
			}
		}

		return $deleted;
	}

	/**
	 * Get taxonomy sync statistics.
	 *
	 * @return array Statistics.
	 */
	public function get_taxonomy_sync_stats() {
		$taxonomies = $this->get_syncable_taxonomies();
		$total      = 0;

		foreach ( $taxonomies as $taxonomy ) {
			$count = wp_count_terms( array( 'taxonomy' => $taxonomy, 'hide_empty' => false ) );
			if ( ! is_wp_error( $count ) ) {
				$total += (int) $count;
			}
		}

		$last_sync = $this->status->get_last_sync_time( 'taxonomy' );

		return array(
			'total'          => $total,
			'taxonomies'     => $taxonomies,
			'last_sync'      => $last_sync,
			'last_sync_ago'  => $this->status->format_time_ago( $last_sync ),
		);
	}

	/**
	 * Get sync status for a specific taxonomy.
	 *
	 * @param string $taxonomy Taxonomy slug.
	 * @return array Sync status with 'synced' boolean and 'last_sync' timestamp.
	 */
	public function get_taxonomy_sync_status( $taxonomy ) {
		$last_sync = get_option( "intufind_taxonomy_sync_{$taxonomy}", null );

		return array(
			'synced'        => ! empty( $last_sync ),
			'last_sync'     => $last_sync ? (int) $last_sync : null,
			'last_sync_ago' => $last_sync ? $this->status->format_time_ago( (int) $last_sync ) : null,
		);
	}

	/**
	 * Update sync time for a specific taxonomy.
	 *
	 * @param string $taxonomy Taxonomy slug.
	 * @return void
	 */
	public function update_taxonomy_sync_time( $taxonomy ) {
		update_option( "intufind_taxonomy_sync_{$taxonomy}", time(), false );
	}

	/**
	 * Clear sync status for a specific taxonomy.
	 *
	 * @param string $taxonomy Taxonomy slug.
	 * @return void
	 */
	public function clear_taxonomy_sync_status( $taxonomy ) {
		delete_option( "intufind_taxonomy_sync_{$taxonomy}" );
	}
}
