<?php
/**
 * Plugin Name: Intufind
 * Plugin URI: https://intufind.com
 * Description: AI-powered search and chat for WordPress. Syncs your content to the cloud for semantic search, intelligent recommendations, and conversational AI.
 * Version: 3.0.5
 * Requires at least: 6.0
 * Requires PHP: 8.0
 * Author: Intufind
 * Author URI: https://intufind.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: intufind
 * Domain Path: /languages
 *
 * @package Intufind
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin constants.
 */
define( 'INTUFIND_VERSION', '3.0.5' );
define( 'INTUFIND_PLUGIN_FILE', __FILE__ );
define( 'INTUFIND_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'INTUFIND_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'INTUFIND_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * API constants.
 */
if ( ! defined( 'INTUFIND_API_URL' ) ) {
	define( 'INTUFIND_API_URL', 'https://api.intufind.com' );
}

/**
 * Option names (prefixed for WordPress.org compliance).
 */
define( 'INTUFIND_OPTION_API_KEY', 'intufind_api_key' );
define( 'INTUFIND_OPTION_JWT_SECRET', 'intufind_jwt_secret' );
define( 'INTUFIND_OPTION_PUBLISHABLE_KEY', 'intufind_publishable_key' );
define( 'INTUFIND_OPTION_WORKSPACE_ID', 'intufind_workspace_id' );
define( 'INTUFIND_OPTION_TENANT_ID', 'intufind_tenant_id' );
define( 'INTUFIND_OPTION_CONNECTED', 'intufind_connected' );
define( 'INTUFIND_OPTION_SYNC_POST_TYPES', 'intufind_sync_post_types' );
define( 'INTUFIND_OPTION_MCP_REGISTERED', 'intufind_mcp_registered' );
define( 'INTUFIND_OPTION_MCP_ERROR', 'intufind_mcp_error' );

/**
 * Autoloader for plugin classes.
 *
 * @param string $class_name The class name to load.
 * @return void
 */
function intufind_autoloader( $class_name ) {
	// Only handle our classes.
	if ( strpos( $class_name, 'Intufind_' ) !== 0 ) {
		return;
	}

	// Convert class name to file name.
	$file_name = 'class-' . strtolower( str_replace( '_', '-', $class_name ) ) . '.php';

	// Check includes directory.
	$includes_path = INTUFIND_PLUGIN_DIR . 'includes/' . $file_name;
	if ( file_exists( $includes_path ) ) {
		require_once $includes_path;
		return;
	}

	// Check admin directory.
	$admin_path = INTUFIND_PLUGIN_DIR . 'admin/' . $file_name;
	if ( file_exists( $admin_path ) ) {
		require_once $admin_path;
		return;
	}
}
spl_autoload_register( 'intufind_autoloader' );

/**
 * Initialize the plugin.
 *
 * @return void
 */
function intufind_init() {
	// Load text domain for translations.
	load_plugin_textdomain( 'intufind', false, dirname( INTUFIND_PLUGIN_BASENAME ) . '/languages' );

	// Initialize the main plugin class.
	$plugin = new Intufind_Plugin();
	$plugin->init();
}
add_action( 'plugins_loaded', 'intufind_init' );

/**
 * Plugin activation hook.
 *
 * @return void
 */
function intufind_activate() {
	// Set default options if not exists.
	if ( false === get_option( INTUFIND_OPTION_CONNECTED ) ) {
		add_option( INTUFIND_OPTION_CONNECTED, false );
	}

	// Flush rewrite rules for REST API endpoints.
	flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'intufind_activate' );

/**
 * Plugin deactivation hook.
 *
 * @return void
 */
function intufind_deactivate() {
	// Clear scheduled events.
	wp_clear_scheduled_hook( 'intufind_scheduled_sync' );

	// Flush rewrite rules.
	flush_rewrite_rules();
}
register_deactivation_hook( __FILE__, 'intufind_deactivate' );

/**
 * Add settings link on plugins page.
 *
 * @param array $links Existing plugin action links.
 * @return array Modified plugin action links.
 */
function intufind_plugin_action_links( $links ) {
	$settings_link = sprintf(
		'<a href="%s">%s</a>',
		esc_url( admin_url( 'admin.php?page=intufind' ) ),
		esc_html__( 'Settings', 'intufind' )
	);
	array_unshift( $links, $settings_link );
	return $links;
}
add_filter( 'plugin_action_links_' . INTUFIND_PLUGIN_BASENAME, 'intufind_plugin_action_links' );
