<?php
/**
 * Uninstall handler.
 *
 * Fired when the plugin is uninstalled. Removes all plugin data from the database
 * and notifies the cloud to delete indexed content.
 *
 * @package Intufind
 */

// Exit if not called by WordPress.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

/**
 * Notify the cloud to delete all indexed data.
 *
 * Makes a direct API call since plugin classes aren't loaded during uninstall.
 *
 * @return void
 */
function intufind_uninstall_notify_cloud() {
	$api_key      = get_option( 'intufind_api_key', '' );
	$workspace_id = get_option( 'intufind_workspace_id', '' );

	// Can't notify without credentials.
	if ( empty( $api_key ) || empty( $workspace_id ) ) {
		return;
	}

	// Use constant if available (e.g., defined in wp-config.php for local dev),
	// otherwise default to production API URL.
	$api_url = defined( 'INTUFIND_API_URL' ) ? INTUFIND_API_URL : 'https://api.intufind.com';

	// Make deprovision request to cloud (best effort).
	$response = wp_remote_post(
		trailingslashit( $api_url ) . 'provision/deprovision',
		array(
			'timeout' => 15,
			'headers' => array(
				'Authorization' => 'Bearer ' . $api_key,
				'Content-Type'  => 'application/json',
				'Accept'        => 'application/json',
				'X-Workspace-ID' => $workspace_id,
			),
			'body'    => wp_json_encode(
				array(
					'siteUrl'   => $workspace_id,
					'purgeData' => true,
				)
			),
		)
	);

	// Log errors but don't fail uninstall.
	if ( is_wp_error( $response ) ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( 'Intufind uninstall: Failed to notify cloud - ' . $response->get_error_message() );
	}
}

/**
 * Clean up plugin data.
 *
 * Removes all options, transients, post meta, and cron jobs.
 */
function intufind_uninstall_cleanup() {
	// Notify cloud first while we still have credentials.
	intufind_uninstall_notify_cloud();

	// List of options to delete.
	// NOTE: Keep this list in sync with ajax_disconnect() in class-intufind-admin.php.
	// We use string literals here because plugin constants aren't available during uninstall.
	$options = array(
		// Core connection options.
		'intufind_api_key',
		'intufind_jwt_secret',
		'intufind_publishable_key',
		'intufind_workspace_id',
		'intufind_tenant_id',
		'intufind_connected',
		'intufind_mcp_registered',
		'intufind_mcp_error',
		'intufind_sync_post_types',
		'intufind_sync_taxonomies',
		'intufind_version',
		// Sync settings.
		'intufind_auto_sync_enabled',
		'intufind_sync_interval',
		'intufind_sync_hour',
		'intufind_sync_on_connect',
		'intufind_sync_schedule_version',
		// Search settings.
		'intufind_enable_wp_search',
		'intufind_enable_woo_search',
		'intufind_enable_mixed_search',
		'intufind_search_limit',
		'intufind_search_cache_duration',
		// Chat settings.
		'intufind_enable_chat_widget',
		// Recommendations settings.
		'intufind_enable_related_products',
		'intufind_enable_upsells',
		'intufind_enable_cross_sells',
		'intufind_max_recommendations',
		'intufind_recommendations_cache_duration',
	);

	// Delete options.
	foreach ( $options as $option ) {
		delete_option( $option );
	}

	// Delete dynamic options (sync timestamps for post types and taxonomies).
	global $wpdb;
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		$wpdb->prepare(
			"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
			'intufind_last_%_sync',
			'intufind_taxonomy_sync_%'
		)
	);

	// Delete transients (pattern matches all intufind transients including rate limits).
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		$wpdb->prepare(
			"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
			'%_transient_intufind_%',
			'%_transient_timeout_intufind_%'
		)
	);

	// Clear scheduled events.
	wp_clear_scheduled_hook( 'intufind_scheduled_sync' );
	wp_clear_scheduled_hook( 'intufind_process_sync_batch' );
	wp_clear_scheduled_hook( 'intufind_process_sync_batch_taxonomy' );

	// Delete post meta (sync status tracking).
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		$wpdb->prepare(
			"DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
			'_intufind_%'
		)
	);

	// Delete user meta (dismissed notices for all users).
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		$wpdb->prepare(
			"DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE %s",
			'intufind_%'
		)
	);
}

// Run cleanup.
intufind_uninstall_cleanup();
